import { notifications } from '@mantine/notifications';
import axios from 'axios';
import CryptoJS from 'crypto-js';
export function extractFirstWord(url: string): string | null {
  // Split the URL at the '?' character to separate the base URL and query parameters
  const [baseUrl] = url.split('?');

  // Split the base URL into parts by '/'
  const parts = baseUrl.split('/');

  // Return the first word from the URL if it exists, otherwise return null
  return parts?.length > 1 ? parts[3] : null;
}

export const formatDate = (dateInput: string | Date): string => {
  const date = new Date(dateInput);

  const day = date.getDate();
  const month = date.toLocaleString('en-US', { month: 'short' }); // 'Dec'
  const year = date.getFullYear();

  return `${day} ${month}, ${year}`;
};

export const formatCurrency = (value: number | undefined | null): string => {
  if (typeof value !== 'number') {
    return `0.00`;
  }
  return `${value.toFixed(2)}`;
};

export function removeEmptyValueFilters(filters: any[]) {
  return filters.filter(
    filter =>
      filter.value !== undefined &&
      filter.value !== null &&
      filter.value !== '' &&
      filter.value !== 'All'
  );
}

export function extractPageInfo(
  pageInfo: string
): { currentPage: number; totalPages: number } | null {
  const match = pageInfo?.match(/Page (\d+) of (\d+)/);
  if (match) {
    const currentPage = parseInt(match[1]);
    const totalPages = parseInt(match[2]);
    return { currentPage, totalPages };
  }
  return null;
}

export const organizeDropDownData = (data: { name: string; id: string }[]) =>
  data?.map(item => ({ label: item.name, value: item?.id?.toString() }));

export function capitalizeFirstLetter(inputString: string) {
  if (!inputString) return inputString; // Handle empty string or null input
  return inputString.charAt(0).toUpperCase() + inputString.slice(1);
}

export function capitalizeEveryWord(sentence: any): string {
  if (!sentence) return '';

  return sentence
    .split(' ')
    .map(
      (word: string) =>
        word.charAt(0).toUpperCase() + word.slice(1).toLowerCase()
    )
    .join(' ');
}

export const handleSetParams = (
  searchParams: URLSearchParams,
  searchValues: { [key: string]: any },
  initialSearchValues: { [key: string]: any },
  setSearchParams: (params: URLSearchParams) => void
) => {
  // Check if searchParams is defined and is an instance of URLSearchParams
  if (!(searchParams instanceof URLSearchParams)) {
    console.error(
      'searchParams is not defined or not a URLSearchParams instance'
    );
    return;
  }

  // Check if searchValues is defined and is an object
  if (typeof searchValues !== 'object' || searchValues === null) {
    console.error('searchValues is not defined or not an object');
    return;
  }

  // Clone the current searchParams
  const newParams = new URLSearchParams(searchParams.toString());

  Object.entries(searchValues).forEach(([key, value]) => {
    if (key === 'dateRange' && Array.isArray(value)) {
      const [start, end] = value;
      if (start) {
        newParams.set('dateRangeStart', start.toISOString());
      } else {
        newParams.delete('dateRangeStart');
      }
      if (end) {
        newParams.set('dateRangeEnd', end.toISOString());
      } else {
        newParams.delete('dateRangeEnd');
      }
    } else if (value !== null && value !== undefined && value !== '') {
      // Check if the value differs from the initial value in searchParams
      if (value !== initialSearchValues[key]) {
        newParams.set(key, value);
      } else {
        newParams.delete(key); // Remove the parameter if it matches the initial value
      }
    } else {
      newParams.delete(key);
    }
  });

  // Ensure newParams has any valid changes before setting
  if (newParams.toString() !== searchParams.toString()) {
    setSearchParams(newParams);
  }
};

export function cleanObject(obj: any) {
  // Iterate over the properties of the object
  for (const key in obj) {
    // Check if the property value is null, undefined, or an empty string
    if (
      obj[key] === null ||
      obj[key] === undefined ||
      obj[key] === '' ||
      Number.isNaN(obj[key]) ||
      obj[key]?.length === 0
    ) {
      // If true, delete the property from the object
      delete obj[key];
    }
  }
  return obj;
}

export function capitalizeFirstLetterSentence(
  text: string | undefined
): string {
  if (!text) return '';

  // Trim the text to remove any leading or trailing whitespace.
  const trimmedText = text.trim();

  // Capitalize the first letter and make the rest of the letters lowercase.
  const capitalizedText =
    trimmedText.charAt(0).toUpperCase() + trimmedText.slice(1).toLowerCase();

  return capitalizedText;
}

////////////encryption function/////////////////

const encryptMessage = (message: string): string => {
  if (!message) {
    console.error('Message cannot be empty');
    return '';
  }

  try {
    return CryptoJS.AES.encrypt(
      message,
      process.env.REACT_APP_CRYPTO_KEY ?? 'secret_key'
    ).toString();
  } catch (err) {
    console.error('Encryption failed:', err);
    return '';
  }
};

export const setProcessedSessionStorageData = (
  key: string,
  data: any
): void => {
  const isDevelopment =
    process.env.REACT_APP_APPLICATION_PHASE === 'development';

  const processedData = isDevelopment
    ? JSON.stringify(data)
    : encryptMessage(JSON.stringify(data));

  sessionStorage.setItem(key, processedData);
};

export const handleSuccess = (message: string) => {
  notifications.show({
    title: 'Success',
    message,
    position: 'top-center',
    color: 'teal',
  });
};

export const handleError = (
  error: any,
  fallbackMessage = 'Something went wrong'
) => {
  let message = fallbackMessage;

  try {
    if (axios.isAxiosError(error)) {
      // Axios error handling
      message =
        error?.response?.data?.message?.message || // Nested message (e.g., NestJS)
        error?.response?.data?.message || // Flat message
        error?.response?.data?.error || // Error field
        error?.message || // Fallback to Axios message
        error?.toString() ||
        fallbackMessage;
    } else if (typeof error === 'string') {
      // If error is thrown as a string
      message = error;
    } else if (error instanceof Error) {
      // Standard JS Error object
      message = error.message;
    } else if (typeof error === 'object' && error !== null) {
      // Unknown object error
      message = JSON.stringify(error);
    }
  } catch (err) {
    // In case parsing itself fails
    message = fallbackMessage;
    console.error('Error in error handler:', err);
  }

  notifications.show({
    title: 'Error',
    message,
    color: 'red',
    position: 'top-center',
  });
};

export function extractJsonFromErrorString(errorString: any) {
  const jsonMatch = errorString.match(/\{[\s\S]*\}$/);
  if (!jsonMatch) {
    throw new Error('No JSON object found in the string.');
  }

  try {
    const parsed = JSON.parse(jsonMatch[0]);
    return parsed;
  } catch (err: any) {
    throw new Error('Failed to parse JSON: ' + err.message);
  }
}

export function formatNumber(number: any, locale = 'en-US') {
  if (number === undefined || number === null || isNaN(number)) {
    return '0';
  }
  return new Intl.NumberFormat(locale).format(number);
}

export function formatValue(val: any) {
  if (val === null || val === '') return 'N/A';
  const num = parseFloat(val);
  return val.toString().includes('.') ? num.toFixed(3) : num;
}
