// reducers/userInfoReducer.js

import CryptoJS from 'crypto-js';
import { setProcessedSessionStorageData } from '../../utils/common/function';
import {
  CLEAR_USER_INFO,
  REFRESH_ACCESS_TOKEN,
  SET_CURRENCY,
  SET_LANGUAGE,
  SET_USER_INFO,
} from '../constants/user';

const decryptMessage = (text: string): string => {
  if (!text) {
    console.error('Text cannot be empty');
    return '';
  }

  try {
    const bytes = CryptoJS.AES.decrypt(
      text,
      process.env.REACT_APP_CRYPTO_KEY ?? 'secret_key'
    );
    const originalText = bytes.toString(CryptoJS.enc.Utf8);

    if (!originalText) {
      console.error('Failed to decrypt. Check the text and key.');
      return '';
    }

    return originalText;
  } catch (err) {
    console.error('Decryption failed:', err);
    return '';
  }
};

const getProcessedSessionStorageData = (key: string): any => {
  const dataString = sessionStorage.getItem(key);

  if (!dataString) return null;

  const isDevelopment =
    process.env.REACT_APP_APPLICATION_PHASE === 'development';

  return isDevelopment
    ? JSON.parse(dataString)
    : JSON.parse(decryptMessage(dataString));
};

const DEFAULT_PREFERENCES = {
  language: 'en',
  currency: {
    preferredCurrencyId: '1', // Default USD ID
    details: {
      id: '1',
      code: 'USD',
      name: 'US Dollar',
      symbol: '$',
      isActive: true,
    },
  },
};

const initialState = {
  userInfo: getProcessedSessionStorageData('userInfo'),
  isAuthenticated: !!getProcessedSessionStorageData('userInfo'),
  token: sessionStorage.getItem('token'),
  refreshToken: sessionStorage.getItem('refreshToken'),

  preferences:
    getProcessedSessionStorageData('preferences') || DEFAULT_PREFERENCES,
};
const userInfoReducer = (state = initialState, action: any) => {
  switch (action.type) {
    case SET_USER_INFO:
      if (action.payload) {
        const { user, token, refreshToken } = action.payload;
        sessionStorage.setItem('token', token);
        sessionStorage.setItem('refreshToken', refreshToken);
        setProcessedSessionStorageData('userInfo', user);

        const userCurrency = user.currencyId
          ? {
              currencyId: user.currencyId,
              preferredCurrencyId: user.preferredCurrencyId || '1',
              details:
                user.preferredCurrency || DEFAULT_PREFERENCES.currency.details,
            }
          : state.preferences.currency;

        const updatedPreferences = {
          ...state.preferences,
          currency: userCurrency,
        };

        setProcessedSessionStorageData('preferences', updatedPreferences);
        return {
          ...state,
          userInfo: user,
          isAuthenticated: true,
          token,
          refreshToken,
          preferences: updatedPreferences,
        };
      } else {
        sessionStorage.clear();
        return {
          ...state,
          userInfo: null,
          isAuthenticated: false,
          token: null,
          refreshToken: null,
        };
      }
    case CLEAR_USER_INFO:
      sessionStorage.clear();
      return {
        ...state,
        userInfo: null,
        isAuthenticated: false,
        token: null,
        refreshToken: null,
      };

    case REFRESH_ACCESS_TOKEN:
      sessionStorage.setItem('token', action.payload.token);
      sessionStorage.setItem('refreshToken', action.payload.refreshToken);
      return {
        ...state,
        token: action.payload.token,
      };

    case SET_LANGUAGE: {
      const updatedPreferences = {
        ...state.preferences,
        language: action.payload,
      };
      setProcessedSessionStorageData('preferences', updatedPreferences);
      return {
        ...state,
        preferences: updatedPreferences,
      };
    }

    case SET_CURRENCY: {
      const updatedPreferences = {
        ...state.preferences,
        currency: {
          preferredCurrencyId: action.payload.preferredCurrencyId,
          details: action.payload.details,
        },
      };
      setProcessedSessionStorageData('preferences', updatedPreferences);
      return {
        ...state,
        preferences: updatedPreferences,
      };
    }

    default:
      return state;
  }
};

export default userInfoReducer;
