import { Grid, Paper, Select, TextInput } from '@mantine/core';
import { IconChevronDown, IconCirclePlus, IconEye } from '@tabler/icons-react';

import { useEffect, useMemo, useState } from 'react';
import { FiTrash } from 'react-icons/fi';
import { useSelector } from 'react-redux';
import { useSearchParams } from 'react-router-dom';
import { deleteData, fetchData } from '../../api/api';
import { Button, Table } from '../../components';
import { DebouncedTextInput } from '../../components/DebouncedTextInput';
import { getTranslation } from '../../layout/languages';
import { filterStyle, textAreaStyle } from '../../theme/common.style';
import { paginationInfoValue } from '../../utils/common/constant.objects';
import {
  extractPageInfo,
  handleError,
  handleSetParams,
  handleSuccess,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import { eventCategoriesFromStorage } from './initail.values';
import ViewVoucherDetails from './view-VoucherModal';
import CreateVoucherModal from './voucher-modal';

interface SearchFilter {
  searchValue: string;
  category: string;
  discount: string;
  usageLimit: string;
}

const initialSearchValues = {
  searchValue: '',
  category: '',
  discount: '',
  usageLimit: '',
};

const Voucher = () => {
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };

  const handlePagination = (actionType: string, value?: any) => {
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  };

  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const [opened, setOpened] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [tableData, setTableData] = useState([]);
  const [resetTable, setResetTable] = useState(false);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const category =
      searchParams.get('category') ?? initialSearchValues.category;
    const discount =
      searchParams.get('discount') ?? initialSearchValues.discount;
    const usageLimit =
      searchParams.get('usageLimit') ?? initialSearchValues.usageLimit;

    return {
      searchValue,
      category,
      discount,
      usageLimit,
    };
  };

  const eventCategoryOptions = eventCategoriesFromStorage.map(category => ({
    value: category.id,
    label: category.name,
  }));

  const [searchValues, setSearchValues] = useState<SearchFilter>(
    initializeStateFromQueryParams()
  );
  const [voucherInfo, setvoucherInfo] = useState({
    isModalOpen: false,
    voucher: {},
  });

  const handleFetchDataByFilter = () => {
    setIsLoading(true);

    const filters = removeEmptyValueFilters([
      {
        field: 'voucher.offerName',
        operator: 'like',
        value: searchValues.searchValue,
      },
      {
        field: 'voucher.eventCategoryId',
        operator: 'eq',
        value: searchValues.category,
      },
      {
        field: 'voucher.discountPercentage',
        operator: 'lte',
        value: searchValues.discount ? parseInt(searchValues.discount) : null,
      },
      {
        field: 'voucher.usageLimit',
        operator: 'lte',
        value: searchValues.usageLimit,
      },
    ]);

    const filterObject = JSON.stringify({ filter: filters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });

    const fetchUrl = `vouchers?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        if (!response.data || response.data.length === 0) {
          setTableData(response.data);

          if (response.total > 0) {
            setPaginationInfo({
              ...paginationInfo,
              totalRecords: response.total,
            });
          }
        } else {
          setTableData(response.data);
          const getPages = extractPageInfo(response.pages);
          setPaginationInfo({
            ...paginationInfo,
            totalRecords: response.total,
            totalPages: getPages?.totalPages ?? 0,
          });
        }
      })
      .catch(error => {
        handleError(error, 'Failed to fetch table data');
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };

  useEffect(() => {
    handleSearchButtonClick();
  }, [
    resetTable,
    searchValues.searchValue,
    searchValues.category,
    searchValues.discount,
    searchValues.usageLimit,
    paginationInfo?.currentPage,
    paginationInfo?.rowPerPage,
  ]);

  useEffect(() => {
    const newSearchValues = initializeStateFromQueryParams();
    if (JSON.stringify(newSearchValues) !== JSON.stringify(searchValues)) {
      setSearchValues(newSearchValues);
    }
  }, [searchParams]);

  const handleDeleteVoucher = async (id: string) => {
    setIsLoading(true);
    try {
      await deleteData(`/vouchers/${id}`);
      handleSuccess('Voucher deleted successfully');
      setResetTable(prev => !prev);
    } catch {
      handleError('Failed to delete voucher');
    } finally {
      setIsLoading(false);
    }
  };

  const handleViewClick = (voucher: any) => {
    setvoucherInfo({
      isModalOpen: true,
      voucher,
    });
  };
  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById,
    }));
  };

  const columns = useMemo(
    () => [
      {
        header: (
          <div className="flex text-center items-center rtl:mr-3 rtl:ml-0 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'voucher', 'Date')}
          </div>
        ),
        accessorKey: 'createdAt',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const date = new Date(rowObject?.createdAt).toLocaleDateString(
            'en-GB'
          );
          return (
            <div className="flex text-start ml-2">
              <p className="text-start font-productsans font-normal text-[14px]">
                {date}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'voucher', 'Offer Name')}
          </div>
        ),
        accessorKey: 'offerName',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {rowObject?.offerName}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start font-normal font-productsans text-[14px] ml-2">
            {getTranslation(language, 'voucher', 'Discount Code')}
          </div>
        ),
        accessorKey: 'offerCode',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.offerCode}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'voucher', 'Category')}
          </div>
        ),
        accessorKey: 'eventCategory.name',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {rowObject?.eventCategory?.name}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'voucher', 'Discount')}
          </div>
        ),
        accessorKey: 'discountCap',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {rowObject?.discountCap
                  ? `${rowObject?.discountPercentage}%`
                  : '-'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start font-normal font-productsans text-[14px] ml-2">
            {getTranslation(language, 'voucher', 'Used Time')}
          </div>
        ),
        accessorKey: 'usedTime',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p
                className="text-start font-productsans font-normal text-[14px]"
                style={{ color: '#FF7900' }}
              >
                {rowObject?.usedTime}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'voucher', 'Usage Limit')}
          </div>
        ),
        accessorKey: 'usageLimit',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p
                className="text-start font-productsans font-normal text-[14px]"
                style={{ color: '#55C9CB' }}
              >
                {rowObject?.usageLimit}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'voucher', 'Status')}
          </div>
        ),
        accessorKey: 'status',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const today = new Date();
          const from = new Date(rowObject?.validFrom);
          const to = new Date(rowObject?.validTo);
          const isLive = from <= today && today <= to;

          return (
            <div className="flex text-start">
              <p
                className="text-start font-productsans font-normal text-[14px]"
                style={{ color: isLive ? '#55C9CB' : '#FF6166' }}
              >
                {isLive ? 'Live' : 'Expired'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex items-end justify-end mr-20 rtl:ml-10 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'voucher', 'Action')}
          </div>
        ),
        accessorKey: 'action',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex items-center justify-center font-semibold mr-10">
              <IconEye
                onClick={() => handleViewClick(rowObject)}
                className="align-middle select-none bg-transparent text-center transition-all cursor-pointer rtl:ml-3 rtl:mr-0 mr-3"
                size={22}
              />
              <div className="border-r-[1px] border-r-caption border-color-light h-5 mt-0.5"></div>
              <FiTrash
                className="align-middle select-none bg-transparent text-center transition-all cursor-pointer ml-2.5"
                size={15}
                onClick={() => handleDeleteVoucher(rowObject.id)}
              />
            </div>
          );
        },
      },
    ],
    [language, tableData, resetTable]
  );

  return (
    <>
      <Paper
        className="flex justify-between items-center p-4 min-h-screen"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-3 ml-4" gutter="md" align="center">
          <Grid.Col span={{ base: 12, md: 3, lg: 1.75 }}>
            <div className="text-[33px] font-semibold text-[#ffffff] font-productsans">
              {getTranslation(language, 'navbar', 'Voucher')}
            </div>
          </Grid.Col>

          <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
            <DebouncedTextInput
              placeholder={getTranslation(
                language,
                'voucher',
                'Search Voucher Name'
              )}
              value={searchValues.searchValue}
              onDebouncedChange={e => setValuesById({ searchValue: e })}
              styles={textAreaStyle}
              classNames={{ input: 'text-input' }}
              radius={'6px'}
            />
          </Grid.Col>

          <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
            <Select
              placeholder={getTranslation(
                language,
                'voucher',
                'Select By Category'
              )}
              data={eventCategoryOptions}
              value={searchValues.category}
              clearable
              onChange={value => setValuesById({ category: value || '' })}
              rightSection={<IconChevronDown size={16} color="white" />}
              styles={filterStyle}
              classNames={{ input: 'custom-input' }}
              radius={'6px'}
            />
          </Grid.Col>

          <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
            <Select
              placeholder={getTranslation(language, 'voucher', 'Discount %')}
              data={['10', '20', '30', '40', '50']}
              value={searchValues.discount}
              clearable
              onChange={value => setValuesById({ discount: value || '' })}
              rightSection={<IconChevronDown size={16} color="white" />}
              styles={filterStyle}
              classNames={{ input: 'custom-input' }}
              radius={'6px'}
            />
          </Grid.Col>

          <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
            <TextInput
              placeholder={getTranslation(language, 'voucher', 'Usage Limit')}
              value={searchValues.usageLimit}
              onChange={e => setValuesById({ usageLimit: e.target.value })}
              styles={textAreaStyle}
              classNames={{ input: 'text-input' }}
              radius={'6px'}
            />
          </Grid.Col>

          <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
            <Button
              autoContrast
              size="sm"
              styles={{
                root: {
                  background: '#D83F87',
                },
              }}
              style={{
                height: 48,
                width: '100%',
              }}
              className="px-2"
              onClick={() => setOpened(true)}
            >
              <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                <IconCirclePlus size={17} />
                <div className="ml-1 mt-0.5">
                  {getTranslation(language, 'voucher', 'Create Voucher')}
                </div>
              </div>
            </Button>
          </Grid.Col>
        </Grid>

        <Grid className="py-5 px-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
              style={{ height: 'auto' }}
            >
              <div className="mt-4">
                <Table
                  data={tableData}
                  columns={columns}
                  isLoading={isLoading}
                  paginationInfo={paginationInfo}
                  handlePagination={handlePagination}
                />
              </div>
            </div>
          </Grid.Col>
        </Grid>
        <CreateVoucherModal
          opened={opened}
          setOpened={setOpened}
          onSuccess={handleFetchDataByFilter}
        />
        <ViewVoucherDetails
          opened={voucherInfo?.isModalOpen}
          setOpened={setvoucherInfo}
          voucherData={voucherInfo?.voucher}
        />
      </Paper>
    </>
  );
};

export default Voucher;
