import { Button, Modal, Select, TextInput } from '@mantine/core';
import { DatePickerInput } from '@mantine/dates';
import { IconCalendar, IconChevronDown } from '@tabler/icons-react';
import { useFormik } from 'formik';
import * as Yup from 'yup';
import { postData } from '../../api/api';
import { inputStyle2 } from '../../theme/common.style';
import { handleError, handleSuccess } from '../../utils/common/function';
import { eventCategoriesFromStorage } from './initail.values';
import { useState } from 'react';
import { useSelector } from 'react-redux';
import { getTranslation } from '../../layout/languages';

interface CreateVoucherModalProps {
  opened: boolean;
  setOpened: (opened: boolean) => void;
  onSuccess?: () => void;
}

const CreateVoucherModal: React.FC<CreateVoucherModalProps> = ({
  opened,
  setOpened,
  onSuccess,
}) => {
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const [isLoading, setIsLoading] = useState(false);
  const formik = useFormik({
    initialValues: {
      offerName: '',
      offerCode: '',
      validFrom: '',
      validTo: '',
      discountPercentage: '',
      discountCap: '',
      usageLimit: '',
      eventCategoryId: '',
    },
    validationSchema: Yup.object().shape({
      offerName: Yup.string().required('Name is required'),
      offerCode: Yup.string().required('Code is required'),
      validFrom: Yup.string().required('From Date is required'),
      validTo: Yup.string().required('To Date is required'),
      discountPercentage: Yup.string().required('Discount is required'),
      discountCap: Yup.string().required('Discount Cap is required'),
      usageLimit: Yup.string().required('Limit is required'),
      eventCategoryId: Yup.string().required('Category is required'),
    }),
    onSubmit: async values => {
      const payload = {
        ...values,
        discountPercentage: parseFloat(values.discountPercentage),
        discountCap: parseFloat(values.discountCap),
        usageLimit: parseInt(values.usageLimit),
      };
      setIsLoading(true);
      postData('/vouchers', payload)
        .then(() => {
          handleSuccess('Voucher created successfully!');
          if (onSuccess) {
            onSuccess();
          }
          formik.resetForm();
          setTimeout(() => {
            setOpened(false);
          }, 1000);
        })
        .catch((error: any) => {
          handleError(error);
        })
        .finally(() => setIsLoading(false));
    },
  });

  const eventCategoryOptions = eventCategoriesFromStorage.map(category => ({
    value: category.id,
    label: category.name,
  }));
  const resetAndClose = () => {
    formik.resetForm();
    setOpened(false);
  };

  const handleClose = () => {
    resetAndClose();
  };

  return (
    <Modal
      opened={opened}
      centered
      size="md"
      onClose={() => setOpened(false)}
      withCloseButton={false}
      styles={{
        content: {
          borderRadius: '15px',
          backgroundColor: '#452975',
          padding: 0, // Remove default padding
        },
        body: {
          padding: 0, // Ensure no padding inside modal body
        },
        inner: {
          padding: 0, // Ensures modal wrapper has no padding
          margin: 0,
          maxWidth: '100vw',
        },
      }}
      overlayProps={{
        backgroundOpacity: 0.55,
        blur: 3,
      }}
      transitionProps={{ transition: 'fade-up', duration: 300 }}
      title=""
    >
      <div className="relative w-full flex flex-row justify-between">
        <div className="mt-4 text-[#ffffff] font-productsans ml-4  text-[20px] rtl:mr-80">
          {getTranslation(language, 'voucher', 'Create Voucher')}
        </div>
        <Modal.CloseButton
          onClick={handleClose}
          style={{
            position: 'absolute',
            top: 10,
            right: 10,
            border: 'none',
            outline: 'none',
            boxShadow: 'none',
            color: '#ffffff',
            backgroundColor: 'transparent', // Ensure initial state is transparent
            transition: 'background-color 0.2s ease-in-out',
            width: 33,
            height: 33,
          }}
        />
      </div>

      <div className="mt-6 ">
        <form onSubmit={formik.handleSubmit}>
          <TextInput
            className="custom-input px-5 py-4"
            label={getTranslation(language, 'voucher', 'Offer Name')}
            placeholder={getTranslation(
              language,
              'voucher',
              'Enter Offer Name'
            )}
            name="offerName"
            value={formik.values.offerName}
            onChange={formik.handleChange}
            radius="8px"
            styles={inputStyle2}
            error={
              formik.errors.offerName &&
              (formik.touched.offerName || formik.submitCount > 0)
                ? formik.errors.offerName
                : null
            }
          />
          <TextInput
            className="custom-input px-5 py-4"
            label={getTranslation(language, 'voucher', 'Offer Code')}
            placeholder={getTranslation(
              language,
              'voucher',
              'Enter Offer Code'
            )}
            name="offerCode"
            value={formik.values.offerCode}
            onChange={formik.handleChange}
            radius="8px"
            styles={inputStyle2}
            error={
              formik.errors.offerCode &&
              (formik.touched.offerCode || formik.submitCount > 0)
                ? formik.errors.offerCode
                : null
            }
          />

          <DatePickerInput
            className="custom-input px-5 py-4"
            label={getTranslation(language, 'voucher', 'From Date')}
            placeholder={getTranslation(language, 'voucher', 'Pick a date')}
            name="validFrom"
            value={
              formik.values?.validFrom
                ? new Date(formik.values.validFrom)
                : null
            }
            onChange={(value: Date | null) => {
              formik.setFieldValue('validFrom', value || '');
            }}
            styles={inputStyle2}
            rightSection={<IconCalendar size={20} color="white" />}
            popoverProps={{ withinPortal: true }}
            error={
              formik.errors.validFrom &&
              (formik.touched.validFrom || formik.submitCount > 0)
                ? formik.errors.validFrom
                : null
            }
            radius="8px"
          />

          <DatePickerInput
            className="custom-input px-5 py-4"
            label={getTranslation(language, 'voucher', 'To Date')}
            placeholder={getTranslation(language, 'voucher', 'Pick a date')}
            name="validTo"
            value={
              formik.values?.validTo ? new Date(formik.values.validTo) : null
            }
            onChange={(value: Date | null) => {
              formik.setFieldValue('validTo', value || '');
            }}
            styles={inputStyle2}
            rightSection={<IconCalendar size={20} color="white" />}
            popoverProps={{ withinPortal: true }}
            error={
              formik.errors.validTo &&
              (formik.touched.validTo || formik.submitCount > 0)
                ? formik.errors.validTo
                : null
            }
            radius="8px"
          />

          <Select
            className="custom-input px-5 py-4"
            label={getTranslation(language, 'voucher', 'Event Category')}
            placeholder={getTranslation(language, 'voucher', 'Select Category')}
            name="eventCategoryId"
            data={eventCategoryOptions}
            value={formik.values.eventCategoryId}
            onChange={value =>
              formik.setFieldValue('eventCategoryId', value || '')
            }
            styles={inputStyle2}
            rightSection={<IconChevronDown size={19} color="#ffffff" />}
            error={
              formik.errors.eventCategoryId &&
              (formik.touched.eventCategoryId || formik.submitCount > 0)
                ? formik.errors.eventCategoryId
                : null
            }
            radius="8px"
          />

          <TextInput
            className="custom-input px-5 py-4"
            label={getTranslation(language, 'voucher', 'Discount')}
            placeholder={getTranslation(
              language,
              'voucher',
              'Enter Discount %'
            )}
            name="discountPercentage"
            prefix="%"
            value={formik.values.discountPercentage}
            onChange={formik.handleChange}
            styles={inputStyle2}
            error={
              formik.errors.discountPercentage &&
              (formik.touched.discountPercentage || formik.submitCount > 0)
                ? formik.errors.discountPercentage
                : null
            }
            radius="8px"
          />
          <TextInput
            className="custom-input px-5 py-4"
            label={getTranslation(language, 'voucher', 'Discount Cap')}
            placeholder={getTranslation(
              language,
              'voucher',
              'Enter Discount Cap'
            )}
            name="discountCap"
            prefix="%"
            value={formik.values.discountCap}
            onChange={formik.handleChange}
            styles={inputStyle2}
            error={
              formik.errors.discountCap &&
              (formik.touched.discountCap || formik.submitCount > 0)
                ? formik.errors.discountCap
                : null
            }
            radius="8px"
          />
          <TextInput
            className="custom-input px-5 py-4"
            label={getTranslation(language, 'voucher', 'Usage Limit')}
            placeholder={getTranslation(
              language,
              'voucher',
              'Enter Usage Limit'
            )}
            name="usageLimit"
            value={formik.values.usageLimit}
            onChange={formik.handleChange}
            styles={inputStyle2}
            error={
              formik.errors.usageLimit &&
              (formik.touched.usageLimit || formik.submitCount > 0)
                ? formik.errors.usageLimit
                : null
            }
            radius="8px"
          />
        </form>
        <div className="px-4 mr-2 mb-3 rtl:mr-0 rtl:ml-2">
          <Button
            autoContrast
            m={5}
            size="sm"
            styles={{
              root: {
                background: '#D83F87',
              },
            }}
            style={{
              height: 50,
              width: '100%',
            }}
            className="px-2"
            type="submit"
            onClick={() => {
              formik.handleSubmit();
            }}
            disabled={isLoading}
          >
            <div className="font-productsans font-bold text-[16px] text-[#ffffff] ">
              {getTranslation(
                language,
                'voucher',
                `${isLoading ? 'Loading' : 'Submit'}`
              )}
            </div>
          </Button>
        </div>
      </div>
    </Modal>
  );
};

export default CreateVoucherModal;
