import { Grid, Paper, Select, TextInput } from '@mantine/core';
import { IconChevronDown, IconEdit } from '@tabler/icons-react';
import { useEffect, useMemo, useState } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { Button, Table } from '../../components';
import { filterStyle, textAreaStyle } from '../../theme/common.style';
import {
  AMOUNT_LIMIT_OPTIONS,
  paginationInfoValue,
  SORT_OPTIONS,
} from '../../utils/common/constant.objects';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import { FiTrash } from 'react-icons/fi';
import { RiUserForbidLine } from 'react-icons/ri';
import { deleteData, fetchData, putData } from '../../api/api';
import {
  extractPageInfo,
  handleError,
  handleSetParams,
  handleSuccess,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { DebouncedTextInput } from '../../components/DebouncedTextInput';
import { useSelector } from 'react-redux';
import { getTranslation } from '../../layout/languages';

interface SearchFilter {
  searchValue: string;
  type: string;
  sort: string;
}

const initialSearchValues = {
  searchValue: '',
  type: '',
  sort: '',
};

const Users = () => {
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };

  const handlePagination = (actionType: string, value?: any) => {
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  };

  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const [isLoading, setIsLoading] = useState(false);
  const [tableData, setTableData] = useState<
    { id: string; [key: string]: any }[]
  >([]);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const [resetTable, setResetTable] = useState(false);
  const [accountInfo, setAccountInfo] = useState(false);
  const navigate = useNavigate();

  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;

    return {
      searchValue,
      type,
      sort,
    };
  };

  const [searchValues, setSearchValues] = useState<SearchFilter>(
    initializeStateFromQueryParams()
  );

  const handleFetchDataByFilter = () => {
    setIsLoading(true);

    const filters = removeEmptyValueFilters([
      {
        field: 'users.fullNameOrBusinessName',
        operator: 'like',
        value: searchValues.searchValue,
      },
      {
        field: 'users.roleId',
        operator: 'neq',
        value: '1',
      },
      {
        field: 'users.roleId',
        operator: 'eq',
        value: searchValues.type,
      },
    ]);

    const filterObject = JSON.stringify({ filter: filters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });

    if (searchValues.sort) {
      queryParams.append(
        'orderBy',
        JSON.stringify({
          fullNameOrBusinessName: searchValues.sort,
          displayOrderId: searchValues.sort,
        })
      );
    }

    const fetchUrl = `users?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        if (!response.data || response.data.length === 0) {
          setTableData([]);
          if (response.total > 0) {
            setPaginationInfo({
              ...paginationInfo,
              totalRecords: response.total,
            });
          }
        } else {
          setTableData(response.data);
          const getPages = extractPageInfo(response.pages);
          setPaginationInfo({
            ...paginationInfo,
            totalRecords: response.total,
            totalPages: getPages?.totalPages ?? 0,
          });
        }
      })
      .catch(error => {
        handleError(error, 'Failed to fetch table data');
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };

  useEffect(() => {
    handleSearchButtonClick();
  }, [
    resetTable,
    searchValues.searchValue,
    searchValues.type,
    searchValues.sort,
    paginationInfo?.currentPage,
    paginationInfo?.rowPerPage,
  ]);

  useEffect(() => {
    const newSearchValues = initializeStateFromQueryParams();
    if (JSON.stringify(newSearchValues) !== JSON.stringify(searchValues)) {
      setSearchValues(newSearchValues);
    }
  }, [searchParams]);

  const handleDeleteUser = async (id: string) => {
    setIsLoading(true);
    try {
      await deleteData(`/users/${id}`);
      handleSuccess('User deleted successfully');
      setResetTable(prev => !prev);
    } catch (error: any) {
      handleError(error);
    } finally {
      setIsLoading(false);
    }
  };
  const handleToggleBlock = async (userId: string, currentStatus: boolean) => {
    setIsLoading(true);
    try {
      await putData(`/users/${userId}`, {
        isBlocked: !currentStatus,
      });
      setResetTable(prev => !prev);
    } catch (error) {
      console.error('Failed to update block status:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById,
    }));
  };

  const columns = useMemo(
    () => [
      {
        header: (
          <div className="flex text-start font-normal font-productsans text-[14px]">
            {getTranslation(language, 'user', 'Order ID')}
          </div>
        ),
        accessorKey: 'displayOrderId',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-1">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.displayOrderId}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center ml-3 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'user', 'User Name')}
          </div>
        ),
        accessorKey: 'fullNameOrBusinessName',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {rowObject?.fullNameOrBusinessName || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'user', 'User Type')}
          </div>
        ),
        accessorKey: 'name',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {rowObject?.role?.name}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'user', 'Location')}
          </div>
        ),
        accessorKey: 'liveLocation',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {rowObject?.liveLocation?.name || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'user', 'Email')}
          </div>
        ),
        accessorKey: 'email',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {rowObject?.email || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'user', 'Password')}
          </div>
        ),
        accessorKey: 'password',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start  font-productsans font-normal text-[14px]">
                {rowObject?.password ? rowObject?.password : 'N/A'}{' '}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'user', 'Details')}
          </div>
        ),
        accessorKey: 'detail',
        cell: (info: any) => {
          const rowObject = info?.row?.original;

          const handleViewClick = () => {
            const type = rowObject?.role?.name;
            const id = rowObject?.id;

            switch (type) {
              case 'BUSINESS':
                navigate(`/user/business/${id}`);
                break;
              case 'ENTERTAINER':
                navigate(`/user/entertainer/${id}`);
                break;
              case 'GUEST':
                navigate(`/user/customer/${id}`);
                break;
              default:
                console.warn('Unknown user type:', type);
            }
          };

          return (
            <div className="flex text-start">
              <Button
                variant="subtle"
                color="white"
                style={{
                  padding: 0,
                  fontWeight: 'normal',
                  fontSize: '14px',
                  textDecoration: 'underline',
                }}
                onClick={handleViewClick}
              >
                View
              </Button>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex items-end justify-end mr-10 rtl:-mr-0 rtl:ml-14 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'user', 'Action')}
          </div>
        ),
        accessorKey: 'action',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const { isBlocked } = rowObject;
          return (
            <div className="flex items-center justify-center font-semibold ml-3 -mt-2 -mb-2">
              <div className="flex flex-row justify-between">
                <RiUserForbidLine
                  className={`align-middle select-none bg-transparent text-center transition-all cursor-pointer mr-3 ${
                    isBlocked ? 'text-[#808080]' : 'text-[#fff]'
                  }`}
                  size={22}
                  title={isBlocked ? 'Unblock User' : 'Block User'}
                  onClick={() => handleToggleBlock(rowObject.id, isBlocked)}
                />

                <div className="border-r-[1px] border-r-caption border-color-light h-5 mt-0.5"></div>
                <FiTrash
                  size={18}
                  className="ml-3 cursor-pointer text-red-600"
                  onClick={() => handleDeleteUser(rowObject.id)}
                />
              </div>
            </div>
          );
        },
      },
    ],
    [tableData, language]
  );

  return (
    <>
      <Paper
        className="flex justify-between items-center p-4 min-h-screen "
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 ml-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 6 }}>
            <div className="ml-4 text-[33px] font-semibold text-[#ffffff] font-productsans">
              {getTranslation(language, 'user', 'Users')}
            </div>
          </Grid.Col>

          <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
            <DebouncedTextInput
              placeholder={getTranslation(language, 'user', 'Search By Name')}
              value={searchValues.searchValue}
              onDebouncedChange={e =>
                setValuesById({
                  searchValue: e ?? '',
                })
              }
              styles={textAreaStyle}
              classNames={{ input: 'text-input' }}
            />
          </Grid.Col>

          <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
            <Select
              placeholder={getTranslation(language, 'user', 'Select User Type')}
              data={[
                { value: '3', label: 'BUSINESS' },
                { value: '0', label: 'GUEST' },
                { value: '2', label: 'ENTERTAINER' },
              ]}
              value={searchValues.type}
              clearable
              onChange={value => setValuesById({ type: value || '' })}
              rightSection={<IconChevronDown size={16} color="white" />}
              styles={filterStyle}
              classNames={{ input: 'custom-input' }}
            />
          </Grid.Col>
          <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
            <Select
              placeholder={getTranslation(language, 'user', 'Sort By')}
              clearable
              data={SORT_OPTIONS}
              value={searchValues.sort}
              onChange={value => setValuesById({ sort: value || '' })}
              rightSection={<IconChevronDown size={16} color="white" />}
              styles={filterStyle}
              classNames={{ input: 'custom-input' }}
            />
          </Grid.Col>
        </Grid>

        <Grid className="py-5 px-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
              style={{ height: 'auto' }}
            >
              <div className="mt-4">
                <Table
                  data={tableData}
                  columns={columns}
                  isLoading={isLoading}
                  paginationInfo={paginationInfo}
                  handlePagination={handlePagination}
                />
              </div>
            </div>
          </Grid.Col>
        </Grid>
      </Paper>
    </>
  );
};

export default Users;
