import { useEffect, useState } from 'react';
import { useParams } from 'react-router-dom';
import { fetchData } from '../../api/api';

export default function PublicTicketPage() {
  const { bookingCode } = useParams<{ bookingCode: string }>();
  const [userTicket, setUserTicket] = useState<any>();
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (bookingCode) {
      setLoading(true);
      fetchData(`/tickets/view-ticket-info/${bookingCode}`)
        .then(response => setUserTicket(response))
        .catch(console.error)
        .finally(() => setLoading(false));
    }
  }, [bookingCode]);

  if (loading) return <CenterMessage text="Loading..." />;
  if (!userTicket) return <CenterMessage text="No ticket found" />;

  const { ticket, user, transaction, voucher } = userTicket;

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-[#1B1B2F] via-[#162447] to-[#1B1B2F] text-white px-4">
      <div className="w-full max-w-4xl rounded-2xl shadow-xl backdrop-blur-lg p-6 my-5 space-y-6 bg-gradient-to-br from-[#FFFFFF] via-[#FFFFFF] to-[#FFFFFF]">
        {/* Event Banner Section */}
        <div className="relative h-40 items-center justify-center sm:h-48 md:h-58 overflow-hidden rounded">
          <div className="absolute inset-0 backdrop-blur-sm z-10"></div>
          <img
            src={ticket?.event?.eventImage}
            alt={ticket?.name}
            className="absolute inset-0 w-full h-full object-cover transform hover:scale-105 transition-transform duration-700"
          />
          <div className="absolute inset-0 flex items-center sm:justify-left lg:justify-left md:justify-left justify-center">
            <div className="items-center justify-center relative z-20 text-gray-200 drop-shadow-[0_2px_4px_rgba(0,0,0,0.7)]">
              <div className="inline-block px-2 py-1 rounded-full text-xs font-semibold mb-2 shadow-lg backdrop-blur-lg bg-white/80">
                {ticket?.event?.title}
              </div>
              <h1 className="text-4xl md:text-5xl font-bold mb-2">
                {ticket?.name}
              </h1>
              <div className="inline-flex items-center gap-2 bg-black/40 backdrop-blur-lg px-2 py-1 pb-2 rounded-full shadow-lg">
                <span className="text-lg font-bold">${ticket?.price}</span>
                <span className="text-sm">per ticket</span>
              </div>
            </div>
          </div>
        </div>

        {/* Ticket + User Info */}
        <div className="grid md:grid-cols-2 gap-4 text-sm">
          <Info label="Order #" value={userTicket.displayOrderId} />
          <Info label="Booking Code" value={userTicket.bookingCode} />
          <Info label="No of Seats" value={userTicket.seats?.length || 'N/A'} />
          <Info label="Seats" value={userTicket.seats?.join(', ') || 'N/A'} />
          <Info label="Amount Paid" value={`$${transaction?.actualAmount}`} />
          <Info label="Voucher" value={voucher?.offerName ?? 'N/A'} />
          <Info label="User" value={`${user?.fullNameOrBusinessName}`} />
        </div>

        {/* Event Info */}
        <div className="grid md:grid-cols-2 gap-4 text-sm border-t border-white/20 pt-4 mb-5">
          <Info
            label="Start"
            value={
              ticket?.event?.startDateTime &&
              new Date(ticket?.event?.startDateTime).toLocaleString('en-US')
            }
          />
          <Info
            label="End"
            value={
              ticket?.event?.endDateTime &&
              new Date(ticket?.event?.endDateTime).toLocaleString('en-US')
            }
          />
          <Info label="Location" value={ticket?.event?.venue} />
        </div>
      </div>
    </div>
  );
}

function Info({ label, value }: any) {
  return (
    <div>
      <p className="text-xs text-gray-700">{label}</p>
      <p className="font-medium">{value || '—'}</p>
    </div>
  );
}

function CenterMessage({ text }: { text: string }) {
  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-[#1B1B2F] via-[#162447] to-[#1B1B2F] text-white">
      <p className="text-lg font-medium">{text}</p>
    </div>
  );
}
