import {
  Grid,
  NumberInput,
  Paper,
  Select,
  Textarea,
  TextInput,
} from '@mantine/core';

import { useEffect, useState } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { Button } from '../../components';
import { settingStyle } from '../../theme/common.style';
import {
  initialNotification,
  paginationInfoValue,
} from '../../utils/common/constant.objects';

import * as Yup from 'yup';
import { useSelector } from 'react-redux';

import useScreenSize from '../../hooks/useScreenSize';
import { fetchData, postData, putData } from '../../api/api';
import { useFormik } from 'formik';
import {
  handleError,
  handleSuccess,
  organizeDropDownData,
} from '../../utils/common/function';
import { getTranslation } from '../../layout/languages';
import {
  IconArrowLeft,
  IconChevronDown,
  IconCirclePlus,
} from '@tabler/icons-react';

const SystemUser = () => {
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };

  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const { isSmallScreen } = useScreenSize();
  const [isLoading, setIsLoading] = useState(false);
  const navigate = useNavigate();
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);

  const [formSubmitting, setFormSubmitting] = useState(false);
  const [formSubmittingUser, setFormSubmittingUser] = useState(false);
  const [formSubmittingSystem, setFormSubmittingSystem] = useState(false);

  const formik = useFormik({
    enableReinitialize: true,
    initialValues: {
      firstName: '',
      lastName: '',
      email: '',
      password: '',
    },
    validationSchema: Yup.object().shape({
      firstName: Yup.string().required('First name is required'),
      lastName: Yup.string().required('Last name is required'),
      email: Yup.string().email().required('Email is required'),
      password: Yup.string()
        .required('Password is required')
        .min(8, 'Password must be at least 8 characters long')
        .matches(/\d/, 'Password must contain at least one number'),
    }),
    onSubmit: values => {
      setIsLoading(true);

      postData('/users/admin', values)
        .then(() => {
          handleSuccess('User added successfully');
        })
        .catch((error: any) => {
          handleError(error);
        })
        .finally(() => {
          setIsLoading(false);
        });
    },
  });

  // const handleUserSubmit = async () => {
  //   setFormSubmittingUser(true);
  //   try {
  //     await postData(`/users/admin`, {
  //       firstName: formik.values.firstName,
  //       lastName: formik.values.lastName,
  //       email: formik.values.email,
  //       password: formik.values.password,
  //     });
  //     handleSuccess('User profile updated successfully');
  //   } catch (error) {
  //     handleError(error);
  //   } finally {
  //     setFormSubmittingUser(false);
  //   }
  // };

  return (
    <>
      <Paper
        className="flex justify-between items-center p-4  min-h-screen "
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 ml-4 mb-4">
          <Grid.Col
            span={{ base: 12, md: 12, lg: 4 }}
            className="flex flex-row justify-start"
          >
            <span className="mr-1.5 mt-2.5">
              <div
                className={`rounded-md border-none shadow-sm border bg-color-white bg-opacity-5 w-8 h-8  flex items-center justify-center`}
              >
                <IconArrowLeft
                  className=" align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                  color="#ffffff"
                  onClick={() => navigate(-1)}
                  size={20}
                />
              </div>
            </span>
            <div className="ml-4 text-[31px] font-semibold text-[#ffffff] font-productsans">
              {getTranslation(language, 'setting', 'Add User')}
            </div>
          </Grid.Col>
        </Grid>
        <div
          className="rounded-2xl ml-5 mr-4 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
          style={{ height: 'auto' }}
        >
          <Grid
            className={` ${isSmallScreen ? 'mr-2 ml-4' : 'mr-36 ml-4 rtl:ml-36 rtl:mr-4 mt-4'}`}
          >
            <form onSubmit={formik.handleSubmit}>
              <>
                <Grid>
                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <TextInput
                      withAsterisk
                      label={getTranslation(language, 'setting', 'First Name')}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter First Name'
                      )}
                      name="firstName"
                      value={formik.values?.firstName}
                      onChange={e =>
                        formik.setFieldValue('firstName', e.target.value)
                      }
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                      error={
                        formik.errors.firstName &&
                        (formik.touched.firstName || formik.submitCount > 0)
                          ? formik.errors.firstName
                          : null
                      }
                    />
                  </Grid.Col>

                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <TextInput
                      withAsterisk
                      label={getTranslation(language, 'setting', 'Last Name')}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Last Name'
                      )}
                      name="lastName"
                      onChange={e =>
                        formik.setFieldValue('lastName', e.target.value)
                      }
                      value={formik.values?.lastName}
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                      error={
                        formik.errors.lastName &&
                        (formik.touched.lastName || formik.submitCount > 0)
                          ? formik.errors.lastName
                          : null
                      }
                    />
                  </Grid.Col>

                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <TextInput
                      withAsterisk
                      label={getTranslation(
                        language,
                        'setting',
                        'Email Address'
                      )}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Email Address'
                      )}
                      name="email"
                      onChange={e =>
                        formik.setFieldValue('email', e.target.value)
                      }
                      value={formik.values?.email}
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                      error={
                        formik.errors.email &&
                        (formik.touched.email || formik.submitCount > 0)
                          ? formik.errors.email
                          : null
                      }
                    />
                  </Grid.Col>

                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <TextInput
                      withAsterisk
                      label={getTranslation(language, 'setting', 'Password')}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Password'
                      )}
                      name="password"
                      value={formik.values?.password}
                      onChange={e =>
                        formik.setFieldValue('password', e.target.value)
                      }
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                      error={
                        formik.errors.password &&
                        (formik.touched.password || formik.submitCount > 0)
                          ? formik.errors.password
                          : null
                      }
                    />
                  </Grid.Col>
                </Grid>
              </>
            </form>
            <Grid.Col span={{ base: 12, md: 4, lg: 2.5 }}>
              <div className="py-4 mr-2">
                <Button
                  autoContrast
                  m={5}
                  size="sm"
                  styles={{
                    root: {
                      background: '#D83F87',
                    },
                  }}
                  style={{
                    height: 42,
                    width: '100%',
                  }}
                  className="px-2"
                  isLoading={formSubmitting}
                  onClick={() => {
                    formik.handleSubmit();
                  }}
                >
                  <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                    <div className=" mt-0.5">
                      {getTranslation(language, 'setting', 'Add User')}
                    </div>
                  </div>
                </Button>
              </div>
            </Grid.Col>
          </Grid>
        </div>
      </Paper>
    </>
  );
};

export default SystemUser;
