import {
  Grid,
  NumberInput,
  Paper,
  Select,
  Textarea,
  TextInput,
} from '@mantine/core';

import { useEffect, useState } from 'react';
import { useSearchParams } from 'react-router-dom';
import { Button } from '../../components';
import { settingStyle } from '../../theme/common.style';
import {
  initialNotification,
  paginationInfoValue,
} from '../../utils/common/constant.objects';

import * as Yup from 'yup';
import { useSelector } from 'react-redux';

import useScreenSize from '../../hooks/useScreenSize';
import { fetchData, postData, putData } from '../../api/api';
import { useFormik } from 'formik';
import {
  handleError,
  handleSuccess,
  organizeDropDownData,
} from '../../utils/common/function';
import { getTranslation } from '../../layout/languages';
import { IconChevronDown } from '@tabler/icons-react';
interface UserData {
  firstName: string;
  lastName: string;
  email: string;
  phoneNumber: string;
  password: string;
  bio: string;
}

type SystemSettings = {
  commissionFromTicketSales: number;
  commissionFromTransactions: number;
};

type SystemSettingsResponse = {
  data: SystemSettings;
};

const SystemSettings = () => {
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };

  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const { isSmallScreen } = useScreenSize();
  const [isLoading, setIsLoading] = useState(false);

  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const [userData, setUserData] = useState<UserData | null>(null);
  const [systemSettings, setSystemSettings] = useState<SystemSettings | null>(
    null
  );
  const [formSubmitting, setFormSubmitting] = useState(false);
  const [formSubmittingUser, setFormSubmittingUser] = useState(false);
  const [formSubmittingSystem, setFormSubmittingSystem] = useState(false);
  useEffect(() => {
    const fetchAllData = async () => {
      try {
        setIsLoading(true);

        const [userRes, settingsRes] = await Promise.all([
          fetchData<UserData>(`/users/${userInfo?.id}`),
          fetchData<SystemSettingsResponse>(`/system-settings`),
        ]);

        setUserData(userRes);
        setSystemSettings(settingsRes.data);
      } catch (error) {
        console.error('Error fetching data:', error);
      } finally {
        setIsLoading(false);
      }
    };

    if (userInfo?.id) {
      fetchAllData();
    }
  }, [userInfo?.id]);

  const formik = useFormik({
    enableReinitialize: true,
    initialValues: {
      firstName: userData?.firstName || '',
      lastName: userData?.lastName || '',
      email: userData?.email || '',
      password: userData?.password || '',
      phoneNumber: userData?.phoneNumber || '',
      bio: userData?.bio || '',
      roleId: userInfo?.roleId || '',
      commissionFromTicketSales:
        systemSettings?.commissionFromTicketSales?.toString() || '',
      commissionFromTransactions:
        systemSettings?.commissionFromTransactions?.toString() || '',
    },
    validationSchema: Yup.object().shape({
      firstName: Yup.string().required('First name is required'),
      lastName: Yup.string().required('Last name is required'),
      email: Yup.string().email().required('Email is required'),
      password: Yup.string()
        .required('Password is required')
        .min(8, 'Password must be at least 8 characters long')
        .matches(/\d/, 'Password must contain at least one number'),
      phoneNumber: Yup.string().required('Phone number is required'),
      bio: Yup.string(),
      commissionFromTicketSales: Yup.string().required('Required'),
      commissionFromTransactions: Yup.string().required('Required'),
    }),
    onSubmit: () => {},
  });

  const handleUserSubmit = async () => {
    setFormSubmittingUser(true);
    try {
      await putData(`/users/${userInfo?.id}`, {
        firstName: formik.values.firstName,
        lastName: formik.values.lastName,
        email: formik.values.email,
        password: formik.values.password,
        phoneNumber: formik.values.phoneNumber,
        bio: formik.values.bio,
        roleId: userInfo?.roleId,
      });
      handleSuccess('User profile updated successfully');
    } catch (error) {
      handleError(error);
    } finally {
      setFormSubmittingUser(false);
    }
  };

  const handleSystemSubmit = async () => {
    setFormSubmittingSystem(true);
    try {
      await postData(`/system-settings`, {
        commissionFromTicketSales: Number(
          formik.values.commissionFromTicketSales
        ),
        commissionFromTransactions: Number(
          formik.values.commissionFromTransactions
        ),
      });
      handleSuccess('System settings updated successfully');
    } catch (error) {
      handleError(error);
    } finally {
      setFormSubmittingSystem(false);
    }
  };

  return (
    <>
      <Paper
        className="flex justify-between items-center p-4  min-h-screen "
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 ml-4 mb-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 4 }}>
            <div className="ml-4 text-[31px] font-semibold text-[#ffffff] font-productsans">
              {getTranslation(language, 'setting', 'System Settings')}
            </div>
          </Grid.Col>
        </Grid>
        <div
          className="rounded-2xl ml-5 mr-4 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
          style={{ height: 'auto' }}
        >
          <Grid
            className={` ${isSmallScreen ? 'mr-2 ml-4' : 'mr-36 ml-4 rtl:ml-36 rtl:mr-4'}`}
          >
            <form onSubmit={formik.handleSubmit}>
              <>
                <Grid>
                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <TextInput
                      withAsterisk
                      label={getTranslation(language, 'setting', 'First Name')}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter First Name'
                      )}
                      name="firstName"
                      value={formik.values?.firstName}
                      onChange={formik.handleChange}
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                    />
                  </Grid.Col>

                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <TextInput
                      withAsterisk
                      label={getTranslation(language, 'setting', 'Last Name')}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Last Name'
                      )}
                      name="lastName"
                      onChange={formik.handleChange}
                      value={formik.values?.lastName}
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                    />
                  </Grid.Col>
                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <TextInput
                      withAsterisk
                      label={getTranslation(
                        language,
                        'setting',
                        'Email Address'
                      )}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Email Address'
                      )}
                      name="email"
                      onChange={formik.handleChange}
                      value={formik.values?.email}
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                    />
                  </Grid.Col>
                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <NumberInput
                      withAsterisk
                      label={getTranslation(
                        language,
                        'setting',
                        'Phone Number'
                      )}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Phone Number'
                      )}
                      hideControls
                      name="phoneNumber"
                      value={formik.values?.phoneNumber}
                      onChange={formik.handleChange}
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                    />
                  </Grid.Col>
                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <TextInput
                      withAsterisk
                      label={getTranslation(language, 'setting', 'Password')}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Password'
                      )}
                      name="password"
                      onChange={formik.handleChange}
                      value={formik.values?.password}
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                    />
                  </Grid.Col>
                  <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                    <Select
                      withAsterisk
                      label={getTranslation(language, 'setting', 'Language')}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Language'
                      )}
                      data={organizeDropDownData(referenceData?.languages)}
                      value={''}
                      onChange={value => ''}
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                      rightSection={
                        <IconChevronDown size={18} color="#ffffff" />
                      }
                    />
                  </Grid.Col>
                  <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
                    <Textarea
                      withAsterisk
                      label={getTranslation(language, 'setting', 'Description')}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Description'
                      )}
                      name="bio"
                      value={formik.values?.bio}
                      onChange={formik.handleChange}
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                    />
                  </Grid.Col>
                </Grid>
              </>
            </form>
            <Grid.Col span={{ base: 12, md: 4, lg: 2.5 }}>
              <div className="py-4 mr-2">
                <Button
                  autoContrast
                  m={5}
                  size="sm"
                  styles={{
                    root: {
                      background: '#D83F87',
                    },
                  }}
                  style={{
                    height: 42,
                    width: '100%',
                  }}
                  className="px-2"
                  isLoading={formSubmitting}
                  onClick={handleUserSubmit}
                >
                  <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                    <div className=" mt-0.5">
                      {getTranslation(language, 'setting', 'Save')}
                    </div>
                  </div>
                </Button>
              </div>
            </Grid.Col>
          </Grid>
        </div>
        {/* <Grid className="mt-6 mr-7 ml-4 mb-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 4 }}>
            <div className="ml-4 text-[31px] font-semibold text-[#ffffff] font-productsans">
              {getTranslation(language, 'setting', 'Roles and Permissions')}
            </div>
          </Grid.Col>
        </Grid>
        <div
          className="rounded-2xl ml-5 mr-4 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
          style={{ height: 'auto' }}
        >
          <Grid className="px-4 mt-4">
            <Grid.Col span={{ base: 12, md: 5, lg: 4 }}>
              <TextInput
                withAsterisk
                label={getTranslation(language, 'setting', 'User Name')}
                placeholder={getTranslation(
                  language,
                  'setting',
                  'Enter User Name'
                )}
                value={''}
                onChange={value => ''}
                styles={settingStyle}
                classNames={{ input: 'setting-input' }}
              />
            </Grid.Col>

            <Grid.Col span={{ base: 12, md: 5, lg: 4 }}>
              <TextInput
                withAsterisk
                label={getTranslation(language, 'setting', 'Password')}
                placeholder={getTranslation(
                  language,
                  'setting',
                  'Enter Password'
                )}
                value={''}
                onChange={value => ''}
                styles={settingStyle}
                classNames={{ input: 'setting-input' }}
              />
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 5, lg: 4 }}>
              <Select
                withAsterisk
                label={getTranslation(language, 'setting', 'Roles')}
                placeholder={getTranslation(language, 'setting', 'Select Role')}
                data={organizeDropDownData(referenceData?.userRoles)}
                value={''}
                onChange={value => ''}
                styles={settingStyle}
                classNames={{ input: 'setting-input' }}
                rightSection={<IconChevronDown size={18} color="#ffffff" />}
              />
            </Grid.Col>
            {isMediumScreen && <Grid.Col span={{ md: 5 }}></Grid.Col>}
            <Grid.Col span={{ base: 12, md: 4, lg: 2 }}>
              <div className={` ${isSmallScreen ? '-mb-5' : 'py-4'}`}>
                <Button
                  autoContrast
                  m={5}
                  size="sm"
                  styles={{
                    root: {
                      background: '#D83F87',
                    },
                  }}
                  style={{
                    height: 42,
                    width: '100%',
                  }}
                  className="px-2"
                >
                  <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                    <div className=" mt-0.5">
                      {getTranslation(language, 'setting', 'Save Changes')}
                    </div>
                  </div>
                </Button>
              </div>
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 4, lg: 2 }}>
              <div className="py-4 mr-2">
                <Button
                  autoContrast
                  m={5}
                  size="sm"
                  styles={{
                    root: {
                      background: '#418F90',
                    },
                  }}
                  style={{
                    height: 42,
                    width: '100%',
                  }}
                  className=""
                >
                  <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                    <IconCirclePlus size={17} />
                    <div className="ml-1 mt-0.5">
                      {getTranslation(language, 'setting', 'Add User')}
                    </div>
                  </div>
                </Button>
              </div>
            </Grid.Col>
          </Grid>
        </div> */}
        <Grid className="mt-6 mr-7 ml-4 mb-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 4 }}>
            <div className="ml-4 text-[31px] font-semibold text-[#ffffff] font-productsans">
              {getTranslation(language, 'setting', 'Platform Settings')}
            </div>
          </Grid.Col>
        </Grid>
        <div
          className="rounded-2xl ml-5 mr-4 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
          style={{ height: 'auto' }}
        >
          <Grid className="px-4 py-6">
            <Grid.Col span={{ base: 12, md: 5, lg: 6 }}>
              <form onSubmit={formik.handleSubmit}>
                <Grid>
                  <Grid.Col span={{ base: 12, md: 5, lg: 6 }}>
                    <NumberInput
                      withAsterisk
                      label={getTranslation(
                        language,
                        'setting',
                        'Commission From Ticket Sales'
                      )}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Commission From Ticket Sales'
                      )}
                      hideControls
                      name="commissionFromTicketSales"
                      value={formik.values.commissionFromTicketSales}
                      onChange={value => {
                        formik.setFieldValue(
                          'commissionFromTicketSales',
                          value?.toString() ?? ''
                        );
                      }}
                      error={
                        formik.errors.commissionFromTicketSales &&
                        (formik.touched.commissionFromTicketSales ||
                          formik.submitCount > 0)
                          ? formik.errors.commissionFromTicketSales
                          : null
                      }
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                    />
                  </Grid.Col>

                  <Grid.Col span={{ base: 12, md: 5, lg: 6 }}>
                    <NumberInput
                      withAsterisk
                      label={getTranslation(
                        language,
                        'setting',
                        'Commission From Transactions'
                      )}
                      placeholder={getTranslation(
                        language,
                        'setting',
                        'Enter Commission From Transactions'
                      )}
                      hideControls
                      name="commissionFromTransactions"
                      value={formik.values.commissionFromTransactions}
                      onChange={value => {
                        formik.setFieldValue(
                          'commissionFromTransactions',
                          value?.toString() ?? ''
                        );
                      }}
                      error={
                        formik.errors.commissionFromTransactions &&
                        (formik.touched.commissionFromTransactions ||
                          formik.submitCount > 0)
                          ? formik.errors.commissionFromTransactions
                          : null
                      }
                      styles={settingStyle}
                      classNames={{ input: 'setting-input' }}
                    />
                  </Grid.Col>
                </Grid>
              </form>
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 4, lg: 2 }}>
              <div className="mt-7">
                <Button
                  type="submit"
                  autoContrast
                  m={5}
                  size="sm"
                  styles={{
                    root: {
                      background: '#D83F87',
                    },
                  }}
                  style={{
                    height: 42,
                    width: '100%',
                  }}
                  className="px-2"
                  isLoading={formSubmitting}
                  onClick={handleSystemSubmit}
                >
                  <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                    <div className=" mt-0.5">
                      {getTranslation(language, 'setting', 'Save Changes')}
                    </div>
                  </div>
                </Button>
              </div>
            </Grid.Col>
          </Grid>
        </div>
      </Paper>
    </>
  );
};

export default SystemSettings;
