import { Grid, Paper, Select } from '@mantine/core';
import { IconChevronDown, IconEye } from '@tabler/icons-react';
import { useEffect, useMemo, useState } from 'react';
import { FiTrash } from 'react-icons/fi';
import { useSelector } from 'react-redux';
import { useSearchParams } from 'react-router-dom';
import { deleteData, fetchData } from '../../api/api';
import { Table } from '../../components';
import { DebouncedTextInput } from '../../components/DebouncedTextInput';
import useScreenSize from '../../hooks/useScreenSize';
import { getTranslation } from '../../layout/languages';
import { filterStyle, textAreaStyle } from '../../theme/common.style';
import {
  AMOUNT,
  initialNotification,
  paginationInfoValue,
  SORT_OPTIONS,
} from '../../utils/common/constant.objects';
import { TransactionType } from '../../utils/common/enums';
import {
  extractPageInfo,
  formatDate,
  handleError,
  handleSetParams,
  handleSuccess,
  organizeDropDownData,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import { SearchFilter } from '../events/initial.values';
import SaleView from './sales-views';

import TicketDetailModal from './ticketdetailsModal';

const Tickets = ({ isTicket = false }: { isTicket?: boolean }) => {
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };

  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );

  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const [accountInfo, setaccountInfo] = useState({
    isModalOpen: false,
    id: '',
    row: null,
  });

  const [ticketInfo, setticketInfo] = useState({
    isModalOpen: false,
    ticket: {},
  });
  const { userInfo } = useSelector((state: any) => state?.userInfo);

  const currentRole = userInfo?.roleId;

  const { isSmallScreen } = useScreenSize();
  const [notification, setNotification] = useState(initialNotification);

  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();
    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  const initialSearchValues = {
    searchValue: '',
    type: '',
    limit: '',
    sort: '',
    ticketType: '',
  };

  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;
    const ticketType = searchParams.get('type') ?? initialSearchValues.type;

    return {
      searchValue,
      type,
      limit,
      sort,
      ticketType,
    };
  };

  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );
  const [isLoading, setIsLoading] = useState(false);
  const [tableData, setTableData] = useState([]);
  const [resetTable, setResetTable] = useState(false);

  useEffect(() => {
    const url = new URL(window.location.href);
    if (url.search) {
      url.search = '';
      window.history.replaceState({}, document.title, url.pathname);
    }
    setSearchValues(initialSearchValues);
  }, [location.pathname]);

  const handleFetchDataByFilter = () => {
    if (currentRole === '3' || currentRole === '1') {
      setIsLoading(true);

      const baseFilters = [];

      if (searchValues?.searchValue) {
        baseFilters.push({
          field: 'ticket.displayOrderId',
          operator: 'like',
          value: searchValues.searchValue?.toString(),
        });
      }

      if (searchValues.type) {
        baseFilters.push({
          field: 'ticket.name',
          operator: 'eq',
          value: searchValues.type,
        });
      }

      if (searchValues.limit) {
        baseFilters.push({
          field: 'transaction.actualAmount',
          operator: 'lte',
          value: searchValues.limit,
        });
      }

      const filters = removeEmptyValueFilters(baseFilters);
      const filterObject = JSON.stringify({ filter: filters });
      const queryParams = new URLSearchParams({
        rpp: paginationInfo.rowPerPage.toString(),
        page: (paginationInfo.currentPage === 0
          ? 1
          : paginationInfo.currentPage
        ).toString(),
        filter: filterObject,
      });

      if (searchValues.sort) {
        queryParams.append(
          'orderBy',
          JSON.stringify({
            displayOrderId: searchValues.sort,
            createdAt: searchValues.sort,
          })
        );
      }

      const fetchUrl = `tickets/all${currentRole === '1' ? '' : '/' + userInfo?.id}?${queryParams.toString()}`;

      fetchData(fetchUrl)
        .then((response: any) => {
          if (!response.data || response.data.length === 0) {
            setTableData([]);
            if (response.total > 0) {
              setPaginationInfo({
                ...paginationInfo,
                currentPage: 1,
              });
            }
          } else {
            setTableData(response.data);
            const getPages = extractPageInfo(response.pages);
            setPaginationInfo({
              ...paginationInfo,
              totalRecords: response.total,
              totalPages: getPages?.totalPages ?? 0,
            });
          }
        })
        .catch(error => {
          handleError(error, 'Failed to fetch table data');
        })
        .finally(() => {
          setIsLoading(false);
        });
    } else {
      setIsLoading(true);

      const baseFilters = [];

      if (searchValues?.searchValue) {
        baseFilters.push({
          field: 'ticket.displayOrderId',
          operator: 'like',
          value: searchValues.searchValue?.toString(),
        });
      }

      if (currentRole === '2') {
        baseFilters.push({
          field: 'ticket.name',
          operator: 'eq',
          value: TransactionType.PERFORMANCE_FEE,
        });
      } else if (currentRole === '3' || currentRole === '1') {
        baseFilters.push({
          field: 'ticket.name',
          operator: 'eq',
          value: TransactionType.TICKET_PURCHASE,
        });
      }

      if (searchValues.type) {
        baseFilters.push({
          field: 'ticket.name',
          operator: 'eq',
          value: searchValues.type,
        });
      }

      if (searchValues.limit) {
        baseFilters.push({
          field: 'transaction.amount',
          operator: 'lte',
          value: searchValues.limit,
        });
      }

      const filters = removeEmptyValueFilters(baseFilters);
      const filterObject = JSON.stringify({ filter: filters });
      const queryParams = new URLSearchParams({
        rpp: paginationInfo.rowPerPage.toString(),
        page: (paginationInfo.currentPage === 0
          ? 1
          : paginationInfo.currentPage
        ).toString(),
        filter: filterObject,
      });

      if (searchValues.sort) {
        queryParams.append(
          'orderBy',
          JSON.stringify({
            displayOrderId: searchValues.sort,
            createdAt: searchValues.sort,
          })
        );
      }

      const fetchUrl = `tickets/all/${userInfo?.id}?${queryParams.toString()}`;

      fetchData(fetchUrl)
        .then((response: any) => {
          if (!response.data || response.data.length === 0) {
            setTableData([]);
            if (response.total > 0) {
              setPaginationInfo({
                ...paginationInfo,
                currentPage: 1,
              });
            }
          } else {
            setTableData(response.data);
            const getPages = extractPageInfo(response.pages);
            setPaginationInfo({
              ...paginationInfo,
              totalRecords: response.total,
              totalPages: getPages?.totalPages ?? 0,
            });
          }
        })
        .catch(error => {
          handleError(error, 'Failed to fetch table data');
        })
        .finally(() => {
          setIsLoading(false);
        });
    }
  };

  useEffect(() => {
    handleFetchDataByFilter();
  }, [currency]);
  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };

  useEffect(() => {
    handleSearchButtonClick();
  }, [
    isTicket,
    resetTable,
    paginationInfo?.currentPage,
    searchValues.searchValue,
    searchValues.limit,
    searchValues.sort,
    searchValues.type,
  ]);

  useEffect(() => {
    const newSearchValues = initializeStateFromQueryParams();
    if (JSON.stringify(newSearchValues) !== JSON.stringify(searchValues)) {
      setSearchValues(newSearchValues);
    }
  }, [searchParams]);

  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById,
    }));
  };

  const handleDeleteById = (id: string) => {
    setIsLoading(true);

    deleteData(`/transactions/${id}`)
      .then(() => {
        handleSuccess('Transaction is deleted successfully');
        setResetTable(prev => !prev);
      })
      .catch((error: any) => {
        handleError(error);
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  const columns = [
    {
      header: (
        <div className="flex text-start  font-normal font-productsans text-[14px] rtl:mr-2 ">
          {getTranslation(language, 'sales', 'OrderID')}
        </div>
      ),
      accessorKey: 'orderId',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start rtl:mr-2 ml-2 rtl:ml-0">
            <p className="text-start font-normal font-productsans text-[14px]">
              {rowObject?.displayOrderId}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-center items-center ml-3 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'sales', 'Date')}
        </div>
      ),
      accessorKey: 'date',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start">
            <p className="text-start  font-productsans font-normal text-[14px]">
              {formatDate(rowObject?.createdAt?.substring(0, 10) || '')}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'sales', 'ClientName')}
        </div>
      ),
      accessorKey: 'clientName',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start">
            <p className="text-start  font-productsans font-normal text-[14px]">
              {currentRole === '2'
                ? rowObject?.fromUser?.fullNameOrBusinessName
                  ? rowObject?.fromUser?.fullNameOrBusinessName
                  : 'N/A'
                : rowObject?.user?.fullNameOrBusinessName}
            </p>
          </div>
        );
      },
    },
  ];

  const restrictedRoles = ['3'];

  if (currentRole === '1' || restrictedRoles.includes(currentRole)) {
    columns.push(
      {
        header: (
          <div className="flex text-center items-center font-normal ml-3 font-productsans text-[14px]">
            {getTranslation(language, 'sales', 'Events')}
          </div>
        ),
        accessorKey: 'event',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex items-center text-start">
              {(() => {
                const event =
                  currentRole === '2'
                    ? rowObject?.event
                    : rowObject?.ticket?.event;
                const eventImage = event?.eventImage;
                const eventTitle = event?.title;

                return (
                  <>
                    {eventImage ? (
                      <img
                        src={eventImage}
                        alt={eventTitle || 'Event Image'}
                        className="rounded-md w-9 h-6 object-cover flex-shrink-0"
                        loading="lazy"
                      />
                    ) : (
                      <div className="rounded-md w-9 h-6 bg-gray-300 flex items-center justify-center text-xs font-semibold uppercase flex-shrink-0">
                        {eventTitle?.[0] || 'E'}
                      </div>
                    )}

                    <span className="ml-2 font-productsans font-normal text-[14px] truncate max-w-[160px]">
                      {eventTitle || 'N/A'}
                    </span>
                  </>
                );
              })()}
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-start font-normal font-productsans text-[14px] ml-3' : 'flex text-start ml-9 font-normal font-productsans text-[14px] '}`}
          >
            {getTranslation(language, 'sales', 'Ticket Bought')}
          </div>
        ),
        accessorKey: 'ticketBought',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const ticketBought = rowObject?.seats?.length || 0;
          return (
            <div className="flex text-start">
              <p
                className={` ${isSmallScreen ? 'text-start ml-1 font-productsans font-normal text-[14px]' : 'text-start ml-7 font-productsans font-normal text-[14px] '}`}
              >
                {ticketBought}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-start font-normal font-productsans text-[14px] ml-3' : 'flex text-start ml-9 font-normal font-productsans text-[14px] '}`}
          >
            {getTranslation(language, 'sales', 'Amount Paid')}
          </div>
        ),
        accessorKey: 'amount',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p
                className={` ${isSmallScreen ? 'text-start ml-1 font-productsans font-normal text-[14px]' : 'text-start ml-7 font-productsans font-normal text-[14px] '}`}
              >
                {`${currency === '1' ? '$' : 'BHD ≈'} ${currency === '1' ? rowObject?.transaction?.actualAmount : rowObject?.transaction?.convertedActualAmount || 'N/A'}`}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex items-start justify-start  ml-3 rtl:mr-0 rtl:ml-16 font-normal font-productsans text-[14px]' : 'flex items-end justify-end mr-20 ml-3 rtl:mr-0 rtl:ml-16 font-normal font-productsans text-[14px]'}`}
          >
            {getTranslation(language, 'sales', 'Details')}
          </div>
        ),
        accessorKey: 'details',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div
              className={` ${isSmallScreen ? 'flex items-start justify-start font-normal text-[13px] font-productsans underline cursor-pointer mr-6 rtl:ml-12 rtl:mr-0' : 'flex items-center justify-center font-normal text-[13px] font-productsans underline cursor-pointer'}`}
              onClick={() =>
                setaccountInfo({
                  isModalOpen: true,
                  id: rowObject?.id,
                  row: rowObject,
                })
              }
            >
              {getTranslation(language, 'sales', 'View More')}
            </div>
          );
        },
      }
    );
  } else {
    columns.push(
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'sales', 'Location')}
          </div>
        ),
        accessorKey: 'location',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {[
                  rowObject?.event?.venue,
                  rowObject?.event?.city,
                  rowObject?.event?.country,
                ]
                  .filter(Boolean)
                  .join(', ') || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-start font-normal font-productsans text-[14px] ml-3' : 'flex text-start ml-9 font-normal font-productsans text-[14px] '}`}
          >
            {getTranslation(language, 'sales', 'PerformanceFee')}
          </div>
        ),
        accessorKey: 'fee',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p
                className={` ${isSmallScreen ? 'flex text-start font-normal font-productsans text-[14px] ml-1' : 'flex text-start ml-7 font-normal font-productsans text-[14px] '}`}
              >
                {rowObject?.actualAmount}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex items-end justify-end mr-20 rtl:ml-20 rtl:mr-0 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'sales', 'Action')}
          </div>
        ),
        accessorKey: 'action',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div
              className={` ${isSmallScreen ? 'flex items-center justify-center font-semibold mt-1' : 'flex items-center justify-center font-semibold -mt-2 -mb-2 '}`}
            >
              <div className="flex flex-row justify-between">
                <IconEye
                  className="align-middle select-none bg-transparent text-center transition-all cursor-pointer mr-3 rtl:ml-3 rtl:mr-0"
                  size={22}
                  onClick={() =>
                    setaccountInfo({
                      isModalOpen: true,
                      id: rowObject?.id,
                      row: rowObject,
                    })
                  }
                />
                <div className="border-r-[1px] border-r-caption border-color-light h-5 mt-0.5"></div>
                <FiTrash
                  className="ml-3 rtl:mr-3 rtl:mr-0 mt-1 align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                  size={15}
                  onClick={() => handleDeleteById(rowObject?.id)}
                />
              </div>
            </div>
          );
        },
      }
    );
  }

  const commoncolumns = useMemo(() => {
    return [...columns];
  }, [isTicket, resetTable, tableData, language, currency]);

  const ticketColumn = useMemo(() => {
    return [
      {
        header: (
          <div className="flex text-start  font-normal font-productsans text-[14px]">
            {getTranslation(language, 'sales', 'Order Id')}
          </div>
        ),
        accessorKey: 'orderId',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.displayOrderId}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center ml-3 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'navbar', 'Date')}
          </div>
        ),
        accessorKey: 'date',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start  font-productsans font-normal text-[14px]">
                {formatDate(rowObject?.createdAt?.substring(0, 10) || '')}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center font-normal ml-3 font-productsans text-[14px]">
            {getTranslation(language, 'sales', 'Events')}
          </div>
        ),
        accessorKey: 'event',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex items-center text-start">
              {rowObject?.ticket?.event?.eventImage ? (
                <img
                  src={rowObject.ticket.event.eventImage}
                  alt={rowObject?.ticket?.event?.title || 'Event Image'}
                  className="rounded-md w-10 h-6 object-cover flex-shrink-0"
                  loading="lazy"
                />
              ) : (
                <div className="rounded-md w-10 h-6 bg-gray-300 flex items-center justify-center text-xs font-semibold uppercase flex-shrink-0">
                  {rowObject?.ticket?.event?.title?.[0] || 'E'}
                </div>
              )}

              <span className="ml-2 font-productsans font-normal text-[14px] truncate max-w-[160px]">
                {rowObject?.ticket?.event?.title || 'N/A'}
              </span>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'sales', 'Customer Name')}
          </div>
        ),
        accessorKey: 'clientName',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start  font-productsans font-normal text-[14px]">
                {rowObject?.user?.fullNameOrBusinessName
                  ? rowObject?.user?.fullNameOrBusinessName
                  : 'N/A'}
              </p>
            </div>
          );
        },
      },

      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-start font-normal font-productsans text-[14px]' : 'flex text-start ml-9 font-normal font-productsans text-[14px] '}`}
          >
            {getTranslation(language, 'sales', 'Ticket Info')}
          </div>
        ),
        accessorKey: 'amount',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p
                className={` ${isSmallScreen ? 'text-start ml-1 font-productsans font-normal text-[14px]' : 'text-start ml-7 font-productsans font-normal text-[14px] '}`}
                onClick={() => {
                  setticketInfo({
                    isModalOpen: true,
                    ticket: rowObject,
                  });
                }}
              >
                View Ticket Info
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex items-end justify-end font-normal font-productsans text-[14px]' : 'flex items-end justify-end mr-20 ml-3 font-normal font-productsans text-[14px]'}`}
          >
            {getTranslation(language, 'sales', 'Details')}
          </div>
        ),
        accessorKey: 'details',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div
              className={` ${isSmallScreen ? 'flex items-center justify-center font-normal text-[13px] font-productsans underline cursor-pointer mr-6' : 'flex items-center justify-center font-normal text-[13px] font-productsans underline cursor-pointer'}`}
              onClick={() =>
                setaccountInfo({
                  isModalOpen: true,
                  id: rowObject?.id,
                  row: rowObject,
                })
              }
            >
              View More
            </div>
          );
        },
      },
    ];
  }, [isTicket, resetTable, tableData, language, currency]);

  return (
    <>
      <Paper
        className="flex justify-between items-center p-4 min-h-screen"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 ml-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 6 }}>
            <div className="ml-1 rtl:mr-1 rtl:ml-0 text-[33px] font-semibold text-[#ffffff] font-productsans">
              {isTicket
                ? ` ${getTranslation(language, 'sales', 'Ticket')} `
                : getTranslation(language, 'sales', 'Sales')}
            </div>
          </Grid.Col>

          {currentRole === '2' && (
            <Grid.Col span={{ base: 12, md: 3, lg: 2 }}></Grid.Col>
          )}

          <Grid.Col span={{ base: 12, md: 3, lg: 3 }}>
            <DebouncedTextInput
              placeholder={getTranslation(language, 'sales', 'Search order ID')}
              value={searchValues.searchValue}
              onDebouncedChange={event =>
                setValuesById({ searchValue: event || '' })
              }
              styles={textAreaStyle}
              classNames={{ input: 'text-input' }}
              radius={'6px'}
            />
          </Grid.Col>

          {currentRole !== '2' && (
            <Grid.Col span={{ base: 12, md: 3, lg: 3 }}>
              <Select
                placeholder={getTranslation(
                  language,
                  'sales',
                  'Select Ticket Type'
                )}
                data={organizeDropDownData(referenceData?.ticketTypes)}
                value={searchValues.type}
                clearable
                rightSection={<IconChevronDown size={16} color="white" />}
                onChange={value => setValuesById({ type: value || '' })}
                styles={filterStyle}
                classNames={{ input: 'custom-input' }}
                radius="6px"
              />
            </Grid.Col>
          )}
        </Grid>
        <Grid className="py-5 px-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl  mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
              style={{ height: 'auto' }}
            >
              <div className="mt-4">
                <Table
                  data={tableData}
                  columns={isTicket ? ticketColumn : commoncolumns}
                  isLoading={isLoading}
                  paginationInfo={paginationInfo}
                  handlePagination={handlePagination}
                  isWithActionButton={currentRole === '2'}
                />
              </div>
            </div>
          </Grid.Col>
        </Grid>

        <SaleView
          isTicket
          opened={accountInfo?.isModalOpen}
          setOpened={setaccountInfo}
          id={accountInfo?.id}
          isAdmin={currentRole === '3' || currentRole === '1'}
          row={accountInfo?.row}
        />

        <TicketDetailModal
          opened={ticketInfo?.isModalOpen}
          setOpened={setticketInfo}
          ticketData={ticketInfo?.ticket}
        />
      </Paper>
    </>
  );
};

export default Tickets;
