import { Button, Loader, Modal } from '@mantine/core';
import { IconMapPin } from '@tabler/icons-react';
import { useFormik } from 'formik';
import { useEffect, useState } from 'react';
import { useSelector } from 'react-redux';
import { fetchData, postData } from '../../api/api';
import { getTranslation } from '../../layout/languages';
import { initialNotification } from '../../utils/common/constant.objects';
import { GigStatus, RefundStatus } from '../../utils/common/enums';
import {
  formatDate,
  formatNumber,
  handleError,
  handleSuccess,
} from '../../utils/common/function';
import ReportEvents from './report-events';

const SaleView = ({
  isTicket,
  opened,
  setOpened,
  id,
  isAdmin,
  row,
  isDashboard,
}: any) => {
  const [accountInfo, setaccountInfo] = useState({
    isModalOpen: false,
    id: '',
  });
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  const [isMainModalOpen, setMainModalOpen] = useState(false);
  const [saleData, setsaleData] = useState<any>();
  const { userInfo } = useSelector((state: any) => state?.userInfo);
  const [notification, setNotification] = useState(initialNotification);
  const [isLoading, setIsLoading] = useState(false);
  const [isFetching, setIsFetching] = useState(false);
  const currentRole = userInfo?.roleId;

  const formik = useFormik({
    enableReinitialize: true,
    initialValues: {
      transactionId: isAdmin ? row?.transaction?.id : id,
      requestedByUserId: userInfo?.id,
      reason: '',
    },
    onSubmit: values => {
      handleRefund(values);
    },
  });

  useEffect(() => {
    if (isTicket && row) {
      const ticketObj = {
        ...row,
        event: row.ticket.event,
        transaction: row?.transaction,
        fromUser: row?.user,
        ticket: { ticket: row?.ticket, seats: row?.seats },

        amount: row?.transaction?.actualAmount ?? row?.ticket?.price ?? 0,
      };
      setsaleData(ticketObj);
    } else if (id) {
      fetchSaleData();
    }
  }, [id, isTicket, row]);

  const fetchSaleData = () => {
    setIsFetching(true);

    fetchData(`/transactions/${id}`)
      .then((response: any) => {
        setsaleData(response);
      })
      .catch((error: any) => {
        handleError(error, 'Failed to fetch sale data');
      })
      .finally(() => {
        setIsFetching(false);
      });
  };

  const handleRefund = async (values: any) => {
    setIsLoading(true);
    postData(`/refunds`, values)
      .then(() => {
        handleSuccess('Refund requested successfully');

        setTimeout(() => {
          setOpened(false);
        }, 1000);
      })
      .catch((error: any) => {
        handleError(error);
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  return (
    <Modal
      opened={opened}
      centered
      size="md"
      onClose={() => setOpened(false)}
      withCloseButton={false}
      styles={{
        content: {
          borderRadius: '15px',
          backgroundColor: '#6D3EBD',
          padding: 0,
        },
        body: {
          padding: 0,
        },
        inner: {
          padding: 0,
          margin: 0,
          maxWidth: '100vw',
        },
      }}
      overlayProps={{
        backgroundOpacity: 0.55,
        blur: 3,
      }}
      transitionProps={{ transition: 'fade-up', duration: 300 }}
      title=""
    >
      {isFetching ? (
        <div className="flex justify-center items-center h-64">
          <Loader color="white" size="lg" />
        </div>
      ) : (
        <>
          <div className="relative w-full">
            {saleData?.event?.eventImage ? (
              <img
                src={saleData?.event?.eventImage}
                alt="image"
                className="rounded-t-md h-[180px] w-full object-cover"
              />
            ) : (
              <div className="rounded-t-md h-[30px] w-full object-cover"></div>
            )}
            <Modal.CloseButton
              onClick={() => setOpened(false)}
              style={{
                position: 'absolute',
                top: 10,
                right: 10,
                border: 'none',
                outline: 'none',
                boxShadow: 'none',
                color: '#ffffff',
                backgroundColor: 'transparent',
                transition: 'background-color 0.2s ease-in-out',
                width: 33,
                height: 33,
              }}
            />
          </div>
          <div className="mt-4">
            <div className="flex flex-row justify-between ml-8 mr-6">
              <div className="text-[20px] font-productsans text-[#F4F4FE]">
                {currentRole === '3' || currentRole === '1' || isDashboard ? (
                  <div>
                    {saleData?.event?.title ? saleData?.event?.title : 'N/A'}
                  </div>
                ) : (
                  <div>
                    {saleData?.fromUser?.fullNameOrBusinessName
                      ? saleData?.fromUser?.fullNameOrBusinessName
                      : 'N/A'}
                  </div>
                )}
              </div>
              <div className="text-[18px] font-productsans text-[#F4F4FE]">
                <div>
                  {`${currency === '1' ? '$' : 'BHD ≈'} ${formatNumber(
                    currency === '1'
                      ? saleData?.transaction?.actualAmount
                      : saleData?.transaction?.convertedActualAmount.toFixed(3)
                  )}`}
                </div>
              </div>
            </div>
            <div className="flex flex-row justify-between ml-8 mr-6">
              <div className="text-[14px] font-productsans text-[#F4F4FE]">
                {currentRole === '3' || currentRole === '1' || isDashboard ? (
                  `${saleData?.fromUser?.fullNameOrBusinessName ? saleData?.fromUser?.fullNameOrBusinessName : 'N/A'}`
                ) : (
                  <div className="text-[14px] font-productsans text-[#F4F4FE] font-normal flex flex-row justify-start">
                    <span className="mr-1 mt-0.5">
                      <IconMapPin size={17} />
                    </span>
                    <p>
                      {[
                        saleData?.event?.venue,
                        saleData?.event?.city,
                        saleData?.event?.country,
                      ]
                        .filter(Boolean)
                        .join(', ') || 'N/A'}
                    </p>
                  </div>
                )}
              </div>
              <div className="text-[14px] font-productsans text-[#F4F4FE]">
                {currentRole === '3' || isDashboard
                  ? `${getTranslation(language, 'sales', 'Amount Paid')}`
                  : currentRole === '1'
                    ? `${getTranslation(language, 'sales', 'Amount Received')}`
                    : `${getTranslation(language, 'sales', 'PerformanceFee')}`}
              </div>
            </div>
            <div className="bg-[#ffffff] w-auto h-auto bg-opacity-10 rounded-lg mb-3 ml-2 mr-2 ">
              <div
                className={`flex flex-row justify-between mt-4 text-[#ffffff] ml-1 mr-1 text-[13px] font-productsans`}
              >
                <div className="flex flex-col justify-between">
                  <p className="mt-3 ml-4 rtl:mr-4 rtl:ml-0">
                    {getTranslation(language, 'sales', 'OrderID')}
                  </p>
                  <p className="mb-2 ml-4 rtl:mr-4 rtl:ml-0">
                    {saleData?.displayOrderId
                      ? saleData?.displayOrderId
                      : 'N/A'}
                  </p>
                </div>
                <div className="border-r-[1px] border-r-caption border-color-light mt-4 mb-3"></div>
                <div className="flex flex-col justify-between">
                  <p className="mt-3">
                    {getTranslation(language, 'sales', 'Date')}
                  </p>
                  <p className="mb-2">
                    {formatDate(saleData?.createdAt?.substring(0, 10) || '')}
                  </p>
                </div>
                {currentRole === '2' && (
                  <>
                    <div className="border-r-[1px] border-r-caption border-color-light  mt-4 mb-3"></div>
                    <div className="flex flex-col justify-between">
                      <p className="mt-3 mr-4 rtl:ml-4 rtl:mr-0">
                        {getTranslation(language, 'navbar', 'EventName')}
                      </p>
                      <p className="mr-4 mb-2">
                        {saleData?.event?.title
                          ? saleData?.event?.title
                          : 'N/A'}
                      </p>
                    </div>
                  </>
                )}
                {currentRole === '3' && isDashboard && (
                  <>
                    <div className="border-r-[1px] border-r-caption border-color-light  mt-4 mb-3"></div>
                    <div className="flex flex-col justify-between">
                      <p className="mt-3 mr-4">
                        {getTranslation(language, 'sales', 'Ticket Type')}
                      </p>
                      <p className="mr-4 mb-2">
                        {referenceData?.ticketTypes?.find(
                          (tt: any) => tt.id === saleData?.ticket?.ticket?.name
                        )?.name || 'N/A'}
                      </p>
                    </div>
                  </>
                )}
                {currentRole === '1' && (
                  <>
                    <div className="border-r-[1px] border-r-caption border-color-light  mt-4 mb-3"></div>
                    <div className="flex flex-col justify-between">
                      <p className="mt-3 mr-4">
                        {getTranslation(language, 'sales', 'Commission Type')}
                      </p>
                      <p className="mr-4 mb-2">
                        {' '}
                        {saleData?.transaction?.type}
                      </p>
                    </div>
                  </>
                )}
                {(currentRole === '3' ||
                  currentRole === '1' ||
                  isDashboard) && (
                  <>
                    <div className="border-r-[1px] border-r-caption border-color-light mt-4 mb-3"></div>
                    <div className="flex flex-col justify-between">
                      <p className="mt-3 mr-4 rtl:ml-4 rtl:mr-0">
                        {getTranslation(language, 'sales', 'Ticket Bought')}
                      </p>
                      <p className="mr-4 mb-2 rtl:ml-4 rtl:mr-0">
                        {saleData?.ticket?.seats?.length || 0} Pcs
                      </p>
                    </div>
                  </>
                )}
              </div>
            </div>
            {currentRole === '2' && (
              <div className="px-2 mb-3 mr-2 rtl:ml-2.5 rtl:mr-0">
                <Button
                  autoContrast
                  m={5}
                  size="sm"
                  styles={{
                    root: {
                      background: '#D83F87',
                    },
                  }}
                  style={{
                    height: 42,
                    width: '100%',
                  }}
                  className="px-2"
                  onClick={() => {
                    setaccountInfo({
                      isModalOpen: true,
                      id: saleData?.id,
                    });
                  }}
                >
                  <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                    {getTranslation(language, 'sales', 'Report Event')}
                  </div>
                </Button>
              </div>
            )}
            {currentRole === '3' && !isDashboard && (
              <div className="px-2 mb-3 mr-2 rtl:ml-2 rtl:mr-0">
                <Button
                  autoContrast
                  m={5}
                  size="sm"
                  styles={{
                    root: {
                      background: '#D83F87',
                    },
                  }}
                  style={{
                    height: 42,
                    width: '100%',
                  }}
                  className="px-2"
                  onClick={() => formik.handleSubmit()}
                  loading={isLoading}
                  loaderProps={{ type: 'dots' }}
                >
                  {!isLoading && (
                    <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                      {saleData?.refund?.status === RefundStatus.PROCESSED
                        ? `${getTranslation(language, 'sales', 'Refund')}`
                        : saleData?.refund?.status === GigStatus.PENDING
                          ? `${getTranslation(language, 'sales', 'Pending')}`
                          : isTicket
                            ? `${getTranslation(language, 'sales', 'Cancel Booking')}`
                            : `${getTranslation(language, 'sales', 'Refund')}`}
                    </div>
                  )}
                </Button>
              </div>
            )}
          </div>
          <ReportEvents
            opened={accountInfo?.isModalOpen}
            setOpened={setaccountInfo}
            id={accountInfo?.id}
          />
        </>
      )}
    </Modal>
  );
};

export default SaleView;
