import {
  Button,
  Modal,
  NumberInput,
  Select,
  Textarea,
  TextInput,
} from '@mantine/core';
import { IconChevronDown } from '@tabler/icons-react';
import { useFormik } from 'formik';
import { useEffect, useState } from 'react';
import { useSelector } from 'react-redux';
import * as Yup from 'yup';
import { fetchData, postData } from '../../api/api';
import { inputStyle2, inputStyle3 } from '../../theme/common.style';

import { handleError, handleSuccess } from '../../utils/common/function';
const ReportEvents = ({ opened, setOpened, id }: any) => {
  const [isLoading, setIsLoading] = useState(false);

  const { userInfo } = useSelector((state: any) => state?.userInfo);

  const currentRole = userInfo?.roleId;

  const formik = useFormik({
    enableReinitialize: true,
    initialValues: {
      entertainerId: userInfo?.id,
      eventId: '',
      name: '',
      reason: '',
      description: '',
      isResolved: false,
    },
    validationSchema: Yup.object().shape({
      name: Yup.string().required('Name is required'),
      description: Yup.string().required('Description is required'),

      reason: Yup.string().required('Reason is required'),
    }),
    onSubmit: values => {
      setIsLoading(true);
      postData('/report-event', values)
        .then(() => {
          handleSuccess('The event was successfully reported!');

          // Close the modal after 1 second
          setTimeout(() => {
            setOpened(false);
          }, 1000);
        })
        .catch((error: any) => {
          // Show error notification using the global error handler
          handleError(error);
        })
        .finally(() => {
          // Stop loading regardless of success or failure
          setIsLoading(false);
        });
    },
  });
  const resetAndClose = () => {
    formik.resetForm();
    setOpened(false);
  };

  const handleClose = () => {
    resetAndClose();
  };

  useEffect(() => {
    if (id) {
      setIsLoading(true);

      fetchData(`/transactions/${id}`)
        .then((response: any) => {
          if (response?.eventId) {
            formik.setFieldValue('eventId', response.eventId);
          }
        })
        .catch(err => handleError(err, 'Error fetching transaction'))
        .finally(() => {
          setIsLoading(false);
        });
    }
  }, [id]);

  return (
    <Modal
      opened={opened}
      centered
      size="md"
      onClose={() => handleClose()}
      withCloseButton={false}
      styles={{
        content: {
          borderRadius: '15px',
          backgroundColor: '#452975',
          padding: 0, // Remove default padding
        },
        body: {
          padding: 0, // Ensure no padding inside modal body
        },
        inner: {
          padding: 0, // Ensures modal wrapper has no padding
          margin: 0,
          maxWidth: '100vw',
        },
      }}
      overlayProps={{
        backgroundOpacity: 0.55,
        blur: 3,
      }}
      transitionProps={{ transition: 'fade-up', duration: 300 }}
      title=""
    >
      <div className="relative w-full flex flex-row justify-between">
        <div className="mt-4 text-[#ffffff] font-productsans ml-4  text-[18px]">
          Report Events
        </div>
        <Modal.CloseButton
          onClick={handleClose}
          style={{
            position: 'absolute',
            top: 10,
            right: 10,
            border: 'none',
            outline: 'none',
            boxShadow: 'none',
            color: '#ffffff',
            backgroundColor: 'transparent', // Ensure initial state is transparent
            transition: 'background-color 0.2s ease-in-out',
            width: 33,
            height: 33,
          }}
        />
      </div>

      <div className="mt-8 ">
        <form onSubmit={formik.handleSubmit}>
          <TextInput
            className="custom-input px-4 py-4"
            label="Your Name"
            placeholder="Enter your name"
            value={formik.values?.name}
            onChange={e => formik.setFieldValue('name', e.target.value)}
            error={
              formik.errors.name &&
              (formik.touched.name || formik.submitCount > 0)
                ? formik.errors.name
                : null
            }
            radius="8px"
            styles={inputStyle2}
          />
          <NumberInput
            className="custom-input px-4 py-4"
            label="Event ID"
            placeholder="Enter your id"
            hideControls
            value={
              formik.values.eventId !== '' && formik.values.eventId !== null
                ? String(formik.values.eventId).padStart(8, '0')
                : ''
            }
            onChange={value => {
              const numericValue = Number(value);
              formik.setFieldValue(
                'eventId',
                isNaN(numericValue) ? '' : numericValue
              );
            }}
            error={
              formik.errors.eventId &&
              (formik.touched.eventId || formik.submitCount > 0)
                ? formik.errors.eventId
                : null
            }
            radius="8px"
            styles={inputStyle2}
            disabled
          />

          <Select
            className="custom-input px-4 py-4"
            label="Dispute Reason"
            placeholder="Enter reason"
            rightSection={<IconChevronDown size={20} color="white" />}
            radius="8px"
            styles={inputStyle2}
            data={['Payment did not receive']}
            onChange={value => formik.setFieldValue('reason', value)}
            value={formik.values?.reason}
            error={
              formik.errors.reason &&
              (formik.touched.reason || formik.submitCount > 0)
                ? formik.errors.reason
                : null
            }
          />
          <Textarea
            className="custom-input px-4 py-4"
            label="Description"
            placeholder="Enter Your Issue Here"
            radius="8px"
            styles={inputStyle3}
            onChange={e => formik.setFieldValue('description', e.target.value)}
            value={formik.values?.description}
            error={
              formik.errors.description &&
              (formik.touched.description || formik.submitCount > 0)
                ? formik.errors.description
                : null
            }
          />
        </form>
        <div className="px-2 mr-2 mb-3">
          <Button
            autoContrast
            m={5}
            size="sm"
            styles={{
              root: {
                background: '#D83F87',
              },
            }}
            style={{
              height: 50,
              width: '100%',
            }}
            className="px-2"
            type="submit"
            onClick={() => {
              formik.handleSubmit();
            }}
          >
            <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
              Send Report
            </div>
          </Button>
        </div>
      </div>
    </Modal>
  );
};

export default ReportEvents;
