import {
  Avatar,
  Box,
  Grid,
  Group,
  Loader,
  Paper,
  Select,
  Text,
  TextInput,
} from '@mantine/core';
import {
  IconCalendar,
  IconChevronDown,
  IconClock,
  IconDots,
} from '@tabler/icons-react';
import { DatePickerInput } from '@mantine/dates';
import dayjs from 'dayjs';
import relativeTime from 'dayjs/plugin/relativeTime';
import { useEffect, useMemo, useState } from 'react';
import { useSelector } from 'react-redux';
import { useSearchParams } from 'react-router-dom';
import { fetchData } from '../../api/api';
import img from '../../assets/images/dj-night.png';
import { Button, Table } from '../../components';
import { getTranslation } from '../../layout/languages';
import { filterStyle, textAreaStyle } from '../../theme/common.style';
import {
  AMOUNT,
  paginationInfoValue,
  SORT_OPTIONS,
} from '../../utils/common/constant.objects';
import { GigStatus } from '../../utils/common/enums';
import {
  formatNumber,
  handleError,
  handleSetParams,
  organizeDropDownData,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import SalesReport from '../dashboard/charts/sales-report';
import { initialSearchValues, SearchFilter } from './initial.values';
dayjs.extend(relativeTime);
const ReportingBussiness = () => {
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(false);
  const [searchText, setSearchText] = useState('');
  const [selectedType, setSelectedType] = useState('');
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const [transactionData, setTranscationData] = useState([]);
  const [notifications, setNotifications] = useState<any[]>([]);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  const [chartData, setChartData] = useState<{
    categories: string[];
    values: number[];
  }>({
    categories: [],
    values: [],
  });
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const [resetTable, setResetTable] = useState(false);
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  useEffect(() => {
    const fetchNotifications = async () => {
      try {
        setLoading(true);
        setError(false);
        const url = `notifications/${userInfo?.id}`;
        const response = (await fetchData(url)) as {
          data: Notification[];
        };
        setNotifications(response.data || []);
      } catch (error) {
        handleError(error, 'Error fetching notifications');
        setError(true);
      } finally {
        setLoading(false);
      }
    };

    if (userInfo?.id) {
      fetchNotifications();
    }
  }, [userInfo]);
  const filteredData = transactionData?.filter((item: any) => {
    const matchesSearch = item?.displayOrderId
      ?.toString()
      .toLowerCase()
      .includes(searchText.toLowerCase());

    const matchesType = selectedType
      ? item?.type?.toLowerCase() === selectedType.toLowerCase()
      : true;

    return matchesSearch && matchesType;
  });

  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;
    const searchValue1 =
      searchParams.get('searchValue1') ?? initialSearchValues.searchValue1;
    const searchValue3 =
      searchParams.get('searchValue3') ?? initialSearchValues.searchValue3;
    return {
      searchValue,
      type,
      limit,
      sort,
      searchValue1,
      searchValue3,
    };
  };
  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );

  const handleFetchDataByFilter = () => {
    setLoading(true);

    const baseFilters = [];

    if (searchValues?.searchValue) {
      baseFilters.push({
        field: 'transaction.displayOrderId',
        operator: 'like',
        value: searchValues.searchValue?.toString(),
      });
    }

    baseFilters.push({
      field: 'transaction.toUserId',
      operator: 'eq',
      value: userInfo?.id,
    });

    if (searchValues.type) {
      baseFilters.push({
        field: 'event.ticketTypeId',
        operator: 'eq',
        value: searchValues.type,
      });
    }

    if (searchValues.limit) {
      baseFilters.push({
        field: 'transaction.amount',
        operator: 'lte',
        value: searchValues.limit,
      });
    }
    if (searchValues.searchValue1) {
      baseFilters.push({
        field: 'transaction.createdAt',
        operator: 'gte',
        value: new Date(
          new Date(searchValues.searchValue1).setDate(
            new Date(searchValues.searchValue1).getDate() - 1
          )
        ).toISOString(),
      });
    }

    if (searchValues.searchValue3) {
      baseFilters.push({
        field: 'transaction.createdAt',
        operator: 'lte',
        value: new Date(
          new Date(searchValues.searchValue3).setDate(
            new Date(searchValues.searchValue3).getDate() + 2
          )
        ).toISOString(),
      });
    }

    const filters = removeEmptyValueFilters(baseFilters);
    const filterObject = JSON.stringify({ filter: filters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });

    if (searchValues.sort) {
      queryParams.append(
        'orderBy',
        JSON.stringify({
          displayOrderId: searchValues.sort,
          createdAt: searchValues.sort,
        })
      );
    }

    const fetchUrl = `transactions?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        if (response?.data) {
          setTranscationData(response.data);
        }
      })
      .catch(err => handleError(err, 'Error fetching transaction'))
      .finally(() => {
        setLoading(false);
      });
  };

  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };
  useEffect(() => {
    handleSearchButtonClick();
  }, [
    resetTable,
    searchValues.searchValue,
    searchValues.searchValue1,
    searchValues.searchValue3,
    searchValues.limit,
    searchValues.type,
    searchValues.sort,
  ]);
  useEffect(() => {
    handleFetchDataByFilter();
  }, [currency]);

  const columns = useMemo(
    () => [
      {
        header: (
          <div className="flex text-start ml-1 font-normal font-productsans text-[14px]">
            ID
          </div>
        ),
        accessorKey: 'id',
        cell: (info: any) => {
          const rowObject = info?.row?.original;

          return (
            <div className="flex text-start ml-2">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.displayOrderId}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center ml-7 font-normal font-productsans text-[14px]">
            Client Name
          </div>
        ),
        accessorKey: 'event',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start ml-5 font-productsans font-normal text-[14px]">
                {rowObject?.fromUser?.fullNameOrBusinessName}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center ml-12 font-normal font-productsans text-[14px]">
            Events
          </div>
        ),
        accessorKey: 'event',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-10">
              <div>
                {rowObject?.event?.image ? (
                  <img
                    src={rowObject.event.image}
                    alt="Image"
                    className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4"
                    loading="lazy"
                  />
                ) : (
                  <div className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 bg-gray-300 flex items-center justify-center text-xs font-semibold uppercase">
                    {rowObject?.event?.title?.[0] || 'E'}
                  </div>
                )}
              </div>
              <p className="text-start ml-2 font-productsans font-normal text-[14px]">
                {rowObject?.event?.title || 'N/A'}
              </p>
            </div>
          );
        },
      },

      {
        header: (
          <div className="flex text-center items-center ml-3 font-normal font-productsans text-[14px]">
            Date
          </div>
        ),
        accessorKey: 'createdAt',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const date = new Date(rowObject?.createdAt).toLocaleDateString(
            'en-GB'
          );

          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {date}
              </p>
            </div>
          );
        },
      },

      {
        header: (
          <div className="flex text-start ml-4 font-normal font-productsans text-[14px]">
            Ticket Type
          </div>
        ),
        accessorKey: 'id',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-2">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.event?.ticketType?.name || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-3 font-normal font-productsans text-[14px]">
            Ticket Bought
          </div>
        ),
        accessorKey: 'id',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-0">
              <p className="text-start font-normal font-productsans text-[14px]">
                {'5Pcs'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-0 font-normal font-productsans text-[14px]">
            Ticket Amount
          </div>
        ),
        accessorKey: 'id',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-2">
              <p className="text-start font-normal font-productsans text-[14px]">
                {`${currency === '1' ? '$' : 'BHD ≈'} ${currency === '1' ? formatNumber(rowObject?.amount) : formatNumber(rowObject?.convertedAmount) || 'N/A'}`}
              </p>
            </div>
          );
        },
      },
    ],
    [filteredData, currency]
  );

  const exportCSV = () => {
    if (!filteredData || filteredData.length === 0) return;

    const headers = [
      'ID',
      'Client Name',
      'Events',
      'Date',
      'Ticket Type',
      'Ticket Bought',
      'Ticket Amount',
    ];

    const rows = filteredData.map((row: any) => {
      const formattedDate = new Date(row?.createdAt).toLocaleDateString(
        'en-GB'
      );
      const location = `${row?.event?.venue || ''}, ${row?.event?.city || ''}, ${row?.event?.country || ''}`;
      const paymentStatus =
        row?.status === 'completed'
          ? 'Done!'
          : row?.status === GigStatus.PENDING
            ? 'Pending'
            : row?.status === 'Refunded'
              ? 'Refunded'
              : '';

      return [
        row?.displayOrderId || '',
        row?.fromUser?.fullNameOrBusinessName || '',
        row?.event?.title || '',
        formattedDate,
        row?.type || '',
        '5Pcs',
        row?.amount || '',
      ];
    });

    const csvContent = [headers, ...rows]
      .map(e => e.map(val => `"${val}"`).join(','))
      .join('\n');

    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.setAttribute('download', 'Tickets_Sales_Report.csv');
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };
  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById,
    }));
  };
  return (
    <>
      <Paper
        className="flex justify-between items-center p-4 min-h-screen bg-[radial-gradient(ellipse_at_bottom,_var(--tw-gradient-stops))] from-[#50347d] via-[#3B1E6D] to-[#3B1E6D]"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <div className="flex justify-between items-center mb-4 px-4 mt-2">
          <h2 className="text-white text-2xl font-semibold text-color-white font-productsans">
            Reporting & Analytics
          </h2>
          <div className="flex gap-3">
            <Button
              size="sm"
              style={{
                background: '#27B4B6',
                color: 'white',
                fontFamily: 'Product sans',
              }}
              onClick={exportCSV}
            >
              Export CSV/PDF
            </Button>
          </div>
        </div>
        <Grid className="px-4 py-4" gutter={'xl'}>
          <Grid.Col span={{ base: 12, md: 12, lg: 8 }}>
            <div className="rounded-2xl font-semibold font-productsans text-[22px] h-[100%] text-[#ffffff] p-5 mt-1 bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <div className=" flex flex-row justify-between">
                <div className="ml-1.5 font-productsans">Live Sales</div>
                <div>
                  <IconDots />
                </div>
              </div>
              <div className="mt-3 ">
                <SalesReport
                  data={{
                    categories: chartData?.categories,
                    values: chartData?.values,
                  }}
                />
              </div>
            </div>
          </Grid.Col>
          <Grid.Col span={{ base: 12, md: 12, lg: 4 }}>
            <div className="rounded-2xl h-[432px] font-semibold text-[22px] text-white bg-[#ffffff] p-5 bg-opacity-5 border border-[#623F9D]">
              {/* White heading */}
              <div className="flex justify-between">
                <span className="text-white text-color-white font-productsans ">
                  Notification
                </span>
                <IconDots color="white" />
              </div>

              <div className="mt-4 space-y-2 overflow-y-auto max-h-[340px]">
                {loading ? (
                  <div className="flex items-center justify-center mt-28">
                    <Loader size="lg" color="#C28CFF" />
                  </div>
                ) : notifications.length === 0 ? (
                  <Text
                    style={{
                      color: 'white',
                      fontSize: '16px',
                      fontFamily: 'Product Sans',
                      textAlign: 'center',
                      marginTop: '60px',
                    }}
                  >
                    No notifications found.
                  </Text>
                ) : (
                  notifications?.slice(0, 3).map((item, index) => (
                    <div
                      key={index}
                      className="flex items-center gap-3 p-3 bg-[#ffffff0d] rounded-md"
                    >
                      <div className="bg-[#FFFFFF1A] p-2 rounded-full flex items-center justify-center w-8 h-8">
                        <Avatar
                          src={
                            item?.profile?.profilePicture ||
                            item?.event?.eventImage ||
                            null
                          }
                          alt="User"
                          size={50}
                          radius="xl"
                          color="gray"
                        >
                          {item?.profile?.fullNameOrBusinessName[0] ||
                            item?.event?.title[0] ||
                            item.type[0] ||
                            ''}
                        </Avatar>
                      </div>
                      <div className="flex flex-col justify-between">
                        <p className="font-semibold text-[13px] text-color-white font-productsans">
                          {item?.message}
                        </p>

                        <Box mt="xs">
                          <Group
                            style={{
                              color: 'white',
                              fontSize: '12px',
                              fontWeight: 'bold',
                              fontFamily: 'Product Sans',
                              justifyContent: 'flex-start',
                            }}
                          >
                            <div className="flex flex-row gap-2">
                              <IconClock className="mt-1" size={14} />
                              <Text>{dayjs(item.createdAt).fromNow()}</Text>
                            </div>
                            <div className="flex flex-row gap-2 font-normal">
                              |
                            </div>
                            <div className="flex flex-row gap-2">
                              <IconCalendar className="mt-1" size={14} />
                              <Text>
                                {new Date(item.createdAt).toLocaleDateString()}
                              </Text>
                            </div>
                          </Group>
                        </Box>
                      </div>
                    </div>
                  ))
                )}
              </div>
            </div>
          </Grid.Col>
        </Grid>
        <Grid className="py-5 px-1">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl ml-4 mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
              style={{ height: 'auto' }}
            >
              {/* Heading & Search Filters in One Line */}

              <Grid className="mt-4 mr-7 ">
                <Grid.Col span={{ base: 12, md: 12, lg: 2 }}>
                  <div className="text-[22px] text-color-white font-productsans">
                    Ticket Sales
                  </div>
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.75 }}>
                  <TextInput
                    placeholder={getTranslation(
                      language,
                      'navbar',
                      'Search order ID'
                    )}
                    value={searchValues.searchValue}
                    onChange={event =>
                      setValuesById({ searchValue: event.currentTarget.value })
                    }
                    styles={textAreaStyle}
                    classNames={{ input: 'text-input' }}
                    radius={'6px'}
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.5 }}>
                  <Select
                    placeholder={getTranslation(
                      language,
                      'navbar',
                      'Ticket Type'
                    )}
                    data={organizeDropDownData(referenceData?.ticketTypes)}
                    value={searchValues.type}
                    clearable
                    rightSection={<IconChevronDown size={16} color="white" />}
                    onChange={value => setValuesById({ type: value || '' })}
                    styles={filterStyle}
                    classNames={{ input: 'custom-input' }}
                    radius="6px"
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.65 }}>
                  <Select
                    placeholder={getTranslation(
                      language,
                      'navbar',
                      'Amount Limit'
                    )}
                    data={AMOUNT}
                    value={searchValues.limit}
                    rightSection={<IconChevronDown size={16} color="white" />}
                    onChange={value => value && setValuesById({ limit: value })}
                    styles={filterStyle}
                    classNames={{ input: 'custom-input' }}
                    radius="6px"
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.5 }}>
                  <Select
                    placeholder="Sort By"
                    data={SORT_OPTIONS}
                    value={searchValues.sort}
                    clearable
                    rightSection={<IconChevronDown size={16} color="white" />}
                    onChange={value => setValuesById({ sort: value || '' })}
                    styles={filterStyle}
                    classNames={{ input: 'custom-input' }}
                    radius="6px"
                  />
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 2, lg: 1.25 }}>
                  <div className="text-[14px] mt-2.5 font-productsans px-2">
                    Select Date
                  </div>
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 3, lg: 2.25 }}>
                  <div>
                    <DatePickerInput
                      type="range"
                      placeholder="Pick date range"
                      required
                      value={[
                        searchValues?.searchValue1
                          ? new Date(searchValues.searchValue1)
                          : null,
                        searchValues?.searchValue3
                          ? new Date(searchValues.searchValue3)
                          : null,
                      ]}
                      onChange={(value: any) => {
                        if (value) {
                          setValuesById({
                            searchValue1: value[0]
                              ? value[0].toISOString()
                              : '',
                            searchValue3: value[1]
                              ? value[1].toISOString()
                              : '',
                          });
                        }
                      }}
                      valueFormat="M/D/YY"
                      style={filterStyle}
                      radius={'6px'}
                      styles={{
                        input: {
                          backgroundColor: 'transparent',
                          borderColor: '#FFFFFF26',
                          color: '#ffffff',
                          fontFamily: 'Product Sans',
                          fontSize: '12px',
                          height: '48px',
                        },
                        placeholder: { color: 'white', fontWeight: 'normal' },
                      }}
                      rightSection={
                        <IconChevronDown size={16} color="#ffffff" />
                      }
                      leftSection={<IconCalendar size={20} color="#ffffff" />}
                    />
                  </div>
                </Grid.Col>
              </Grid>
              {/* Table Section */}
              <div className="mt-7">
                <Table
                  data={filteredData}
                  columns={columns}
                  isLoading={loading}
                  paginationInfo={paginationInfo}
                  handlePagination={handlePagination}
                  isPaginated={false}
                  isWithActionButton={false}
                />
              </div>
            </div>
          </Grid.Col>
        </Grid>
      </Paper>
    </>
  );
};

export default ReportingBussiness;
