import { Grid, Paper, ScrollArea, Select } from '@mantine/core';
import { DatePickerInput } from '@mantine/dates';
import { IconCalendar, IconChevronDown, IconDots } from '@tabler/icons-react';
import { useEffect, useState } from 'react';
import { BsDownload } from 'react-icons/bs';
import { useSelector } from 'react-redux';
import { useSearchParams } from 'react-router-dom';
import { fetchData, postData } from '../../api/api';
import { Button, Table } from '../../components';
import { DebouncedTextInput } from '../../components/DebouncedTextInput';
import useScreenSize from '../../hooks/useScreenSize';
import { filterStyle, textAreaStyle } from '../../theme/common.style';
import {
  AMOUNT_LIMIT_OPTIONS,
  paginationInfoValue,
} from '../../utils/common/constant.objects';
import { GigStatus } from '../../utils/common/enums';
import {
  extractPageInfo,
  formatNumber,
  handleError,
  handleSetParams,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import CommissionOverView from '../dashboard/charts/commission-overview';
import LiveAnalytics from '../dashboard/charts/live-analytics';
import { initialSearchValues, SearchFilter } from './initial.values';
const ReportingAdmin = () => {
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  const { isSmallScreen, isMediumScreen, isLargeScreen } = useScreenSize();
  const [value, setValue] = useState<[Date | null, Date | null]>([null, null]);
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  const [searchParams, setSearchParams] = useSearchParams();

  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  const [events, setEvents] = useState([]);
  const [commsion, setCommsion] = useState([]);
  const [dashboardData, setDashboardData] = useState<any>(null);

  const [chartData, setChartData] = useState();
  const [selectedRange, setSelectedRange] = useState<
    'daily' | 'weekly' | 'monthly' | 'annual'
  >('daily');
  const [selectedEarnRange, setSelectedEarnRange] = useState('All Time');
  const [currentPage, setCurrentPage] = useState(1);
  const pageSize = 5;
  const paginatedData = events.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );
  const paginatedDataCommsion = commsion.slice(
    (currentPage - 1) * pageSize,
    currentPage * pageSize
  );
  const [isLoading, setIsLoading] = useState(true);
  useEffect(() => {
    const fetchDashboardStats = async () => {
      const formatDate = (date: Date) => {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
      };

      let startDate: string;
      let endDate: string;

      const today = new Date();

      switch (selectedRange) {
        case 'daily': {
          startDate = formatDate(today);
          const tomorrow = new Date(today);
          tomorrow.setDate(today.getDate() + 1);
          endDate = formatDate(tomorrow);
          break;
        }
        case 'weekly': {
          const startOfWeek = new Date(today);
          startOfWeek.setDate(today.getDate() - 6);
          startDate = formatDate(startOfWeek);
          endDate = formatDate(today);
          break;
        }
        case 'monthly': {
          const startOfMonth = new Date(
            today.getFullYear(),
            today.getMonth(),
            1
          );
          startDate = formatDate(startOfMonth);
          endDate = formatDate(today);
          break;
        }
        case 'annual': {
          const startOfYear = new Date(today.getFullYear(), 0, 1);
          startDate = formatDate(startOfYear);
          endDate = formatDate(today);
          break;
        }
        default: {
          startDate = formatDate(today);
          const tomorrow = new Date(today);
          tomorrow.setDate(today.getDate() + 1);
          endDate = formatDate(tomorrow);
          break;
        }
      }

      // Custom range
      if (value[0] && value[1]) {
        startDate = formatDate(value[0]);
        const isSameDay = value[0].toDateString() === value[1].toDateString();

        if (isSameDay) {
          const nextDay = new Date(value[1]);
          nextDay.setDate(nextDay.getDate() + 1);
          endDate = formatDate(nextDay);
        } else {
          endDate = formatDate(value[1]);
        }
      }

      const payload = {
        startDate,
        endDate,
        commissionTimeline: selectedEarnRange,
      };

      try {
        const response: any = await postData(
          'statistics/admin-dashboard',
          payload
        );
        setDashboardData(response.data);

        // Format data for charts if needed
        if (response.data.userEngagement) {
          setChartData(response.data.userEngagement);
        }
      } catch (error) {
        console.error('Error fetching dashboard stats:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchDashboardStats();
  }, [selectedRange, value, selectedEarnRange]);

  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;
    const searchValue1 =
      searchParams.get('searchValue1') ?? initialSearchValues.searchValue1;
    const searchValue3 =
      searchParams.get('searchValue3') ?? initialSearchValues.searchValue3;
    const searchValue5 =
      searchParams.get('searchValue5') ?? initialSearchValues.searchValue5;
    const searchValue4 =
      searchParams.get('searchValue4') ?? initialSearchValues.searchValue4;
    return {
      searchValue,
      type,
      limit,
      sort,
      searchValue1,
      searchValue3,
      searchValue5,
      searchValue4,
    };
  };
  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );
  const { userInfo } = useSelector((state: any) => state?.userInfo);

  const currentRole = userInfo?.roleId;

  // State for table data
  const [tableData, setTableData] = useState([]);
  const today = new Date();
  const startOfToday = new Date(today.setHours(0, 0, 0, 0)).toISOString();
  const endOfToday = new Date(today.setHours(23, 59, 59, 999)).toISOString();
  // State for reset button
  const [resetTable, setResetTable] = useState(false);
  const handleFetchDataByFilter = () => {
    setIsLoading(true);

    const baseFilters = [
      {
        field: 'event.title',
        operator: 'eq',
        value: searchValues.searchValue,
      },
      {
        field: 'event.status',
        operator: 'eq',
        value: searchValues.type,
      },
    ];

    const filters = [...baseFilters]; // Start with baseFilters

    filters.push({
      field: 'event.startDateTime',
      operator: 'gte',
      value: new Date().toISOString(),
    });
    if (searchValues.searchValue1) {
      filters.push({
        field: 'event.createdAt',
        operator: 'gte',
        value: new Date(
          new Date(searchValues.searchValue1).setDate(
            new Date(searchValues.searchValue1).getDate() - 1
          )
        ).toISOString(),
      });
    }

    if (searchValues.searchValue3) {
      filters.push({
        field: 'event.createdAt',
        operator: 'lte',
        value: new Date(
          new Date(searchValues.searchValue3).setDate(
            new Date(searchValues.searchValue3).getDate() + 2
          )
        ).toISOString(),
      });
    }

    const finalFilters = removeEmptyValueFilters(filters);

    const filterObject = JSON.stringify({ filter: finalFilters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });

    if (searchValues.sort) {
      queryParams.append(
        'orderBy',
        JSON.stringify({
          title: searchValues.sort,
          displayOrderId: searchValues.sort,
        })
      );
    }

    const fetchUrl = `events?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        if (!response.data || response.data.length === 0) {
          setTableData(response.data);
          if (response.total > 0) {
            setPaginationInfo({
              ...paginationInfo,
              totalRecords: response.total,
            });
          }
        } else {
          setTableData(response.data);
          const getPages = extractPageInfo(response.pages);
          setPaginationInfo({
            ...paginationInfo,
            totalRecords: response.total,
            totalPages: getPages?.totalPages ?? 0,
          });
        }
      })
      .catch(error => console.error(error))
      .finally(() => {
        setIsLoading(false);
      });
  };

  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };

  // Effect for handling search button click
  useEffect(() => {
    handleSearchButtonClick();
  }, [
    resetTable,
    searchValues.searchValue,
    searchValues.type,
    searchValues.limit,
    searchValues.sort,
    searchValues.searchValue1,
    searchValues.searchValue3,
    paginationInfo?.currentPage,
    paginationInfo?.rowPerPage,
  ]);
  const getCommission = async () => {
    setIsLoading(true);
    const startDate = searchValues?.searchValue5;
    const endDate = searchValues?.searchValue4;

    // Construct query parameters for date range
    const queryParams = new URLSearchParams();
    if (startDate) queryParams.append('startDate', startDate);
    if (endDate) queryParams.append('endDate', endDate);
    try {
      const res: any = await fetchData(
        `/transactions/commission?${queryParams.toString()}`
      );
      setCommsion(res); // Assuming res is array of event objects
    } catch (error) {
      handleError(error, 'Failed to fetch commission');
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    if (searchValues?.searchValue5 && searchValues?.searchValue4) {
      getCommission();
    }
  }, [searchValues?.searchValue5, searchValues?.searchValue4, currency]);

  const columns = [
    {
      header: (
        <div className="text-start font-productsans text-[14px] font-normal ml-3">
          ID
        </div>
      ),
      accessorKey: 'displayOrderId',
      cell: (info: any) => (
        <p className="text-start font-productsans text-[14px] font-normal ml-3">
          {info?.row?.original?.displayOrderId}
        </p>
      ),
    },
    {
      header: (
        <div className="text-start font-productsans text-[14px] font-normal ml-3">
          Client Name
        </div>
      ),
      accessorKey: 'fullNameOrBusinessName',
      cell: (info: any) => (
        <p className="text-start font-productsans text-[14px] font-normal ml-3">
          {info?.row?.original?.organizedBy?.fullNameOrBusinessName
            ? info?.row?.original?.organizedBy?.fullNameOrBusinessName
            : 'N/A'}
        </p>
      ),
    },
    {
      header: (
        <div className="text-start font-productsans text-[14px] font-normal ml-3">
          Event
        </div>
      ),
      accessorKey: 'title',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex items-center text-start">
            {rowObject?.eventImage ? (
              <img
                src={rowObject.eventImage}
                alt="Image"
                className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 object-cover"
                loading="lazy"
              />
            ) : (
              <div className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 bg-gray-300 flex items-center justify-center text-xs font-semibold uppercase">
                {rowObject?.title?.[0] || 'E'}
              </div>
            )}
            <p className="ml-2 rtl:mr-2 font-productsans font-normal text-[14px] text-start">
              {rowObject?.title || 'N/A'}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="text-start font-productsans text-[14px] font-normal ml-3">
          Location
        </div>
      ),
      accessorKey: 'venue',
      cell: (info: any) => (
        <p className="text-start font-productsans text-[14px] font-normal ml-3">
          {info?.row?.original?.venue}
        </p>
      ),
    },
    {
      header: (
        <div className="text-start font-productsans text-[14px] font-normal ml-3">
          Date
        </div>
      ),
      accessorKey: 'startDateTime',
      cell: (info: any) => {
        const date = new Date(info?.row?.original?.startDateTime);
        return (
          <p className="text-start font-productsans text-[14px] font-normal ml-3">
            {date.toLocaleDateString()}
          </p>
        );
      },
    },
    {
      header: (
        <div className="text-start font-productsans text-[14px] font-normal ml-3">
          Status
        </div>
      ),
      accessorKey: 'status',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        const Status = rowObject?.status?.toLowerCase();
        return (
          <p
            className="text-start font-productsans text-[14px] font-normal ml-3"
            style={{
              color: Status === GigStatus.PENDING ? '#FF7900' : '#20E2E4',
            }}
          >
            {info?.row?.original?.status}
          </p>
        );
      },
    },
  ];

  const columnsCommission = [
    {
      header: (
        <div className="flex text-start ml-5 font-normal font-productsans text-[14px]">
          ID
        </div>
      ),
      accessorKey: 'displayOrderId',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start ml-2">
            <p className="text-start font-normal font-productsans text-[14px]">
              {rowObject?.displayOrderId}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-center items-center ml-12 font-normal font-productsans text-[14px]">
          Client Name
        </div>
      ),
      accessorKey: 'title',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start">
            <p className="text-start ml-6 font-productsans font-normal text-[14px]">
              {rowObject?.event?.organizedBy?.fullNameOrBusinessName
                ? rowObject?.event?.organizedBy?.fullNameOrBusinessName
                : 'N/A'}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-center items-center ml-12 font-normal font-productsans text-[14px]">
          Events
        </div>
      ),
      accessorKey: 'title',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex items-center text-start">
            {rowObject?.event?.eventImage ? (
              <img
                src={rowObject.event.eventImage}
                alt="Image"
                className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 object-cover"
                loading="lazy"
              />
            ) : (
              <div className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 bg-gray-300 flex items-center justify-center text-xs font-semibold uppercase">
                {rowObject?.event?.title?.[0] || 'E'}
              </div>
            )}
            <p className="ml-2 rtl:mr-2 font-productsans font-normal text-[14px] text-start">
              {rowObject?.event?.title || 'N/A'}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-start ml-5 font-normal font-productsans text-[14px]">
          Commission Type
        </div>
      ),
      accessorKey: 'type',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start ml-2">
            <p className="text-start font-normal font-productsans text-[14px]">
              {rowObject?.type}
            </p>
          </div>
        );
      },
    },

    {
      header: (
        <div className="flex text-start ml-5 font-normal font-productsans text-[14px]">
          Date
        </div>
      ),
      accessorKey: 'createdAt',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        const date = new Date(info?.row?.original?.createdAt);

        return (
          <div className="flex text-start ml-2">
            <p className="text-start font-normal font-productsans text-[14px]">
              {date.toLocaleDateString()}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-start  font-normal font-productsans text-[14px]">
          Commission Earned
        </div>
      ),
      accessorKey: 'amount',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start ml-2">
            <p className="text-start font-normal font-productsans text-[14px]">
              {`${currency === '1' ? '$' : 'BHD ≈'} ${currency === '1' ? formatNumber(rowObject?.amount) : formatNumber(rowObject?.convertedAmount) || 'N/A'}`}
            </p>
          </div>
        );
      },
    },
  ];
  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById, // Merge the new values with the existing state
    }));
  };

  const exportToCSV = (data: any[], filename: string) => {
    if (!data || data.length === 0) {
      alert('No data available to export.');
      return;
    }

    const header = Object.keys(data[0]).join(',');
    const csvRows = data.map(row =>
      Object.values(row)
        .map(value => `"${String(value).replace(/"/g, '""')}"`)
        .join(',')
    );

    const csvContent = [header, ...csvRows].join('\n');
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });

    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.setAttribute('download', `${filename}.csv`);
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  return (
    <>
      <Paper
        className="flex justify-between items-center p-4  min-h-screen "
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="flex justify-between items-center mb-4 mt-2 px-5">
          <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
            <h2 className="text-white text-2xl font-semibold text-color-white font-productsans">
              Reporting & Analytics
            </h2>
          </Grid.Col>
          <Grid.Col
            span={{ base: 12, md: 6, lg: 6 }}
            className={`${isSmallScreen ? `` : `flex flex-row justify-end`}`}
          >
            <div className="flex gap-3">
              <Button
                size="sm"
                style={{
                  background: '#27B4B6',
                  color: 'white',
                  fontFamily: 'Product sans',
                }}
              >
                Export CSV/PDF
              </Button>
              <Button
                size="sm"
                style={{
                  background: '#D83F87',
                  color: 'white',
                  fontFamily: 'Product sans',
                }}
              >
                Generate Reports
              </Button>
            </div>
          </Grid.Col>
        </Grid>
        <Grid className="px-4 py-4" gutter={'xl'}>
          <Grid.Col span={{ base: 12, md: 12, lg: 8 }}>
            <div className="rounded-2xl font-semibold font-productsans text-[22px] h-[100%] text-[#ffffff] p-5 mt-1 bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <div className=" flex flex-row justify-between">
                <div className="ml-1.5 font-productsans">Live Analytics</div>

                <div>
                  <IconDots />
                </div>
              </div>

              <div className="mt-3 ">
                <LiveAnalytics data={chartData} />
              </div>
            </div>
          </Grid.Col>
          <Grid.Col span={{ base: 12, md: 12, lg: 4 }}>
            <div className="rounded-2xl font-semibold text-[22px] text-white p-5 bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              {/* <CommissionChart /> */}
              <CommissionOverView
                commissionData={dashboardData?.commission}
                comparisonData={
                  dashboardData?.ticketSoldCommissionOverTime?.comparison
                }
              />
            </div>
          </Grid.Col>
        </Grid>
        <Grid className="py-5 px-1">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <Grid className="py-5 px-1">
              <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
                <div
                  className="rounded-2xl ml-4 mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
                  style={{ height: 'auto' }}
                >
                  {/* Heading & Filters */}
                  <Grid className="mt-4 ml-2" gutter="md" align="center">
                    <Grid.Col span={{ base: 12, md: 12, lg: 1.95 }}>
                      <div className="text-[22px] text-color-white font-productsans">
                        Running Event
                      </div>
                    </Grid.Col>

                    <Grid.Col
                      span={{ base: 12, sm: 6, md: 3, lg: 1.75 }}
                      className="ml-7 mt-1 mr-2"
                    >
                      <Button
                        style={{
                          backgroundColor: '#27B4B6',
                          color: 'white',

                          alignItems: 'center',

                          height: '40px',
                          width: '145px',
                        }}
                        className="px-2 py-2 rounded-md text-[24px] font-productsans"
                        onClick={() =>
                          exportToCSV(tableData, 'running_event_data')
                        }
                      >
                        Export CSV/PDF
                        <BsDownload size={16} className="ml-1" />
                      </Button>
                    </Grid.Col>

                    <Grid.Col span={{ base: 12, sm: 6, md: 3, lg: 1.45 }}>
                      <DebouncedTextInput
                        placeholder="Search By Event Title"
                        value={searchValues.searchValue ?? ''}
                        onChange={e =>
                          setValuesById({
                            searchValue: e.target.value ?? '',
                          })
                        }
                        styles={textAreaStyle}
                        onDebouncedChange={val => console.log(val)}
                        classNames={{ input: 'text-input' }}
                        radius="6px"
                      />
                    </Grid.Col>

                    <Grid.Col span={{ base: 12, sm: 6, md: 3, lg: 1.5 }}>
                      <Select
                        placeholder="Select Status"
                        data={[
                          'Pending',
                          'Waiting for Entertainer',
                          'Pending Payment',
                          'Rejected',
                          'Live',
                          'Ended',
                        ]}
                        rightSection={
                          <IconChevronDown size={16} color="white" />
                        }
                        value={searchValues.type ?? ''}
                        onChange={e =>
                          setValuesById({
                            type: e ?? '',
                          })
                        }
                        styles={filterStyle}
                        classNames={{ input: 'custom-input' }}
                        radius="6px"
                      />
                    </Grid.Col>

                    <Grid.Col span={{ base: 12, sm: 6, md: 3, lg: 1.5 }}>
                      <Select
                        placeholder="Amount Limit 10-20"
                        data={AMOUNT_LIMIT_OPTIONS}
                        value={paginationInfo.rowPerPage}
                        rightSection={
                          <IconChevronDown size={16} color="white" />
                        }
                        onChange={value =>
                          value && handlePagination('rowPerPage', value)
                        }
                        styles={filterStyle}
                        classNames={{ input: 'custom-input' }}
                        radius="6px"
                      />
                    </Grid.Col>

                    <Grid.Col span={{ base: 12, sm: 6, md: 2, lg: 1.25 }}>
                      <div className="text-[14px] mt-2.5 font-productsans px-2">
                        Select Date
                      </div>
                    </Grid.Col>

                    <Grid.Col span={{ base: 12, sm: 6, md: 4, lg: 2.25 }}>
                      <DatePickerInput
                        type="range"
                        placeholder="Pick date range"
                        value={[
                          searchValues?.searchValue1
                            ? new Date(searchValues.searchValue1)
                            : null,
                          searchValues?.searchValue3
                            ? new Date(searchValues.searchValue3)
                            : null,
                        ]}
                        onChange={(value: any) => {
                          if (value) {
                            setValuesById({
                              searchValue1: value[0]
                                ? value[0].toISOString()
                                : '',
                              searchValue3: value[1]
                                ? value[1].toISOString()
                                : '',
                            });
                          }
                        }}
                        valueFormat="M/D/YY"
                        styles={{
                          input: {
                            backgroundColor: 'transparent',
                            borderColor: '#FFFFFF26',
                            color: '#ffffff',
                            fontFamily: 'Product Sans',
                            fontSize: '12px',
                            height: '48px',
                            borderRadius: '6px',
                          },
                          placeholder: {
                            color: 'white',
                            fontWeight: 'normal',
                          },
                        }}
                        rightSection={
                          <IconChevronDown size={16} color="#ffffff" />
                        }
                        leftSection={<IconCalendar size={20} color="#ffffff" />}
                      />
                    </Grid.Col>
                  </Grid>

                  {/* Table Section */}
                  <div className="mt-7">
                    <ScrollArea
                      h={300}
                      scrollbars="y"
                      className="custom-scroll"
                      type="scroll"
                    >
                      <Table
                        data={tableData}
                        columns={columns}
                        isLoading={isLoading}
                        isPaginated={false}
                        handlePagination={handlePagination}
                      />
                    </ScrollArea>
                  </div>
                </div>
              </Grid.Col>
            </Grid>
          </Grid.Col>
        </Grid>
        <Grid className="py-5 px-1">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div className="rounded-2xl ml-4 mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              {/* Heading & Search Filters in One Line */}
              <Grid className=" mt-3">
                <Grid.Col span={{ base: 12, md: 3, lg: 5.75 }}>
                  {/* Title */}
                  <div className="text-[22px] text-color-white font-productsans">
                    Commission Earned
                  </div>
                </Grid.Col>
                {/* Search Filters - Responsive */}

                <Grid.Col
                  span={{ base: 12, md: 3.5, lg: 1.75 }}
                  className="ml-7 mt-1"
                >
                  {/* Export CSV/PDF Button */}
                  <Button
                    style={{
                      backgroundColor: '#27B4B6',
                      color: 'white',
                      display: 'flex',
                      alignItems: 'center',
                      gap: '6px',
                      height: '40px',
                      width: 'auto',
                    }}
                    className="px-4 py-2 rounded-md text-[24px] font-productsans"
                    onClick={() => exportToCSV(commsion, 'commission_earned')}
                  >
                    Export CSV/PDF
                    <BsDownload size={16} className="ml-3" />
                  </Button>
                </Grid.Col>
                {/* Date Input */}
                <Grid.Col span={{ base: 12, md: 2, lg: 1.25 }}>
                  <div className="text-[14px] mt-2.5 font-productsans px-2">
                    Select Date
                  </div>
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 3, lg: 2.75 }}>
                  <div>
                    <DatePickerInput
                      type="range"
                      placeholder="Pick date range"
                      required
                      value={[
                        searchValues?.searchValue5
                          ? new Date(searchValues.searchValue5)
                          : null,
                        searchValues?.searchValue4
                          ? new Date(searchValues.searchValue4)
                          : null,
                      ]}
                      onChange={(value: any) => {
                        if (value) {
                          setValuesById({
                            searchValue5: value[0]
                              ? value[0].toISOString()
                              : '',
                            searchValue4: value[1]
                              ? value[1].toISOString()
                              : '',
                          });
                        }
                      }}
                      valueFormat="M/D/YY"
                      style={filterStyle}
                      radius={'6px'}
                      styles={{
                        input: {
                          backgroundColor: 'transparent',
                          borderColor: '#FFFFFF26',
                          color: '#ffffff',
                          fontFamily: 'Product Sans',
                          fontSize: '12px',
                          height: '48px',
                        },
                        placeholder: { color: 'white', fontWeight: 'normal' },
                      }}
                      rightSection={
                        <IconChevronDown size={16} color="#ffffff" />
                      }
                      leftSection={<IconCalendar size={20} color="#ffffff" />}
                    />
                  </div>
                </Grid.Col>
              </Grid>

              {/* Table Section */}
              <div className="mt-7">
                <ScrollArea
                  h={300}
                  scrollbars="y"
                  className="custom-scroll"
                  type="scroll"
                >
                  <Table
                    data={commsion}
                    columns={columnsCommission}
                    isLoading={isLoading}
                    isPaginated={false}
                    handlePagination={handlePagination}
                  />
                </ScrollArea>
              </div>
            </div>
          </Grid.Col>
        </Grid>
      </Paper>
    </>
  );
};

export default ReportingAdmin;
