import { Button, Grid, LoadingOverlay, Modal } from '@mantine/core';
import { IconMapPin } from '@tabler/icons-react';
import { useEffect, useState } from 'react';
import { MdArrowOutward } from 'react-icons/md';
import { fetchData, putData } from '../../api/api';
import img from '../../assets/images/sale.png';
import { DisputeResolveAction } from '../../utils/common/enums';
import {
  formatDate,
  handleError,
  handleSuccess,
} from '../../utils/common/function';

interface ViewCaseProps {
  opened: boolean;
  setOpened: any;
  id: string | null;
  resetTable: any;
}

const ViewCase = ({ opened, setOpened, id, resetTable }: ViewCaseProps) => {
  const [disputeData, setDisputeData] = useState<any>(null);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    if (id) {
      fetchData(`/disputes/${id}`)
        .then((data: any) => {
          setDisputeData(data);
        })
        .catch((err: any) => handleError(err, 'Failed to fetch dispute data'));
    }
  }, [id]);

  const handleResolveDispute = async () => {
    if (!id) return;

    setLoading(true);
    try {
      const dispute =
        disputeData?.disputeType === 'Ticket'
          ? `/disputes/${id}/ticket-resolve`
          : `/disputes/${id}/resolve`;
      await putData(dispute, {
        action: DisputeResolveAction.REFUND,
      });
      resetTable();
      handleSuccess('Dispute resolved successfully');
      setOpened(false);
    } catch (err) {
      handleError(err, 'Failed to resolve dispute');
    } finally {
      setLoading(false);
    }
  };

  if (!opened) return null;

  return (
    <Modal
      opened={opened}
      centered
      size="md"
      onClose={() => setOpened(false)}
      withCloseButton={false}
      styles={{
        content: {
          borderRadius: '15px',
          backgroundColor: '#6D3EBD',
          padding: 0,
        },
        body: {
          padding: 0,
        },
        inner: {
          padding: 0,
          margin: 0,
          maxWidth: '100vw',
        },
      }}
      overlayProps={{
        backgroundOpacity: 0.55,
        blur: 3,
      }}
      transitionProps={{ transition: 'fade-up', duration: 300 }}
    >
      <LoadingOverlay visible={loading} overlayProps={{ blur: 2 }} />
      <div className="relative w-full">
        <img
          src={img}
          alt="Event"
          className="rounded-t-md h-[100px] w-full object-cover"
          loading="lazy"
        />
        <Modal.CloseButton
          onClick={() => setOpened(false)}
          style={{
            position: 'absolute',
            top: 10,
            right: 10,
            color: '#ffffff',
            backgroundColor: 'transparent',
            width: 33,
            height: 33,
          }}
        />
      </div>

      <div className="mt-4">
        <div className="flex flex-row justify-between ml-6 mr-5">
          <div className="text-[18px] font-productsans text-[#F4F4FE]">
            {disputeData?.event?.title}
          </div>
          <div className="text-[14px] font-productsans text-[#F4F4FE] mt-2">
            Event ID
          </div>
        </div>

        <div className="flex flex-row justify-between ml-6 mr-5">
          <div className="text-[14px] font-productsans text-[#F4F4FE] flex flex-row justify-start">
            <IconMapPin size={17} className="mr-1 mt-0.5" />
            <p>
              {disputeData?.event?.venue &&
              disputeData?.event?.city &&
              disputeData?.event?.country
                ? `${disputeData.event.venue}, ${disputeData.event.city}, ${disputeData.event.country}`
                : disputeData?.event?.venue ||
                  disputeData?.event?.city ||
                  disputeData?.event?.country ||
                  'N/A'}
            </p>
          </div>
          <div className="text-[18px] font-normal font-productsans text-[#F4F4FE]">
            {disputeData?.event?.displayOrderId}
          </div>
        </div>

        <div className="bg-white bg-opacity-10 rounded-lg mb-3 mx-2">
          <div className="flex flex-row justify-between mt-4 text-[#fff] ml-5 mr-5 text-[13px] font-productsans">
            <div className="flex flex-col">
              <p className="mt-3 ml-4 font-normal">Event Type</p>
              <p className="mb-2 ml-4">Rock Concert</p>
            </div>
            <div className="border-r border-r-caption border-opacity-30 my-4"></div>
            <div className="flex flex-col">
              <p className="mt-3 font-normal">Date</p>
              <p className="mb-2">
                {formatDate(
                  disputeData?.event?.startDateTime?.substring(0, 10)
                ) || 'N/A'}
              </p>
            </div>
            <div className="border-r border-r-caption border-opacity-30 my-4"></div>
            <div className="flex flex-col">
              <p className="mt-3 mr-4 font-normal">Event Size</p>
              <p className="mr-4 mb-2">
                {disputeData?.event?.maxAttendees || 'N/A'}
              </p>
            </div>
          </div>
        </div>

        <div className="text-[#BBACD7] font-productsans text-[14px] px-6 py-3">
          {disputeData?.event?.description}
        </div>

        <div className="text-[14px] px-6 py-3 bg-[#3B1E6D] bg-opacity-30">
          <div className="text-[#fff] text-[18px] font-normal font-productsans">
            Case Details:
          </div>

          <Grid>
            <Grid.Col span={8.5}>
              <div className="flex flex-row justify-between mt-4 font-normal">
                <div className="text-[#fff]">From:</div>
                <div className="text-[#fff] mx-1">Dispute Reason:</div>
              </div>
              <div className="flex flex-row justify-between -mt-1">
                <div className="text-[#BBACD7] text-[12px]">
                  {disputeData?.raisedBy?.fullNameOrBusinessName || 'N/A'}
                </div>
                <div className="text-[#BBACD7] text-[12px] ml-4">
                  {disputeData?.disputeReason || 'N/A'}
                </div>
              </div>
            </Grid.Col>
          </Grid>

          <div className="text-[#fff] text-[14px] font-normal font-productsans mt-3">
            Message
          </div>
          <div className="text-[#BBACD7] font-productsans text-[13px] py-2">
            {disputeData?.description || 'No message provided'}
          </div>

          <Grid>
            <Grid.Col span={9}>
              <div className="flex flex-row justify-between mt-3">
                <div className="text-[#fff] text-[14px] font-normal">
                  Raised By
                </div>
                <div className="text-[#fff] text-[14px] font-normal ml-7">
                  Against
                </div>
              </div>

              <div className="flex flex-row justify-between -mt-2">
                <div className="text-[#BBACD7D9] text-[13px]">
                  <div className="flex flex-row justify-start underline text-[#fff]">
                    <p>
                      {disputeData?.raisedBy?.fullNameOrBusinessName || 'N/A'}
                      {disputeData?.disputeType === 'Gig'
                        ? '(Business)'
                        : '(User)'}
                    </p>
                    <MdArrowOutward size={16} className="ml-1 mt-1" />
                  </div>
                  <p>{disputeData?.raisedBy?.email || 'N/A'}</p>
                  <p>{disputeData?.raisedBy?.phoneNumber || 'N/A'}</p>
                </div>

                <div className="text-[#BBACD7D9] text-[13px]">
                  <div className="flex flex-row justify-start underline text-[#fff]">
                    <p>
                      {disputeData?.againstUser?.fullNameOrBusinessName ||
                        'N/A'}
                      {disputeData?.disputeType === 'Ticket'
                        ? '(Business)'
                        : '(Entertainer)'}
                    </p>
                    <MdArrowOutward size={16} className="ml-1 mt-1" />
                  </div>
                  <p>{disputeData?.againstUser?.email || 'N/A'}</p>
                  <p>{disputeData?.againstUser?.phoneNumber || 'N/A'}</p>
                </div>
              </div>
            </Grid.Col>
          </Grid>
        </div>

        <div className="px-2 mb-3 mr-1 mt-2">
          <Button
            fullWidth
            size="sm"
            styles={{
              root: {
                background: '#DD3033',
                height: 42,
                '&:hover': {
                  background: '#C0292B',
                },
              },
            }}
            onClick={handleResolveDispute}
            disabled={loading}
          >
            <div className="font-productsans font-normal text-[14px] text-[#fff]">
              Issue Refund
            </div>
          </Button>
        </div>
      </div>
    </Modal>
  );
};

export default ViewCase;
