import { Grid, Paper } from '@mantine/core';
import { IconCalendar, IconChevronDown } from '@tabler/icons-react';
import img from '../../assets/images/dj-night.png';

import { useEffect, useMemo, useState } from 'react';
import { useSearchParams } from 'react-router-dom';
import { Button, Table } from '../../components';
import { filterStyle } from '../../theme/common.style';
import { paginationInfoValue } from '../../utils/common/constant.objects';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';

import { DatePickerInput } from '@mantine/dates';
import { useSelector } from 'react-redux';
import { fetchData, putData } from '../../api/api';
import useScreenSize from '../../hooks/useScreenSize';
import { getTranslation } from '../../layout/languages';
import { DisputeResolveAction, DisputeStatus } from '../../utils/common/enums';
import {
  extractPageInfo,
  formatDate,
  handleError,
  handleSetParams,
  handleSuccess,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { SearchFilter } from './initial.values';
import ViewCase from './view-case';
const RefundDisputes = () => {
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const [disputeInfo, setdisputeInfo] = useState({
    isModalOpen: false,
    id: '',
  });
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const { isSmallScreen } = useScreenSize();
  const [value, setValue] = useState<[Date | null, Date | null]>([null, null]);

  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  /////////////////////////////

  const initialSearchValues = {
    searchValue: '',
    type: '',
    limit: '',
    sort: '',
  };
  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;

    return {
      searchValue,
      type,
      limit,
      sort,
    };
  };
  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );
  const [isLoading, setIsLoading] = useState(false);

  // State for table data
  const [tableData, setTableData] = useState([]);

  // State for reset button
  const [resetTable, setResetTable] = useState(false);

  const handleFetchDataByFilter = () => {
    setIsLoading(true);

    const baseFilters = [
      {
        field: 'dispute.status',
        operator: 'neq',
        value: DisputeStatus.RESOLVED.toString(),
      },
      {
        field: 'dispute.status',
        operator: 'neq',
        value: DisputeStatus.REJECTED.toString(),
      },
    ];

    if (value[0] && value[1]) {
      const [startDate, endDate] = value;
      const adjustedEndDate = new Date(endDate);
      adjustedEndDate.setDate(adjustedEndDate.getDate() + 1);

      baseFilters.push({
        field: 'dispute.createdAt',
        operator: 'gte',
        value: startDate.toISOString(),
      });

      baseFilters.push({
        field: 'dispute.createdAt',
        operator: 'lt',
        value: adjustedEndDate.toISOString(),
      });
    } else if (value[0]) {
      baseFilters.push({
        field: 'dispute.createdAt',
        operator: 'gte',
        value: value[0].toISOString(),
      });
    } else if (value[1]) {
      const adjustedEndDate = new Date(value[1]);
      adjustedEndDate.setDate(adjustedEndDate.getDate() + 1);

      baseFilters.push({
        field: 'dispute.createdAt',
        operator: 'lt',
        value: adjustedEndDate.toISOString(),
      });
    }

    const filters = removeEmptyValueFilters(baseFilters);

    const filterObject = JSON.stringify({ filter: filters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });

    const fetchUrl = `disputes?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        if (!response.data || response.data.length === 0) {
          setTableData(response.data);
          if (response.total > 0) {
            setPaginationInfo({
              ...paginationInfo,
              totalRecords: response.total,
            });
          }
        } else {
          setTableData(response.data);
          const getPages = extractPageInfo(response.pages);
          setPaginationInfo({
            ...paginationInfo,
            totalRecords: response.total,
            totalPages: getPages?.totalPages ?? 0,
          });
        }
      })
      .catch(error => {
        handleError(error, 'Failed to fetch table data');
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };

  // Effect for handling search button click
  useEffect(() => {
    handleSearchButtonClick();
  }, [
    resetTable,
    paginationInfo?.currentPage,
    searchValues.searchValue,
    searchValues.limit,
    searchValues.sort,
    value,
  ]);

  // Effect to update state when searchParams change
  useEffect(() => {
    const newSearchValues = initializeStateFromQueryParams();

    if (JSON.stringify(newSearchValues) !== JSON.stringify(searchValues)) {
      setSearchValues(newSearchValues);
    }
  }, [searchParams]);

  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById, // Merge the new values with the existing state
    }));
  };

  const columns = useMemo(
    () => [
      {
        header: (
          <div className="flex text-start -ml-2 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'refund', 'ID')}
          </div>
        ),
        accessorKey: 'Id',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start -ml-2 ">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.displayOrderId}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-center items-center font-normal ml-3 font-productsans text-[14px]' : 'flex text-center items-center font-normal -ml-4 font-productsans text-[14px]'}`}
          >
            {getTranslation(language, 'refund', 'Dispute By')}
          </div>
        ),
        accessorKey: 'disputeBy',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div
              className={` ${
                isSmallScreen ? 'flex text-start ml-3' : 'flex text-start -ml-6'
              }`}
            >
              <p className="text-start  font-productsans font-normal text-[14px]">
                {rowObject?.raisedBy?.fullNameOrBusinessName}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-center items-center font-normal ml-3 font-productsans text-[14px]' : 'flex text-center items-center font-normal font-productsans text-[14px]'}`}
          >
            {getTranslation(language, 'refund', 'Events')}
          </div>
        ),
        accessorKey: 'event',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div
              className={` ${
                isSmallScreen ? 'flex text-start ml-3' : 'flex text-start -ml-3'
              }`}
            >
              <div>
                <img
                  src={img}
                  alt="image"
                  className="rounded-md w-9 h-6"
                  loading="lazy"
                />
              </div>
              <p className="text-start ml-2 font-productsans font-normal text-[14px]">
                {rowObject?.event?.title}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-3 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'refund', 'Location')}
          </div>
        ),
        accessorKey: 'location',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-0.5">
              <p
                className={` ${isSmallScreen ? 'text-start ml-4 font-productsans font-normal text-[14px]' : 'text-start  font-productsans font-normal text-[14px]'}`}
              >
                {[
                  rowObject?.event?.venue,
                  rowObject?.event?.city,
                  rowObject?.event?.country,
                ]
                  .filter(Boolean)
                  .join(', ') || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-center items-center font-normal ml-3 font-productsans text-[14px]' : 'flex text-center items-center font-normal ml-16 font-productsans text-[14px]'}`}
          >
            {getTranslation(language, 'refund', 'Status')}
          </div>
        ),
        accessorKey: 'status',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p
                className={` ${isSmallScreen ? 'text-start ml-5 font-productsans font-normal text-[14px]' : 'text-start ml-14 font-productsans font-normal text-[14px]'}`}
              >
                {rowObject?.status}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-center items-center font-normal ml-3 font-productsans text-[14px]' : 'flex text-center items-center font-normal ml-10 font-productsans text-[14px]'}`}
          >
            {getTranslation(language, 'refund', 'Date')}
          </div>
        ),
        accessorKey: 'date',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p
                className={` ${isSmallScreen ? 'text-start ml-5 font-productsans font-normal text-[14px]' : 'text-start ml-8 font-productsans font-normal text-[14px]'}`}
              >
                {formatDate(rowObject?.createdAt?.substring(0, 10) || '')}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex items-end justify-end mr-16 font-normal font-productsans text-[14px]' : 'flex items-end justify-end mr-14  font-normal font-productsans text-[14px]'}`}
          >
            {getTranslation(language, 'refund', 'Details')}
          </div>
        ),
        accessorKey: 'details',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div
              className={` ${isSmallScreen ? 'flex items-start justify-start font-normal ml-6 text-[13px] font-productsans underline cursor-pointer' : 'flex items-center justify-center font-normal text-[13px] font-productsans underline cursor-pointer'}`}
              onClick={() =>
                setdisputeInfo({
                  isModalOpen: true,
                  id: rowObject?.id,
                })
              }
            >
              View Case
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex items-end justify-end mr-20 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'refund', 'Action')}
          </div>
        ),
        accessorKey: 'action',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div
              className={` ${isSmallScreen ? 'flex items-center justify-center font-semibold  mt-2' : 'flex items-center justify-center font-semibold  -mt-2 -mb-2'}`}
            >
              <Button
                autoContrast
                m={5}
                size="sm"
                styles={{
                  root: {
                    background: '#418F90',
                  },
                }}
                style={{ height: 30 }}
                onClick={() =>
                  setdisputeInfo({ isModalOpen: true, id: rowObject?.id })
                }
              >
                <div className="font-productsans font-normal w-auto text-[14px] text-[#ffffff]">
                  Resolved!
                </div>
              </Button>

              <Button
                autoContrast
                m={5}
                size="sm"
                styles={{
                  root: {
                    background: '#ED272B',
                  },
                }}
                style={{ height: 30, width: 79 }}
                onClick={() =>
                  handleDeleteById(rowObject?.id, rowObject?.disputeType)
                }
              >
                <div className="font-productsans font-normal text-[14px] text-[#ffffff] ml-0.5 w-10">
                  Delete
                </div>
              </Button>
            </div>
          );
        },
      },
    ],
    [tableData, language]
  );
  const handleDeleteById = (id: string, disputeType: string) => {
    setIsLoading(true);

    const dispute =
      disputeType === 'Ticket'
        ? `/disputes/${id}/ticket-resolve`
        : `/disputes/${id}/resolve`;
    putData(dispute, {
      action: DisputeResolveAction.REJECT,
    })
      .then(() => {
        handleSuccess('Dispute deleted successfully');
        setResetTable(prev => !prev);
      })
      .catch((error: any) => {
        handleError(error);
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  return (
    <>
      <Paper
        // className="flex justify-between items-center p-4 h-full bg-gradient-to-b from-[#3B1E6D] from-0% via-[#6534BA] via-80% to-[#23073f]"
        className="flex justify-between items-center p-4  min-h-screen overflow-x-hidden"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 ">
          <Grid.Col span={{ base: 12, md: 7, lg: 6 }}>
            <div className="ml-4 text-[33px] font-semibold text-[#ffffff] font-productsans">
              {getTranslation(language, 'refund', 'Refund & Disputes')}
            </div>
          </Grid.Col>

          <Grid.Col
            span={{ base: 4.5, md: 2, lg: 1.75 }}
            className="-mr-2 rtl:-mr-20"
          >
            <div
              className={` ${isSmallScreen ? 'text-[20px] mt-2.5 font-semibold font-productsans text-[#ffffff] flex items-start justify-start ml-4' : 'text-[20px] mt-2.5 ml-4 font-semibold font-productsans text-[#ffffff] '}`}
            >
              {getTranslation(language, 'refund', 'Select Date')}
            </div>
          </Grid.Col>
          <Grid.Col span={{ base: 7.5, md: 3, lg: 4 }} className="rtl:-mr-6 ">
            <div className="mt-2 ml-4">
              <DatePickerInput
                type="range"
                placeholder={getTranslation(
                  language,
                  'refund',
                  'Pick date range'
                )}
                required
                value={value}
                onChange={setValue}
                style={filterStyle}
                valueFormat="M/D/YY"
                styles={{
                  input: {
                    backgroundColor: 'transparent',
                    borderColor: '#FFFFFF26',
                    color: '#ffffff',
                    fontFamily: 'Product Sans',
                    fontSize: '12px',
                  },
                  placeholder: { color: 'white', fontWeight: 'normal' },
                }}
                rightSection={<IconChevronDown size={16} color="#ffffff" />}
                leftSection={<IconCalendar size={20} color="#ffffff" />}
              />
            </div>
          </Grid.Col>
        </Grid>
        <Grid className="py-5 px-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl  mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
              style={{ height: 'auto' }}
            >
              <div className="mt-4">
                <Table
                  data={tableData}
                  columns={columns}
                  isLoading={isLoading}
                  paginationInfo={paginationInfo}
                  handlePagination={handlePagination}
                />
              </div>
            </div>
          </Grid.Col>
        </Grid>
        <ViewCase
          opened={disputeInfo?.isModalOpen}
          setOpened={setdisputeInfo}
          id={disputeInfo?.id}
          resetTable={() => setResetTable(!resetTable)}
        />
      </Paper>
    </>
  );
};

export default RefundDisputes;
