import { Modal } from '@mantine/core';
import { useSelector } from 'react-redux';
import { getTranslation } from '../../layout/languages';
import { useEffect, useState } from 'react';
import { fetchData } from '../../api/api';
import { handleError } from '../../utils/common/function';
// import { IconCheckCircle, IconXCircle } from '@tabler/icons-react';
import { FaCheckCircle, FaTimesCircle } from 'react-icons/fa';
import { IoMdCheckmark } from 'react-icons/io';
import { RxCross2 } from 'react-icons/rx';

const ViewProfileDetails = ({ opened, setOpened }: any) => {
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const [isLoading, setIsLoading] = useState(false);
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const [userData, setUserData] = useState<any>();

  useEffect(() => {
    if (!opened || !userInfo?.id) return;

    setIsLoading(true);

    fetchData(`/users/${userInfo?.id}/profile-completion`)
      .then((response: any) => {
        setUserData(response);
      })
      .catch((error: any) => {
        handleError(error, 'Failed to fetch user data');
      })
      .finally(() => {
        setIsLoading(false);
      });
  }, [opened, userInfo?.id]);

  return (
    <Modal
      opened={opened}
      centered
      size="md"
      onClose={() => setOpened(false)}
      withCloseButton={false}
      styles={{
        content: {
          borderRadius: '15px',
          backgroundColor: '#6D3EBD',
          padding: 0,
        },
        body: {
          padding: 0,
        },
      }}
      overlayProps={{
        backgroundOpacity: 0.55,
        blur: 3,
      }}
      transitionProps={{ transition: 'fade-up', duration: 300 }}
    >
      {/* Close button */}
      <div className="relative w-full">
        <Modal.CloseButton
          onClick={() => setOpened(false)}
          style={{
            position: 'absolute',
            top: 10,
            right: 10,
            color: '#fff',
            backgroundColor: 'transparent',
            border: 'none',
          }}
        />
      </div>

      <div className="mt-6 px-8 py-2">
        {/* Title */}
        <h2 className="text-color-white text-[20px] font-bold mb-2">
          {getTranslation(language, 'event', 'Profile Completion Report')}
        </h2>
        <p className="text-gray-200 text-sm mb-5">
          {userData?.percentage || 0}% completed
        </p>

        {/* Completed Fields */}
        <div className="mb-6">
          <h3 className="text-color-white font-semibold text-lg mb-3">
            Completed Fields
          </h3>
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
            {userData?.completedFields?.map((field: string, idx: number) => (
              <div
                key={idx}
                className="flex items-center gap-2 bg-green-500/20 border border-green-400 rounded-lg px-3 py-2"
              >
                <div className="bg-color-primary rounded-full p-0.5 inline-flex items-center justify-center">
                  <IoMdCheckmark className="text-color-white" size={20} />
                </div>

                <span className="text-color-white text-sm ">{field}</span>
              </div>
            ))}
          </div>
        </div>

        {/* Incomplete Fields */}
        <div>
          <h3 className="text-color-white font-semibold text-lg mb-3">
            Incomplete Fields
          </h3>
          <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
            {userData?.incompleteFields?.map((field: string, idx: number) => (
              <div
                key={idx}
                className="flex items-center gap-2 bg-red-500/20 border border-red-400 rounded-lg px-3 py-2"
              >
                <div className="bg-color-error rounded-full p-0.5 inline-flex items-center justify-center">
                  <RxCross2 className="text-color-white" size={18} />
                </div>

                <span className="text-color-white text-sm">{field}</span>
              </div>
            ))}
          </div>
        </div>
      </div>
    </Modal>
  );
};

export default ViewProfileDetails;
