import { useEffect, useRef, useState } from 'react';
import {
  Container,
  Paper,
  Title,
  Text,
  Button,
  FileInput,
  TextInput,
  Select,
  Textarea,
  Grid,
  Loader,
  Image,
  Group,
  NumberInput,
  Radio,
  Center,
} from '@mantine/core';
import { FaUpload } from 'react-icons/fa';

import { inputStyle4, inputStyle5 } from '../../theme/common.style';
import { fetchData, putData, uploadFile } from '../../api/api';
import { useFormik } from 'formik';
import { useSelector } from 'react-redux';

import { cleanObject, organizeDropDownData } from '../../utils/common/function';
import { Dropzone, MIME_TYPES } from '@mantine/dropzone';
import fb from '../../assets/images/fb.png';
import insta from '../../assets/images/insta.png';
import X from '../../assets/images/X.png';
import tiktok from '../../assets/images/tiktok.webp';
import snap from '../../assets/images/snap.png';
import { IoMdImages } from 'react-icons/io';
import { IconDownload, IconFile, IconTax, IconUser } from '@tabler/icons-react';
import { leftSectionIcon, numberInputStyle } from '../../theme/public.style';
import ViewProfileDetails from './profile-progress.modal';
interface profileCompletion {
  percentage: number;
}
interface UserData {
  fullNameOrBusinessName: string;
  rating: string;
  gmbProfile: string;
  businessWebsite: string;
  location: string;
  bio: string;
  clubType: string;
  address: string;
  city: string;
  country: string;
  zipCode: number;
  snapChatLink: string;
  instagramLink: string;
  email: string;
  phoneNumber: number;
  xLink: string;
  faceBookLink: string;
  tiktokLink: string;
  profilePicture: string;
  portfolioPics: string[] | null;
  clubTypeId: string;
  portfolioImages: string[];
  taxId?: string;
  businessDocumentType?: string;
  taxIdDocumentUrl?: string;
  crNumber?: string;
  profileCompletion: profileCompletion;
}
const MyProfileBussiness = () => {
  const [profilePic, setProfilePic] = useState<File | null>(null);
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const [portfolioPics, setPortfolioPics] = useState<(File | string)[]>([]);
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const [userData, setUserData] = useState<UserData | null>(null);
  const [loading, setLoading] = useState<boolean>(true);
  const [formSubmitting, setFormSubmitting] = useState(false);
  const [viewMoreOpen, setViewMoreOpen] = useState(false);
  const [uploadingImages, setUploadingImages] = useState<string[]>([]);
  const [profileInfo, setprofileInfo] = useState({
    isModalOpen: false,
  });

  useEffect(() => {
    const getUserData = async () => {
      try {
        setLoading(true);
        const data = await fetchData<UserData>(`/users/${userInfo?.id}`);
        setUserData(data);
      } catch (error) {
        console.error('Error fetching user data:', error);
      } finally {
        setLoading(false); // Stop loading in both success/failure
      }
    };
    if (userInfo?.id) {
      getUserData();
    }
  }, [userInfo?.id]);
  const formik = useFormik({
    enableReinitialize: true, // This allows Formik to reinitialize when `initialValues` changes
    initialValues: {
      fullNameOrBusinessName: userData?.fullNameOrBusinessName || 'N/A',
      profilePicture: userData?.profilePicture || '',
      portfolioPics: userData?.portfolioPics || [],
      rating: userData?.rating || 'N/A',
      email: userData?.email || '',
      phoneNumber: userData?.phoneNumber || 'N/A',
      clubTypeId: userData?.clubTypeId || '',
      address: userData?.address || 'N/A',
      gmbProfile: userData?.gmbProfile || 'N/A',
      businessWebsite: userData?.businessWebsite || 'N/A',
      city: userData?.city || 'N/A',
      country: userData?.country || 'N/A',
      zipCode: userData?.zipCode || 'N/A',
      bio: userData?.bio || 'N/A',
      faceBookLink: userData?.faceBookLink || 'N/A',
      tiktokLink: userData?.tiktokLink || 'N/A',
      instagramLink: userData?.instagramLink || 'N/A',
      xLink: userData?.xLink || 'N/A',
      snapChatLink: userData?.snapChatLink || 'N/A',
      roleId: userInfo?.roleId || 'N/A',
      taxId: userData?.taxId || '',
      businessDocumentType: userData?.businessDocumentType || '',
      taxIdDocumentUrl: userData?.taxIdDocumentUrl || '',
      crNumber: userData?.crNumber || '',
    },

    onSubmit: async values => {
      const formData = new FormData();
      formData.append('fullNameOrBusinessName', values.fullNameOrBusinessName);
      formData.append('email', values.email);
      formData.append('rating', values.rating);
      formData.append('phoneNumber', values.phoneNumber.toString());
      formData.append('clubTypeId', values.clubTypeId);
      formData.append('address', values.address);
      formData.append('gmbProfile', values.gmbProfile);
      formData.append('businessWebsite', values.businessWebsite);
      formData.append('city', values.city);
      formData.append('country', values.country);
      formData.append('zipCode', values.zipCode.toString());
      formData.append('profilePicture', values.profilePicture ?? 'null');
      formData.append('bio', values.bio);
      formData.append('faceBookLink', values.faceBookLink);
      formData.append('tiktokLink', values.tiktokLink);
      formData.append('instagramLink', values.instagramLink);
      formData.append('xLink', values.xLink);
      formData.append('snapChatLink', values.snapChatLink);
      formData.append('roleId', userInfo.roleId);
      formData.append(
        'portfolioPics',
        values.portfolioPics ? JSON.stringify(values.portfolioPics) : 'null'
      );
      setFormSubmitting(true);

      try {
        const cleanedValues = cleanObject({
          ...values,
          roleId: userInfo.roleId,
          portfolioImages: values?.portfolioPics,
        });

        await putData(`/users/${userInfo?.id}`, cleanedValues);
        const updatedUser = await fetchData<UserData>(`/users/${userInfo?.id}`);
        setUserData(updatedUser);
      } catch (error) {
        console.error('Error updating profile:', error);
      } finally {
        setFormSubmitting(false); // Stop loading in both success/failure
      }
    },
  });

  const handleFileChange = async (file: File | null) => {
    if (!file || !file.type.startsWith('image/')) {
      console.warn('Only image files are allowed.');
      return;
    }

    try {
      const response = await uploadFile<{ url: string }>('file/upload', file);
      if (response?.url) {
        formik.setFieldValue('profilePicture', response.url);
        // const updatedUser = await fetchData<UserData>(`/users/${userInfo?.id}`);
        // setUserData(updatedUser);
      }
    } catch (error) {
      console.error('Upload failed:', error);
    }
  };
  const handleDrop = async (files: File[]) => {
    const validFiles = files.slice(0, 6 - portfolioPics.length);
    const uploadedUrls: string[] = [];

    for (const file of validFiles) {
      const tempPreview = URL.createObjectURL(file);
      setUploadingImages(prev => [...prev, tempPreview]);
      if (!file.type.startsWith('image/')) continue;
      try {
        const res = await uploadFile<{ url: string }>('file/upload', file);
        if (res?.url) {
          uploadedUrls.push(res.url);
        }
      } catch (err) {
        console.error('Image upload failed:', err);
      } finally {
        setUploadingImages(prev => prev.filter(url => url !== tempPreview));
      }
    }

    const updated = [...portfolioPics, ...uploadedUrls].slice(0, 6);
    setPortfolioPics(updated);
    formik.setFieldValue('portfolioPics', updated);
  };
  const allImages = [
    ...(userData?.portfolioImages || []),
    ...(formik.values?.portfolioPics || []),
    ...uploadingImages,
  ].filter((item, index, self) => self.indexOf(item) === index);

  const pathRef = useRef<SVGPathElement>(null);
  const [pos, setPos] = useState<{ x: number; y: number }>({ x: 0, y: 0 });

  const percentage: number = userData?.profileCompletion?.percentage ?? 0;
  const rectWidth = 128;
  const rectHeight = 128;
  const rectRx = 30;

  // total perimeter length
  const perimeter =
    2 * (rectWidth - 2 * rectRx + rectHeight - 2 * rectRx) +
    2 * Math.PI * rectRx;

  // stroke progress
  const progressLength = (perimeter * percentage) / 100;
  const displayValue = percentage === 0 ? 2 : percentage;

  useEffect(() => {
    if (pathRef.current) {
      const point = pathRef.current.getPointAtLength(progressLength);
      setPos({ x: point.x, y: point.y });
    }
  }, [progressLength]);
  let gradientStops;
  if (percentage >= 70) {
    gradientStops = (
      <>
        <stop offset="0%" stopColor="#00C853" /> {/* light green */}
        <stop offset="100%" stopColor="#2E7D32" /> {/* dark green */}
      </>
    );
  } else if (percentage >= 40) {
    gradientStops = (
      <>
        <stop offset="0%" stopColor="#FFD600" /> {/* light yellow */}
        <stop offset="100%" stopColor="#F9A825" /> {/* darker yellow */}
      </>
    );
  } else {
    gradientStops = (
      <>
        <stop offset="0%" stopColor="#FF5252" /> {/* light red */}
        <stop offset="100%" stopColor="#B71C1C" /> {/* dark red */}
      </>
    );
  }
  return (
    <>
      {loading ? (
        <Paper
          bg="radial-gradient(#6534BA 43%, #3B1E6D)"
          style={{
            padding: '40px 0',
          }}
          className="min-h-screen"
        >
          <Center style={{ height: '100vh' }}>
            <Loader size="lg" color="#C28CFF" />
          </Center>
        </Paper>
      ) : (
        <Paper
          bg="radial-gradient(#6534BA 43%, #3B1E6D)"
          style={{
            padding: '40px 0',
          }}
          className="min-h-screen overflow-hidden"
        >
          <Container
            fluid
            style={{
              padding: '0px 50px',
            }}
          >
            <form onSubmit={formik.handleSubmit}>
              <Grid>
                <Grid.Col span={{ base: 12, md: 12 }}>
                  <Grid>
                    <Grid.Col span={12}>
                      <Title
                        style={{
                          color: 'white',
                          marginBottom: '20px',
                          textAlign: 'left',
                          fontSize: '25px',
                          fontFamily: 'Product Sans',
                        }}
                      >
                        My Profile
                      </Title>
                    </Grid.Col>
                  </Grid>
                  <Grid className="mb-3">
                    {/* Profile Picture Preview */}
                    <Grid.Col span={{ base: 12, md: 12, lg: 2 }}>
                      <div
                        style={{
                          position: 'relative',
                          width: 140, // little bigger than 120 to leave space for stroke
                          height: 140,
                          display: 'flex',
                          alignItems: 'center',
                          justifyContent: 'center',
                        }}
                        onClick={() => {
                          setprofileInfo({
                            isModalOpen: true,
                          });
                        }}
                      >
                        <svg
                          width={150}
                          height={140}
                          style={{
                            position: 'absolute',
                            top: 0,
                            left: 0,
                            zIndex: 10,
                          }}
                        >
                          <defs>
                            <linearGradient
                              id="progress-gradient"
                              x1="0%"
                              y1="0%"
                              x2="100%"
                              y2="0%"
                            >
                              {gradientStops}
                            </linearGradient>
                          </defs>

                          {/* base path */}
                          <path
                            ref={pathRef}
                            id="progressPath"
                            d={`
          M 70 6
          H ${6 + rectRx}
          Q 6 6, 6 ${6 + rectRx}
          V ${134 - rectRx}
          Q 6 134, ${6 + rectRx} 134
          H ${134 - rectRx}
          Q 134 134, 134 ${134 - rectRx}
          V ${6 + rectRx}
          Q 134 6, ${134 - rectRx} 6
          H 70
        `}
                            fill="none"
                            stroke="#E0E0E0"
                            strokeWidth={6}
                          />

                          {/* progress stroke */}
                          <path
                            d={`
          M 70 6
          H ${6 + rectRx}
          Q 6 6, 6 ${6 + rectRx}
          V ${134 - rectRx}
          Q 6 134, ${6 + rectRx} 134
          H ${134 - rectRx}
          Q 134 134, 134 ${134 - rectRx}
          V ${6 + rectRx}
          Q 134 6, ${134 - rectRx} 6
          H 70
        `}
                            fill="none"
                            stroke="url(#progress-gradient)"
                            strokeWidth={6}
                            strokeDasharray={`${progressLength}, ${perimeter}`}
                            strokeLinecap="round"
                          />

                          {/* circle background + text */}
                          {/* circle background + text */}
                          <g transform={`translate(${pos.x}, ${pos.y})`}>
                            {/* grey background circle */}
                            <circle
                              r="16"
                              fill="#E0E0E0"
                              stroke="#999"
                              strokeWidth="2"
                              transform="translate(-3, 0)" // shift left & up
                            />

                            {/* percentage text inside the circle */}
                            <text
                              fill="#000"
                              fontSize="12"
                              fontWeight="bold"
                              dominantBaseline="middle"
                              textAnchor="middle"
                              dy="2"
                              transform="translate(0, 0)" // move text with circle
                            >
                              {percentage}%
                            </text>
                          </g>
                        </svg>

                        <div className="relative w-[110px] h-[110px] rounded-[20%] bg-gray-300 overflow-hidden group">
                          {formik.values?.profilePicture ? (
                            <img
                              src={formik.values.profilePicture}
                              alt="Profile"
                              style={{
                                width: '100%',
                                height: '100%',
                                objectFit: 'cover',
                              }}
                            />
                          ) : (
                            <IconUser
                              size={40}
                              color="#9E9E9E"
                              style={{
                                marginLeft: '5px',
                                marginTop: '2px',
                                width: '90%',
                                height: '90%',
                                objectFit: 'cover',
                              }}
                            />
                          )}
                          <div className="absolute inset-0 z-10 flex items-center justify-center text-center bg-color-black bg-opacity-50 text-color-white text-sm font-medium rounded-[20%] opacity-0 transition-opacity duration-300 group-hover:opacity-100">
                            Click to View Profile Progress
                          </div>
                        </div>
                      </div>
                    </Grid.Col>

                    {/* Right Side: Heading + Buttons + Text */}

                    <Grid.Col span={{ base: 12, md: 12, lg: 10 }}>
                      <div>
                        <Title
                          style={{
                            color: 'white',
                            marginBottom: '8px',
                            marginTop: '8px',
                            textAlign: 'left',
                            fontSize: '20px',
                            fontFamily: 'Product Sans',
                          }}
                        >
                          Profile Picture
                        </Title>
                      </div>
                      <div className="flex flex-row items-start justify-start">
                        <div>
                          <FileInput
                            accept="image/*"
                            placeholder="Upload New"
                            onChange={file => {
                              setProfilePic(file);
                              handleFileChange(file);
                            }}
                            leftSection={<FaUpload color="white" />}
                            styles={{
                              input: {
                                width: 'auto',
                                backgroundColor: '#D83F87',
                                color: 'white',
                                textAlign: 'center',
                                fontWeight: 'bold',
                                cursor: 'pointer',
                                height: '40px',
                                fontFamily: 'Product Sans',
                                border: 'none',
                              },
                              placeholder: { color: 'white', fontSize: '14px' },
                            }}
                          />
                        </div>
                        <div>
                          <Button
                            onClick={() => {
                              setProfilePic(null);
                              formik.setFieldValue('profilePicture', '');
                            }}
                            style={{
                              width: 'auto',
                              backgroundColor: '#2A114A',
                              color: 'white',
                              fontWeight: 'bold',
                              height: '40px',
                              fontSize: '14px',
                              fontFamily: 'Product Sans',
                            }}
                            className="ml-4"
                          >
                            Remove Picture
                          </Button>
                        </div>
                      </div>
                      <div className="mt-2">
                        <Text
                          style={{
                            color: 'white',
                            marginBottom: '20px',
                            textAlign: 'left',
                            paddingLeft: '0',
                            fontFamily: 'Product Sans',
                          }}
                        >
                          Support: JPG.
                        </Text>
                      </div>
                    </Grid.Col>
                  </Grid>

                  {/* Profile Form */}
                  <Grid>
                    <Grid.Col
                      className="businessProfileleftGrid"
                      span={{ base: 12, md: 7 }}
                    >
                      <form className="Profile-form font-productsans">
                        <Grid gutter="sm">
                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <NumberInput
                              className="py-2"
                              label="Tax ID"
                              placeholder="Enter Tax Id"
                              value={formik.values?.taxId}
                              hideControls
                              disabled
                              radius="6px"
                              styles={inputStyle4}
                              error={
                                formik.touched.taxId && formik.errors.taxId
                              }
                            />
                          </Grid.Col>
                          {formik.values?.businessDocumentType === 'ein' && (
                            <Grid.Col span={{ base: 12, md: 6 }}>
                              <FileInput
                                label="Document"
                                id="fileInput"
                                placeholder={
                                  formik.values?.taxIdDocumentUrl
                                    ? 'Click the down arrow to download the file.'
                                    : ''
                                }
                                styles={inputStyle4}
                                radius={'6px'}
                                className="py-2"
                                accept=".pdf,.doc,.docx,.png,.jpg,.jpeg"
                                clearable
                                disabled
                                rightSection={
                                  formik.values?.taxIdDocumentUrl ? (
                                    <a
                                      href={formik.values?.taxIdDocumentUrl}
                                      download
                                      target="_blank"
                                      rel="noopener noreferrer"
                                      onClick={e => e.stopPropagation()}
                                      style={{
                                        display: 'flex',
                                        alignItems: 'center',
                                      }}
                                    >
                                      <IconDownload size={20} color="white" />
                                    </a>
                                  ) : null
                                }
                              />
                            </Grid.Col>
                          )}

                          {formik.values?.businessDocumentType === 'cr' && (
                            <Grid.Col span={{ base: 12, md: 6 }}>
                              <TextInput
                                className="py-2"
                                label="CR Number"
                                placeholder="Enter CR Number"
                                value={formik.values?.crNumber}
                                disabled
                                radius="6px"
                                styles={inputStyle4}
                                error={
                                  formik.touched.crNumber &&
                                  formik.errors.crNumber
                                }
                              />
                            </Grid.Col>
                          )}
                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <TextInput
                              className="py-2"
                              label="Email"
                              placeholder="Enter your price"
                              name="email"
                              onChange={formik.handleChange}
                              value={formik.values?.email}
                              disabled
                              styles={inputStyle4}
                            />
                          </Grid.Col>
                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <TextInput
                              className="py-2"
                              label="Phone Number"
                              placeholder="Enter your phone number"
                              name="phoneNumber"
                              onChange={formik.handleChange}
                              value={formik.values?.phoneNumber}
                              disabled
                              styles={inputStyle4}
                            />
                          </Grid.Col>
                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <TextInput
                              className="py-2"
                              label="Full Name"
                              placeholder="Enter your full name"
                              name="fullNameOrBusinessName"
                              onChange={formik.handleChange}
                              value={formik.values?.fullNameOrBusinessName}
                              required
                              styles={inputStyle4}
                            />
                          </Grid.Col>

                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <TextInput
                              className="py-2"
                              label="Address"
                              placeholder="Enter your address"
                              name="address"
                              onChange={formik.handleChange}
                              value={formik.values?.address}
                              required
                              styles={inputStyle4}
                            />
                          </Grid.Col>
                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <TextInput
                              className="py-2"
                              label="City"
                              placeholder="Enter your city"
                              name="city"
                              onChange={formik.handleChange}
                              value={formik.values?.city}
                              required
                              styles={inputStyle4}
                            />
                          </Grid.Col>
                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <TextInput
                              className="py-2"
                              label="Country"
                              placeholder="Enter your country"
                              name="country"
                              onChange={formik.handleChange}
                              value={formik.values?.country}
                              required
                              styles={inputStyle4}
                            />
                          </Grid.Col>
                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <TextInput
                              className="py-2"
                              label="Zip Code"
                              placeholder="Enter your zip code"
                              name="zipCode"
                              onChange={formik.handleChange}
                              value={formik.values?.zipCode}
                              required
                              styles={inputStyle4}
                            />
                          </Grid.Col>

                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <Select
                              className="py-2"
                              label="Club Type"
                              placeholder="Select club type"
                              name="clubType"
                              data={organizeDropDownData(
                                referenceData?.clubTypes
                              )}
                              onChange={value =>
                                formik.setFieldValue('clubTypeId', value)
                              }
                              value={formik.values?.clubTypeId}
                              required
                              styles={inputStyle4}
                            />
                          </Grid.Col>

                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <Select
                              className="py-2"
                              label="Your Club Location"
                              placeholder="Choose your location"
                              name="address"
                              data={[
                                'New York',
                                'Los Angeles',
                                'Chicago',
                                'Miami',
                              ]}
                              onChange={value =>
                                formik.setFieldValue('address', value)
                              }
                              value={formik.values?.address}
                              styles={inputStyle4}
                            />
                          </Grid.Col>
                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <TextInput
                              className="py-2"
                              label="GMB Profile"
                              placeholder="Enter your gmb profile"
                              name="gmbProfile"
                              onChange={formik.handleChange}
                              value={formik.values?.gmbProfile}
                              required
                              styles={inputStyle4}
                            />
                          </Grid.Col>
                          <Grid.Col span={{ base: 12, md: 6 }}>
                            <TextInput
                              className="py-2"
                              label="Business Website"
                              placeholder="Enter your business website"
                              name="businessWebsite"
                              onChange={formik.handleChange}
                              value={formik.values?.businessWebsite}
                              required
                              styles={inputStyle4}
                            />
                          </Grid.Col>

                          <Grid.Col span={12}>
                            <Textarea
                              radius="8px"
                              label="BIO"
                              placeholder="Tell us about yourself..."
                              minRows={4}
                              className="py-2 Profile-textArea"
                              name="bio"
                              onChange={formik.handleChange}
                              value={formik.values?.bio}
                              required
                              styles={inputStyle4}
                            />
                          </Grid.Col>

                          <Grid.Col span={12}>
                            <Grid gutter="sm">
                              <Grid.Col span={{ base: 12, md: 3 }}>
                                <TextInput
                                  className="py-2"
                                  label="Facebook"
                                  leftSection={
                                    <Image
                                      src={fb}
                                      alt="fb"
                                      width={20}
                                      height={20}
                                    />
                                  }
                                  placeholder="@username"
                                  name="faceBookLink"
                                  onChange={formik.handleChange}
                                  value={formik.values?.faceBookLink}
                                  styles={inputStyle5}
                                />
                              </Grid.Col>
                              <Grid.Col span={{ base: 12, md: 3 }}>
                                <TextInput
                                  className="py-2"
                                  label="Instagram"
                                  leftSection={
                                    <Image
                                      src={insta}
                                      alt="fb"
                                      width={20}
                                      height={20}
                                    />
                                  }
                                  placeholder="@username"
                                  name="instagramLink"
                                  onChange={formik.handleChange}
                                  value={formik.values?.instagramLink}
                                  styles={inputStyle5}
                                />
                              </Grid.Col>
                              <Grid.Col span={{ base: 12, md: 3 }}>
                                <TextInput
                                  className="py-2"
                                  label="Twitter"
                                  leftSection={
                                    <Image
                                      src={X}
                                      alt="fb"
                                      width={20}
                                      height={20}
                                    />
                                  }
                                  placeholder="@username"
                                  name="xLink"
                                  onChange={formik.handleChange}
                                  value={formik.values?.xLink}
                                  styles={inputStyle5}
                                />
                              </Grid.Col>
                              <Grid.Col span={{ base: 12, md: 3 }}>
                                <TextInput
                                  className="py-2"
                                  label="Snapchat"
                                  leftSection={
                                    <Image
                                      src={snap}
                                      alt="fb"
                                      width={20}
                                      height={20}
                                    />
                                  }
                                  placeholder="@username"
                                  name="snapChatLink"
                                  onChange={formik.handleChange}
                                  value={formik.values?.snapChatLink}
                                  styles={inputStyle5}
                                />
                              </Grid.Col>
                              <Grid.Col span={{ base: 12, md: 3 }}>
                                <TextInput
                                  className="py-2"
                                  label="Tiktok"
                                  leftSection={
                                    <Image
                                      src={tiktok}
                                      alt="tiktok"
                                      width={25}
                                      height={25}
                                    />
                                  }
                                  placeholder="@username"
                                  name="tiktokLink"
                                  onChange={formik.handleChange}
                                  value={formik.values?.tiktokLink}
                                  styles={inputStyle5}
                                />
                              </Grid.Col>
                            </Grid>
                          </Grid.Col>
                        </Grid>
                      </form>
                    </Grid.Col>

                    <Grid.Col
                      mx="auto"
                      span={{ base: 12, md: 4 }}
                      style={{ maxWidth: '500px' }}
                    >
                      <Text
                        style={{
                          color: 'white',
                          marginTop: '-10px',
                          marginBottom: '10px',
                          paddingLeft: '10px',
                          fontFamily: 'Inter',
                          fontWeight: 600,
                          fontSize: '16px',
                          lineHeight: '100%',
                          letterSpacing: '0%',
                          verticalAlign: 'bottom',
                        }}
                      >
                        Upload Portfolio Pictures
                      </Text>

                      <div
                        style={{
                          fontFamily: 'Product Sans',
                        }}
                      >
                        <Dropzone
                          onDrop={handleDrop}
                          maxSize={3 * 1024 ** 2}
                          accept={[
                            MIME_TYPES.png,
                            MIME_TYPES.jpeg,
                            MIME_TYPES.webp,
                          ]}
                          style={{
                            border: '2px dashed #ccc',
                            borderRadius: '10px',
                            backgroundColor: '#6534BA',
                            padding: '20px',
                            cursor: 'pointer',
                            height: '200px',
                            display: 'flex',
                            justifyContent: 'center',
                            alignItems: 'center',
                            flexDirection: 'column',
                            textAlign: 'center',
                          }}
                        >
                          <div
                            style={{
                              display: 'flex',
                              flexDirection: 'column',
                              alignItems: 'center',
                              gap: '10px',
                            }}
                          >
                            <IoMdImages size={36} color="white" />
                            <Text color="white" size="sm">
                              Drop your images here or
                            </Text>
                            <button
                              style={{
                                padding: '6px 12px',
                                backgroundColor: '#6534BA',
                                color: '#ffff',
                                border: 'none',
                                borderRadius: '6px',
                                cursor: 'pointer',
                                fontWeight: 600,
                              }}
                              onClick={e => {
                                e.stopPropagation();
                                document
                                  .querySelector<HTMLInputElement>(
                                    '.mantine-Dropzone-root input'
                                  )
                                  ?.click();
                              }}
                            >
                              Select Images
                            </button>
                          </div>
                        </Dropzone>
                      </div>

                      {/* Display uploaded images */}
                      <div
                        style={{
                          display: 'grid',
                          gridTemplateColumns: 'repeat(3, 1fr)',
                          gap: '10px',
                          marginTop: '20px',
                        }}
                      >
                        {allImages.slice(0, 6).map((img, i) => (
                          <div
                            key={i}
                            style={{
                              height: '100px',
                              backgroundColor: '#E0E0E0',
                              borderRadius: '10px',
                              overflow: 'hidden',
                              position: 'relative',
                              cursor:
                                i === 5 && allImages.length > 6
                                  ? 'pointer'
                                  : 'default',
                            }}
                            onClick={() => {
                              if (i === 5 && allImages.length > 6) {
                                setViewMoreOpen(true);
                              }
                            }}
                          >
                            <img
                              src={img}
                              alt="Portfolio"
                              style={{
                                width: '100%',
                                height: '100%',
                                objectFit: 'cover',
                                opacity: uploadingImages.includes(img)
                                  ? 0.4
                                  : 1,
                              }}
                            />
                            {uploadingImages.includes(img) && (
                              <div
                                style={{
                                  position: 'absolute',
                                  top: '40%',
                                  left: '50%',
                                  transform: 'translate(-50%, -50%)',
                                }}
                              >
                                <Loader size="sm" color="gray" />
                              </div>
                            )}

                            {i === 5 && allImages.length > 6 && (
                              <div
                                style={{
                                  position: 'absolute',
                                  top: 0,
                                  left: 0,
                                  height: '100%',
                                  width: '100%',
                                  backgroundColor: 'rgba(0,0,0,0.5)',
                                  display: 'flex',
                                  justifyContent: 'center',
                                  alignItems: 'center',
                                  color: 'white',
                                  fontWeight: 'bold',
                                  fontSize: '14px',
                                }}
                              >
                                View More
                              </div>
                            )}
                          </div>
                        ))}
                      </div>
                    </Grid.Col>
                  </Grid>

                  {/* End of Profile Form */}

                  {/* Save & Update Button */}
                  <Button
                    type="submit"
                    style={{
                      backgroundColor: '#D83F87',
                      color: 'white',
                      width: '25%',
                      fontFamily: 'Product Sans',
                      // marginTop: '-40px',
                    }}
                    size="md"
                    loading={formSubmitting}
                  >
                    Save & Update
                  </Button>
                </Grid.Col>
              </Grid>
            </form>
          </Container>
          <ViewProfileDetails
            opened={profileInfo?.isModalOpen}
            setOpened={setprofileInfo}
          />
        </Paper>
      )}
    </>
  );
};

export default MyProfileBussiness;
