import {
  Button,
  Checkbox,
  Grid,
  Group,
  Loader,
  Modal,
  NumberInput,
  Paper,
  ScrollArea,
  Text,
  TextInput,
} from '@mantine/core';
import { DatePickerInput } from '@mantine/dates';
import {
  IconArrowLeft,
  IconCalendar,
  IconChevronDown,
} from '@tabler/icons-react';
import { useFormik } from 'formik';
import { useEffect, useMemo, useState } from 'react';
import { BsSend } from 'react-icons/bs';
import { useSelector } from 'react-redux';
import { useNavigate, useSearchParams } from 'react-router-dom';
import * as Yup from 'yup';
import { fetchData, postData, putData } from '../../api/api';
import { Table } from '../../components';

import { filterStyle, inputStyle4 } from '../../theme/common.style';
import { paginationInfoValue } from '../../utils/common/constant.objects';
import { TransactionType } from '../../utils/common/enums';
import {
  formatDate,
  formatNumber,
  handleError,
  handleSetParams,
  handleSuccess,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';

interface PaymentMethod {
  id: string;
  email: string;
  isDefault: boolean;
  createdAt: string;
}

interface SearchFilter {
  searchValue: string;
  type: string;

  sort: string;
  limit: string;
  searchValue1: string;

  searchValue3: string;
}
const initialSearchValues: SearchFilter = {
  searchValue: '',

  type: '',
  sort: '',
  limit: '',
  searchValue3: new Date(
    new Date().setDate(new Date().getDate() + 7)
  ).toISOString(),
  searchValue1: new Date().toISOString(),
};

const PaymentWithdrawal = () => {
  // Redux selectors
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );

  // State management

  const [searchParams, setSearchParams] = useSearchParams();
  const [isLoading, setIsLoading] = useState(false);
  const [paymentInProgress, setPaymentInProgress] = useState(false);
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const [resetTable, setResetTable] = useState(false);
  const [walletBalance, setWalletBalance] = useState<any>();
  const [refetch, setRefetch] = useState(false);
  const [opened, setOpened] = useState(false);
  const [transactionData, setTransactionData] = useState([]);
  const navigate = useNavigate();
  const [paymentMethods, setPaymentMethods] = useState({
    data: [],
    message: '',
  });
  const [selectedPaymentMethod, setSelectedPaymentMethod] = useState<any>(null);
  const [addPaymentModalOpened, setAddPaymentModalOpened] = useState(false);
  const [changePaymentModalOpened, setChangePaymentModalOpened] =
    useState(false);
  const [loadingPaymentMethods, setLoadingPaymentMethods] = useState(true);
  const [loadingWallet, setLoadingWallet] = useState(true);
  useEffect(() => {
    handleFetchDataByFilter();
  }, [currency]);
  const addPaymentFormik = useFormik({
    initialValues: {
      email: '',
      isDefault: false,
      type: TransactionType.WITHDRAWAL,
    },
    validationSchema: Yup.object().shape({
      email: Yup.string().email('Invalid email').required('Email is required'),
    }),
    onSubmit: async values => {
      setLoadingPaymentMethods(true);
      try {
        const response = await postData(`/payment-methods`, {
          ...values,
          userId: userInfo.id,
        });
        if (response) {
          handleSuccess('Payment method added successfully');
          fetchPaymentMethods();
          setAddPaymentModalOpened(false);
          addPaymentFormik.resetForm();
        }
      } catch (error) {
        handleError(error);
      } finally {
        setLoadingPaymentMethods(false);
      }
    },
  });

  // Add these functions after your existing helper functions
  const fetchPaymentMethods = async () => {
    if (!userInfo.id) return;

    setLoadingPaymentMethods(true);
    try {
      const response: any = await fetchData(
        `/payment-methods/user/withdrawal/${userInfo.id}`
      );
      if (response) {
        setPaymentMethods(response);
        console.log('Payment methods fetched successfully:', response);

        // Set the default payment method as selected
        const defaultMethod = response?.data?.find(
          (method: any) =>
            method.isDefault && method.type === TransactionType.WITHDRAWAL
        );
        if (defaultMethod) {
          setSelectedPaymentMethod(defaultMethod);
          formik.setFieldValue('email', defaultMethod.email);
        } else if (response?.data?.length > 0) {
          setSelectedPaymentMethod(response?.data?.[0]);
          formik.setFieldValue('email', response?.data[0]?.email);
        }
      }
    } catch (error) {
      console.error('Error fetching payment methods:', error);
    } finally {
      setLoadingPaymentMethods(false);
    }
  };

  const handleSetDefaultPaymentMethod = async (paymentMethodId: string) => {
    setLoadingPaymentMethods(true);
    try {
      const response = await putData(`/payment-methods/${userInfo.id}`, {
        paymentMethodId,
        isDefault: true,
      });
      if (response) {
        handleSuccess('Default payment method updated');
        fetchPaymentMethods();
      }
    } catch (error) {
      handleError(error);
    } finally {
      setLoadingPaymentMethods(false);
    }
  };

  const handleDeletePaymentMethod = async (paymentMethodId: string) => {
    setLoadingPaymentMethods(true);
    try {
      const response = await postData(
        `/payment-methods/${userInfo.id}/delete`,
        {
          paymentMethodId,
        }
      );
      if (response) {
        handleSuccess('Payment method deleted successfully');
        fetchPaymentMethods();
      }
    } catch (error) {
      handleError(error);
    } finally {
      setLoadingPaymentMethods(false);
    }
  };

  useEffect(() => {
    const baseFilters = [
      {
        field: 'transaction.type',
        operator: 'eq',
        value: TransactionType.WITHDRAWAL,
      },
    ];

    baseFilters.push({
      field: 'transaction.toUserId',
      operator: 'eq',
      value: userInfo?.id,
    });
    const filters = removeEmptyValueFilters(baseFilters);
    const filterObject = JSON.stringify({ filter: filters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });
    const fetchUrl = `transactions?${queryParams.toString()}`;
    fetchData(fetchUrl)
      .then((response: any) => {
        if (response?.data) {
          setTransactionData(response.data);
        }
      })
      .catch(err => handleError(err, 'Error fetching transaction'))
      .finally(() => {
        setIsLoading(false);
      });
  }, [refetch]);

  useEffect(() => {
    fetchPaymentMethods();
  }, [userInfo.id]);

  // Helper functions
  function initialPaginationFromQueryParams() {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  }

  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );

  // Effects
  useEffect(() => {
    if (userInfo.id) {
      setLoadingWallet(true);
      fetchData(`wallet/${userInfo.id}/withdrawal`)
        .then((res: any) => {
          setWalletBalance(res);

          setLoadingWallet(false);
        })
        .catch(err => {
          console.log('Error', err);
          setLoadingWallet(false);
        });
    }
  }, [refetch, currency]);

  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  // Form configuration
  const formik = useFormik({
    enableReinitialize: true,
    initialValues: {
      email: '',
      amount: '',
    },
    validationSchema: Yup.object().shape({
      amount: Yup.number()
        .min(1)
        .max(walletBalance?.balance)
        .required('Required'),
    }),
    onSubmit: values => {
      setPaymentInProgress(true);

      postData(`/wallet/withdraw-funds/${userInfo.id}`, {
        ...values,
        email: selectedPaymentMethod?.email,
      })
        .then((response: any) => {
          console.log(response);

          if (
            response?.paypalTransaction?.batch_header?.batch_status ===
              'PENDING' ||
            response?.paypalTransaction?.batch_header?.batch_status ===
              'COMPLETED'
          ) {
            setOpened(true);
            handleSuccess('Withdrawal Successful.');
          }
        })
        .catch((error: any) => {
          handleError(error);
        })
        .finally(() => {
          setPaymentInProgress(false);
          setRefetch(prev => !prev);
          formik.resetForm();
        });
    },
  });

  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;
    const searchValue1 =
      searchParams.get('searchValue1') ?? initialSearchValues.searchValue1;
    const searchValue3 =
      searchParams.get('searchValue3') ?? initialSearchValues.searchValue3;
    return {
      searchValue,
      type,
      limit,
      sort,
      searchValue1,
      searchValue3,
    };
  };
  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );

  const handleFetchDataByFilter = () => {
    setIsLoading(true);

    const baseFilters = [
      {
        field: 'transaction.type',
        operator: 'eq',
        value: TransactionType.WITHDRAWAL.toString(),
      },
    ];

    if (searchValues?.searchValue) {
      baseFilters.push({
        field: 'transaction.paypalPayoutId',
        operator: 'like',
        value: searchValues.searchValue?.toString(),
      });
    }

    baseFilters.push({
      field: 'transaction.toUserId',
      operator: 'eq',
      value: userInfo?.id,
    });

    if (searchValues.searchValue1) {
      baseFilters.push({
        field: 'transaction.createdAt',
        operator: 'gte',
        value: new Date(
          new Date(searchValues.searchValue1).setDate(
            new Date(searchValues.searchValue1).getDate() - 1
          )
        ).toISOString(),
      });
    }

    if (searchValues.searchValue3) {
      baseFilters.push({
        field: 'transaction.createdAt',
        operator: 'lte',
        value: new Date(
          new Date(searchValues.searchValue3).setDate(
            new Date(searchValues.searchValue3).getDate() + 2
          )
        ).toISOString(),
      });
    }

    const filters = removeEmptyValueFilters(baseFilters);
    const filterObject = JSON.stringify({ filter: filters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });

    if (searchValues.sort) {
      queryParams.append(
        'orderBy',
        JSON.stringify({
          displayOrderId: searchValues.sort,
          createdAt: searchValues.sort,
        })
      );
    }

    const fetchUrl = `transactions?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        if (response?.data) {
          setTransactionData(response.data);
        }
      })
      .catch(err => handleError(err, 'Error fetching transaction'))
      .finally(() => {
        setIsLoading(false);
      });
  };

  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };
  useEffect(() => {
    handleSearchButtonClick();
  }, [
    resetTable,
    searchValues.searchValue,
    searchValues.searchValue1,
    searchValues.searchValue3,
    searchValues.limit,
    searchValues.type,
    searchValues.sort,
  ]);

  // Table columns configuration
  const columns = useMemo(
    () => [
      {
        header: (
          <div className="flex text-start font-normal font-productsans text-[14px]">
            Date
          </div>
        ),
        accessorKey: 'date',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-normal font-productsans text-[14px]">
                {formatDate(rowObject?.createdAt?.substring(0, 10) || '')}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start items-center font-normal font-productsans text-[14px]">
            Method
          </div>
        ),
        accessorKey: 'method',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.paypalPayout?.status?.items
                  ? rowObject?.paypalPayout?.status?.items[0]?.payout_item
                      ?.recipient_wallet
                  : 'PayPal Account'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start items-center font-normal font-productsans text-[14px]">
            Account Name
          </div>
        ),
        accessorKey: 'accountName',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-productsans font-normal text-[14px]">
                {rowObject?.paypalPayout?.payoutEmail ||
                  rowObject?.toUser.email}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start font-normal font-productsans text-[14px]">
            Amount
          </div>
        ),
        accessorKey: 'amount',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-normal font-productsans text-[14px]">
                {`${currency === '1' ? '$' : 'BHD ≈'} ${currency === '1' ? formatNumber(rowObject?.amount) : formatNumber(rowObject?.convertedAmount) || 'N/A'}`}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start font-normal font-productsans text-[14px]">
            Status
          </div>
        ),
        accessorKey: 'status',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.paypalPayout?.status?.batch_header?.batch_status ??
                  'Success'}
              </p>
            </div>
          );
        },
      },
    ],
    [transactionData, currency]
  );
  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById,
    }));
  };

  return (
    <>
      <Paper
        className="flex flex-col justify-between items-center p-4 h-auto bg-[radial-gradient(ellipse_at_bottom,_var(--tw-gradient-stops))] from-[#50347d] via-[#3B1E6D] to-[#3B1E6D]"
        radius={0}
        mih={'70%'}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        {/* Header Section */}
        <Group mb="md">
          <div className="rounded-md border-none ml-3 shadow-sm border bg-color-white bg-opacity-5 w-8 h-8 flex items-center justify-center">
            <IconArrowLeft
              className="align-middle select-none bg-transparent text-center transition-all cursor-pointer"
              color="#ffffff"
              onClick={() => navigate(-1)}
              size={20}
            />
          </div>

          <Text
            style={{
              fontSize: '26px',
              fontWeight: 'bold',
              color: 'white',
              fontFamily: 'Product Sans',
            }}
          >
            Earnings &nbsp;
            <Text
              component="span"
              style={{
                fontSize: '26px',
                fontWeight: 'bold',
                fontFamily: 'Product Sans',
                color: '#D83F87',
              }}
            >
              &gt; Withdrawal
            </Text>
          </Text>
        </Group>

        {/* Main Content Grid */}
        <Grid className="px-4 py-4 w-full" gutter="xl">
          {/* From & To Section */}
          <Grid.Col span={{ base: 12, md: 12, lg: 4 }}>
            <div className="rounded-2xl font-semibold font-productsans text-[18px] bg-[#ffffff] text-color-white p-6 bg-opacity-5 border border-[#623F9D] flex flex-col">
              {/* From Section */}
              <div className="mb-4">
                <div className="text-color-white font-productsans mb-2 font-normal">
                  From:
                </div>
                <div className="flex flex-row items-center gap-2 ">
                  <div className="text-color-white text-[28px] font-productsans font-bold">
                    {loadingWallet ? (
                      <div className=" h-8 w-30 bg-gray-300 rounded-md flex items-center justify-center animate-pulse" />
                    ) : (
                      <p> ${formatNumber(walletBalance?.balance || '0')}</p>
                    )}
                  </div>
                  {currency !== '1' && (
                    <div className="text-color-white text-[18px] font-productsans font-bold">
                      {loadingWallet ? (
                        <div className=" h-8 w-30 bg-gray-300 rounded-md flex items-center justify-center animate-pulse" />
                      ) : (
                        <p>
                          {' '}
                          (BHD ≈{' '}
                          {formatNumber(walletBalance?.convertedBalance || '0')}
                          )
                        </p>
                      )}
                    </div>
                  )}
                </div>
                <div className="mt-2">
                  <button
                    onClick={() => setAddPaymentModalOpened(true)}
                    className="bg-[#D83F87] text-color-white px-8 py-2 font-productsans rounded-full text-[16px] font-semibold hover:bg-[#C73680] transition-colors"
                  >
                    Add Payment Method
                  </button>
                </div>
              </div>

              {/* To Section */}
              <div className="text-color-white font-productsans mt-1 mb-2 font-normal">
                To:
              </div>

              {/* Payment Card */}
              {loadingPaymentMethods ? (
                <div className="bg-[#A46FFE] p-6 rounded-lg shadow-md text-color-white mb-1 min-h-[140px] flex flex-col justify-between items-center align-center">
                  <Loader />
                </div>
              ) : selectedPaymentMethod ? (
                <div className="bg-[#A46FFE] p-6 rounded-lg shadow-md text-color-white mb-1 min-h-[140px] flex flex-col justify-between">
                  <div className="flex-1">
                    <div className="text-lg font-semibold font-productsans">
                      PayPal Account
                    </div>
                    <div className="font-productsans mt-2 text-sm break-all overflow-hidden">
                      {selectedPaymentMethod.email}
                    </div>
                    {selectedPaymentMethod.isDefault && (
                      <div className="text-xs font-productsans mt-1 text-[#6534BA]">
                        Default Payment Method
                      </div>
                    )}
                  </div>
                  <div className="mt-3">
                    <button
                      onClick={() => setChangePaymentModalOpened(true)}
                      disabled={loadingPaymentMethods}
                      className="bg-[#D83F87] text-color-white px-4 py-2 rounded-full text-[14px] font-productsans font-semibold w-full hover:bg-[#C73680] transition-colors disabled:opacity-50"
                    >
                      {loadingPaymentMethods
                        ? 'Loading...'
                        : 'Change Payment Method'}
                    </button>
                  </div>
                </div>
              ) : (
                <div className="bg-[#A46FFE] p-6 rounded-lg shadow-md text-color-white mb-1 min-h-[140px] flex flex-col justify-between">
                  <div className="flex-1">
                    <div className="text-lg font-semibold font-productsans">
                      No Payment Method
                    </div>
                    <div className="text-sm font-productsans mt-2">
                      Please add a payment method to continue
                    </div>
                  </div>
                  <div className="mt-3">
                    <button
                      onClick={() => setAddPaymentModalOpened(true)}
                      className="bg-[#D83F87] text-color-white px-4 py-2 rounded-full text-[14px] font-productsans font-semibold w-full hover:bg-[#C73680] transition-colors"
                    >
                      Add Payment Method
                    </button>
                  </div>
                </div>
              )}
            </div>
          </Grid.Col>

          {/* Withdrawal Payment Section */}
          <Grid.Col span={{ base: 12, md: 12, lg: 8 }}>
            <div className="rounded-2xl font-semibold text-[20px] text-color-white bg-[#ffffff] p-5 bg-opacity-5 border border-[#623F9D]">
              {/* Title */}
              <div className="text-color-white mb-3 font-productsans mt-3">
                Withdrawal Payment
              </div>

              {/* Form */}
              <form onSubmit={formik.handleSubmit}>
                <Grid>
                  {/* Amount Input */}
                  <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
                    <NumberInput
                      className="py-3"
                      label={`Withdraw Amount ($)`}
                      placeholder="Enter Withdraw Amount"
                      value={formik.values?.amount}
                      onChange={value => formik.setFieldValue('amount', value)}
                      error={
                        formik.errors.amount &&
                        (formik.touched.amount || formik.submitCount > 0)
                          ? formik.errors.amount
                          : null
                      }
                      disabled={paymentInProgress || !selectedPaymentMethod}
                      radius="8px"
                      min={0}
                      max={walletBalance?.balance}
                      hideControls
                      styles={inputStyle4}
                    />
                  </Grid.Col>

                  {/* Remaining Amount Display */}
                  <Grid.Col
                    span={{ base: 12, md: 12, lg: 12 }}
                    className="flex justify-end items-end"
                  >
                    <div className="flex items-center justify-end text-end mb-2 align-center font-productsans text-[#ffffff] text-[16px] font-normal">
                      Remaining Amount: ${' '}
                      {formatNumber(
                        Number(walletBalance?.balance) -
                          Number(formik.values?.amount) || '0'
                      )}
                      {currency !== '1' && !isLoading && (
                        <span className="ml-1 text-[13px]">
                          (BHD ≈
                          {(
                            Number(walletBalance?.convertedBalance) -
                            Number(formik.values?.amount) *
                              Number(walletBalance?.currencyRate)
                          ).toFixed(3)}
                          )
                        </span>
                      )}
                    </div>
                  </Grid.Col>

                  {/* Submit Button */}
                  <button
                    disabled={paymentInProgress || !selectedPaymentMethod}
                    type="submit"
                    className="w-full bg-[#D83F87] font-productsans px-5 py-2 rounded-md text-[18px] font-semibold text-color-white cursor-pointer disabled:opacity-50 disabled:cursor-not-allowed"
                  >
                    Withdraw Amount
                  </button>

                  {/* Terms & Conditions */}
                  <div className="font-productsans py-2 mt-1 text-[16px] font-normal ml-2 text-[#ffffff] mb-3.5">
                    T&C Applied
                  </div>
                </Grid>
              </form>
            </div>
          </Grid.Col>
        </Grid>

        {/* Withdrawal History Section */}
        <Grid className="py-5 px-4 w-full">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl font-semibold text-color-white p-4 font-productsans bg-[#ffffff] bg-opacity-5 border border-[#623F9D] bg-transparent"
              style={{ minHeight: '339px' }}
            >
              {/* Heading & Search Filters */}
              <Grid className="mt-4 mr-7">
                {/* Title */}
                <Grid.Col span={{ base: 12, md: 12, lg: 7.75 }}>
                  <div className="text-[22px] text-color-white font-productsans">
                    Withdrawal History
                  </div>
                </Grid.Col>

                {/* Date Label */}
                <Grid.Col span={{ base: 12, md: 2, lg: 1.25 }}>
                  <div className="text-[14px] mt-2.5 font-productsans px-2">
                    Select Date
                  </div>
                </Grid.Col>

                {/* Date Range Picker */}
                <Grid.Col span={{ base: 12, md: 3, lg: 3 }}>
                  <div>
                    <DatePickerInput
                      type="range"
                      placeholder="Pick date range"
                      required
                      value={[
                        searchValues?.searchValue1
                          ? new Date(searchValues.searchValue1)
                          : null,
                        searchValues?.searchValue3
                          ? new Date(searchValues.searchValue3)
                          : null,
                      ]}
                      onChange={(value: any) => {
                        if (value) {
                          setValuesById({
                            searchValue1: value[0]
                              ? value[0].toISOString()
                              : '',
                            searchValue3: value[1]
                              ? value[1].toISOString()
                              : '',
                          });
                        }
                      }}
                      style={filterStyle}
                      radius={'6px'}
                      styles={{
                        input: {
                          backgroundColor: 'transparent',
                          borderColor: '#FFFFFF26',
                          color: '#ffffff',
                          fontFamily: 'Product Sans',
                          fontSize: '12px',
                          height: '48px',
                        },
                        placeholder: { color: 'white', fontWeight: 'normal' },
                      }}
                      rightSection={
                        <IconChevronDown size={16} color="#ffffff" />
                      }
                      leftSection={<IconCalendar size={20} color="#ffffff" />}
                    />
                  </div>
                </Grid.Col>
              </Grid>

              {/* Table Section */}
              <div className="mt-5 overflow-x-auto">
                <ScrollArea
                  h={300}
                  scrollbars="y"
                  className="custom-scroll"
                  type="scroll"
                >
                  <Table
                    data={transactionData}
                    columns={columns}
                    isLoading={isLoading}
                    paginationInfo={paginationInfo}
                    handlePagination={handlePagination}
                    isPaginated={false}
                  />
                </ScrollArea>
              </div>
            </div>
          </Grid.Col>
        </Grid>

        {/* Add Payment Method Modal */}
        <Modal
          opened={addPaymentModalOpened}
          centered
          size="md"
          onClose={() => {
            setAddPaymentModalOpened(false);
            addPaymentFormik.resetForm();
          }}
          withCloseButton={true}
          styles={{
            content: {
              borderRadius: '15px',
              backgroundColor: '#452975',
              padding: 0,
            },
            body: {
              padding: '25px',
              backgroundColor: '#452975',
              color: 'white',
            },
            header: {
              backgroundColor: '#452975',
              borderBottom: 'none',
              padding: '25px 25px 0 25px',
            },
            title: {
              color: 'white',
              fontSize: '24px',
              fontWeight: 'bold',
              fontFamily: 'Product Sans',
            },
            close: {
              color: 'white',
              '&:hover': {
                backgroundColor: 'rgba(255, 255, 255, 0.1)',
              },
            },
          }}
          overlayProps={{
            backgroundOpacity: 0.55,
            blur: 3,
          }}
          transitionProps={{ transition: 'fade-up', duration: 300 }}
          title="Add Payment Method"
        >
          <form onSubmit={addPaymentFormik.handleSubmit}>
            <div className="space-y-4">
              <TextInput
                label="PayPal Email Address"
                placeholder="Enter your PayPal email"
                value={addPaymentFormik.values.email}
                onChange={event =>
                  addPaymentFormik.setFieldValue(
                    'email',
                    event.currentTarget.value
                  )
                }
                error={
                  addPaymentFormik.errors.email &&
                  (addPaymentFormik.touched.email ||
                    addPaymentFormik.submitCount > 0)
                    ? addPaymentFormik.errors.email
                    : null
                }
                styles={{
                  label: {
                    color: 'white',
                    fontSize: '16px',
                    fontFamily: 'Product Sans',
                    marginBottom: '8px',
                  },
                  input: {
                    backgroundColor: 'rgba(255, 255, 255, 0.1)',
                    borderColor: '#623F9D',
                    color: 'white',
                    fontSize: '16px',
                    fontFamily: 'Product Sans',
                    '&::placeholder': {
                      color: 'rgba(255, 255, 255, 0.6)',
                    },
                    '&:focus': {
                      borderColor: '#A46FFE',
                    },
                  },
                  error: {
                    color: '#ff6b6b',
                  },
                }}
                radius="8px"
              />

              <Checkbox
                label="Set as default payment method"
                checked={addPaymentFormik.values.isDefault}
                onChange={event =>
                  addPaymentFormik.setFieldValue(
                    'isDefault',
                    event.currentTarget.checked
                  )
                }
                styles={{
                  label: {
                    color: 'white',
                    fontSize: '16px',
                    fontFamily: 'Product Sans',
                  },
                  input: {
                    backgroundColor: 'transparent',
                    borderColor: '#623F9D',
                    '&:checked': {
                      backgroundColor: '#A46FFE',
                      borderColor: '#A46FFE',
                    },
                  },
                }}
              />

              <div className="flex gap-3 mt-6">
                <Button
                  type="button"
                  variant="outline"
                  fullWidth
                  onClick={() => {
                    setAddPaymentModalOpened(false);
                    addPaymentFormik.resetForm();
                  }}
                  styles={{
                    root: {
                      borderColor: '#623F9D',
                      color: 'white',
                      '&:hover': {
                        backgroundColor: 'rgba(255, 255, 255, 0.1)',
                      },
                    },
                  }}
                >
                  Cancel
                </Button>
                <Button
                  type="submit"
                  fullWidth
                  loading={loadingPaymentMethods}
                  style={{
                    backgroundColor: '#D83F87',
                    color: 'white',
                    fontFamily: 'Product Sans',
                  }}
                >
                  Add Payment Method
                </Button>
              </div>
            </div>
          </form>
        </Modal>

        {/* Change Payment Method Modal */}
        <Modal
          opened={changePaymentModalOpened}
          centered
          size="md"
          onClose={() => setChangePaymentModalOpened(false)}
          withCloseButton={true}
          styles={{
            content: {
              borderRadius: '15px',
              backgroundColor: '#452975',
              padding: 0,
            },
            body: {
              padding: '25px',
              backgroundColor: '#452975',
              color: 'white',
            },
            header: {
              backgroundColor: '#452975',
              borderBottom: 'none',
              padding: '25px 25px 0 25px',
            },
            title: {
              color: 'white',
              fontSize: '24px',
              fontWeight: 'bold',
              fontFamily: 'Product Sans',
            },
            close: {
              color: 'white',
              '&:hover': {
                backgroundColor: 'rgba(255, 255, 255, 0.1)',
              },
            },
          }}
          overlayProps={{
            backgroundOpacity: 0.55,
            blur: 3,
          }}
          transitionProps={{ transition: 'fade-up', duration: 300 }}
          title="Change Payment Method"
        >
          <div className="space-y-4">
            {paymentMethods?.data?.length === 0 ? (
              <div className="text-center py-8">
                <Text className="text-white font-productsans">
                  No payment methods available. Please add a payment method
                  first.
                </Text>
                <Button
                  onClick={() => {
                    setChangePaymentModalOpened(false);
                    setAddPaymentModalOpened(true);
                  }}
                  className="mt-4"
                  style={{
                    backgroundColor: '#D83F87',
                    color: 'white',
                    fontFamily: 'Product Sans',
                  }}
                >
                  Add Payment Method
                </Button>
              </div>
            ) : (
              <>
                <Text className="text-white font-productsans mb-4">
                  Select a payment method:
                </Text>

                <div className="space-y-3 max-h-60 overflow-y-auto">
                  {paymentMethods?.data?.map((method: any) => (
                    <div
                      key={method.id}
                      className={`p-4 rounded-lg border-2 cursor-pointer transition-all ${
                        selectedPaymentMethod?.id === method.id
                          ? 'border-[#A46FFE] bg-[#A46FFE] bg-opacity-20'
                          : 'border-[#623F9D] bg-[#623F9D] bg-opacity-30'
                      }`}
                      onClick={() => setSelectedPaymentMethod(method)}
                    >
                      <div className="flex justify-between items-start">
                        <div className="flex-1">
                          <div className="text-white font-productsans font-semibold">
                            {method.email}
                          </div>
                          {method.isDefault && (
                            <div className="text-[#D83F87] font-productsans text-sm mt-1">
                              Default Method
                            </div>
                          )}
                          <div className="text-gray-300 font-productsans text-sm mt-1">
                            Added:{' '}
                            {new Date(method.createdAt).toLocaleDateString()}
                          </div>
                        </div>

                        <div className="flex gap-2">
                          {!method.isDefault && (
                            <button
                              onClick={e => {
                                e.stopPropagation();
                                handleSetDefaultPaymentMethod(method.id);
                              }}
                              className="text-[#A46FFE] font-productsans text-sm hover:text-[#D83F87] transition-colors"
                              disabled={loadingPaymentMethods}
                            >
                              Set Default
                            </button>
                          )}

                          <button
                            onClick={e => {
                              e.stopPropagation();
                              if (
                                window.confirm(
                                  'Are you sure you want to delete this payment method?'
                                )
                              ) {
                                handleDeletePaymentMethod(method.id);
                              }
                            }}
                            className="text-red-400 font-productsans text-sm hover:text-red-300 transition-colors"
                            disabled={loadingPaymentMethods}
                          >
                            Delete
                          </button>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>

                <div className="flex gap-3 mt-6">
                  <Button
                    variant="outline"
                    fullWidth
                    onClick={() => setChangePaymentModalOpened(false)}
                    styles={{
                      root: {
                        borderColor: '#623F9D',
                        color: 'white',
                        '&:hover': {
                          backgroundColor: 'rgba(255, 255, 255, 0.1)',
                        },
                      },
                    }}
                  >
                    Cancel
                  </Button>
                  <Button
                    fullWidth
                    onClick={() => setChangePaymentModalOpened(false)}
                    style={{
                      backgroundColor: '#D83F87',
                      color: 'white',
                      fontFamily: 'Product Sans',
                    }}
                  >
                    Confirm Selection
                  </Button>
                </div>
              </>
            )}
          </div>
        </Modal>
        {/* Success Modal */}
        <Modal
          opened={opened}
          centered
          size="md"
          onClose={() => setOpened(false)}
          withCloseButton={false}
          styles={{
            content: {
              borderRadius: '15px',
              backgroundColor: '#452975',
              padding: 0,
            },
            body: {
              padding: '25px',
              backgroundColor: '#452975',
              color: 'white',
              textAlign: 'center',
            },
            inner: {
              padding: 0,
              margin: 0,
              maxWidth: '100vw',
            },
          }}
          overlayProps={{
            backgroundOpacity: 0.55,
            blur: 3,
          }}
          transitionProps={{ transition: 'fade-up', duration: 300 }}
          title=""
        >
          <div className="flex flex-col items-center">
            {/* Round Icon Background */}
            <div className="w-16 h-16 bg-[#A46FFE] rounded-full flex items-center justify-center">
              <BsSend className="text-color-white text-2xl" />
            </div>

            {/* Success Message */}
            <h2 className="text-color-white text-2xl font-productsans mt-4">
              Amount Withdrawal Successful!
            </h2>
            <p className="text-color-white text-sm mt-2 px-4">
              Your withdrawal request has been successfully processed. Please
              claim your funds within the next 30 days in your PayPal Account.
            </p>

            {/* Close Button */}
            <div className="mt-6 w-full">
              <Button
                fullWidth
                style={{
                  backgroundColor: '#D83F87',
                  color: 'white',
                  height: 50,
                  borderRadius: '10px',
                  fontSize: '16px',
                  fontWeight: 'bold',
                }}
                onClick={() => setOpened(false)}
              >
                Close
              </Button>
            </div>
          </div>
        </Modal>
        {/* Payment Processing Modal */}
        <Modal
          opened={paymentInProgress}
          centered
          size="md"
          onClose={() => {}} // Prevent closing during loading
          withCloseButton={false}
          closeOnClickOutside={false}
          closeOnEscape={false}
          styles={{
            content: {
              borderRadius: '15px',
              backgroundColor: '#452975',
              padding: 0,
            },
            body: {
              padding: '25px',
              backgroundColor: '#452975',
              color: 'white',
              textAlign: 'center',
            },
            inner: {
              padding: 0,
              margin: 0,
              maxWidth: '100vw',
            },
          }}
          overlayProps={{
            backgroundOpacity: 0.75,
            blur: 3,
          }}
          transitionProps={{ transition: 'fade-up', duration: 300 }}
          title=""
        >
          <div className="flex flex-col items-center">
            {/* Spinning Loader */}
            <div className="w-16 h-16 bg-[#A46FFE] rounded-full flex items-center justify-center animate-pulse">
              <div className="w-8 h-8 border-4 border-white border-t-transparent rounded-full animate-spin"></div>
            </div>

            {/* Processing Message */}
            <h2 className="text-color-white text-2xl font-productsans mt-4">
              Processing Payment...
            </h2>
            <p className="text-color-white text-sm mt-2 px-4">
              Please wait while we process your withdrawal request. This may
              take a few moments.
            </p>

            {/* Processing Steps */}
            <div className="mt-4 text-left w-full">
              <div className="flex items-center text-color-white text-sm mb-2">
                <div className="w-2 h-2 bg-[#A46FFE] rounded-full mr-3"></div>
                Processing withdrawal request...
              </div>
              <div className="flex items-center text-color-white text-sm">
                <div className="w-2 h-2 bg-gray-400 rounded-full mr-3"></div>
                Finalizing transaction...
              </div>
            </div>
          </div>
        </Modal>
      </Paper>
    </>
  );
};

export default PaymentWithdrawal;
