import { Grid, Paper, Select } from '@mantine/core';
import { IconCalendar, IconChevronDown } from '@tabler/icons-react';
import { DatePickerInput } from '@mantine/dates';
import { useEffect, useMemo, useState } from 'react';
import { useSelector } from 'react-redux';
import { useNavigate, useParams, useSearchParams } from 'react-router-dom';
import { fetchData } from '../../api/api';
import { Table } from '../../components';
import { DebouncedTextInput } from '../../components/DebouncedTextInput';
import { getTranslation } from '../../layout/languages';
import { filterStyle, textAreaStyle } from '../../theme/common.style';
import {
  AMOUNT,
  paginationInfoValue,
  SORT_OPTIONS,
} from '../../utils/common/constant.objects';
import { TransactionType } from '../../utils/common/enums';
import {
  formatDate,
  formatNumber,
  handleError,
  handleSetParams,
  organizeDropDownData,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import { initialSearchValues, SearchFilter } from '../reporting/initial.values';
import PaymentWalletAdmin from './payment-wallet-admin';
import PaymentWalletChart from './payment-wallet-chart';
import PaymentWalletBusiness from './payments-wallet-bussiness';
const PaymentWalletEntertainer = () => {
  const navigate = useNavigate();
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const [isLoading, setIsLoading] = useState(false);
  const [transactionData, setTranscationData] = useState([]);
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  const [data, setData] = useState<any>();
  const { id } = useParams();
  useEffect(() => {
    handleFetchDataByFilter();
  }, [currency]);
  useEffect(() => {
    if (!userInfo.id) return;

    setIsLoading(true);

    fetchData(`/transactions/entertainer/${userInfo?.id}`)
      .then((response: any) => {
        setData(response);
      })
      .catch((error: any) => {
        handleError(error, 'Failed to fetch user data');
      })
      .finally(() => {
        setIsLoading(false);
      });
  }, [userInfo?.id, currency]);
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;
    const searchValue1 =
      searchParams.get('searchValue1') ?? initialSearchValues.searchValue1;
    const searchValue3 =
      searchParams.get('searchValue3') ?? initialSearchValues.searchValue3;
    return {
      searchValue,
      type,
      limit,
      sort,
      searchValue1,
      searchValue3,
    };
  };
  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );
  const handleFetchDataByFilter = () => {
    setIsLoading(true);

    const baseFilters = [];

    if (searchValues?.searchValue) {
      baseFilters.push({
        field: 'transaction.displayOrderId',
        operator: 'like',
        value: searchValues.searchValue?.toString(),
      });
    }

    baseFilters.push({
      field: 'transaction.toUserId',
      operator: 'eq',
      value: userInfo?.id,
    });
    baseFilters.push({
      field: 'transaction.type',
      operator: 'eq',
      value: TransactionType.PERFORMANCE_FEE,
    });

    if (searchValues.type) {
      baseFilters.push({
        field: 'event.categoryId',
        operator: 'eq',
        value: searchValues.type,
      });
    }

    if (searchValues.limit) {
      baseFilters.push({
        field: 'transaction.amount',
        operator: 'lte',
        value: searchValues.limit,
      });
    }
    if (searchValues.searchValue1) {
      baseFilters.push({
        field: 'transaction.createdAt',
        operator: 'gte',
        value: new Date(
          new Date(searchValues.searchValue1).setDate(
            new Date(searchValues.searchValue1).getDate() - 1
          )
        ).toISOString(),
      });
    }

    if (searchValues.searchValue3) {
      baseFilters.push({
        field: 'transaction.createdAt',
        operator: 'lte',
        value: new Date(
          new Date(searchValues.searchValue3).setDate(
            new Date(searchValues.searchValue3).getDate() + 2
          )
        ).toISOString(),
      });
    }

    const filters = removeEmptyValueFilters(baseFilters);
    const filterObject = JSON.stringify({ filter: filters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });

    if (searchValues.sort) {
      queryParams.append(
        'orderBy',
        JSON.stringify({
          displayOrderId: searchValues.sort,
          createdAt: searchValues.sort,
        })
      );
    }

    const fetchUrl = `transactions?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        if (response?.data) {
          setTranscationData(response.data);
        }
      })
      .catch(err => handleError(err, 'Error fetching transaction'))
      .finally(() => {
        setIsLoading(false);
      });
  };
  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };
  useEffect(() => {
    handleSearchButtonClick();
  }, [
    searchValues.searchValue,
    searchValues.searchValue1,
    searchValues.searchValue3,
    searchValues.limit,
    searchValues.type,
    searchValues.sort,
  ]);
  const columns = useMemo(
    () => [
      {
        header: (
          <div className="flex text-start font-normal font-productsans text-[14px]">
            ID
          </div>
        ),
        accessorKey: 'id',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-2">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.displayOrderId}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center ml-2 font-normal font-productsans text-[14px]">
            Event Name
          </div>
        ),
        accessorKey: 'event',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex items-center text-start">
              {rowObject?.event?.eventImage ? (
                <img
                  src={rowObject.event.eventImage}
                  alt="Image"
                  className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 object-cover"
                  loading="lazy"
                />
              ) : (
                <div className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 bg-gray-300 flex items-center justify-center text-xs font-semibold uppercase">
                  {rowObject?.event?.title?.[0] || 'E'}
                </div>
              )}
              <p className="text-start ml-2 font-productsans font-normal text-[14px]">
                {rowObject?.event?.title}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center ml-4 font-normal font-productsans text-[14px]">
            Event Type
          </div>
        ),
        accessorKey: 'event',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start ml-2 font-productsans font-normal text-[14px]">
                {rowObject?.event?.category?.name}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-5 font-normal font-productsans text-[14px]">
            Location
          </div>
        ),
        accessorKey: 'id',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const row = info?.row?.original ?? {};
          const city = row.event?.city?.trim();
          const country = row.event?.country?.trim();
          const venue = row.event?.venue?.trim();

          const location =
            [venue, city, country].filter(Boolean).join(', ') || 'N/A';
          return (
            <div className="flex text-start ml-2">
              <p className="text-start font-normal font-productsans text-[14px]">
                {location}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-3 font-normal font-productsans text-[14px]">
            Date
          </div>
        ),
        accessorKey: 'date',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start">
              <p className="text-start  font-productsans font-normal text-[14px]">
                {formatDate(rowObject?.createdAt?.substring(0, 10) || '')}
              </p>
            </div>
          );
        },
      },

      {
        header: (
          <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
            Amount
          </div>
        ),
        accessorKey: 'amount',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-0">
              <p className="text-start font-normal font-productsans text-[14px]">
                {`${currency === '1' ? '$' : 'BHD ≈'} ${currency === '1' ? formatNumber(rowObject?.amount) : formatNumber(rowObject?.convertedAmount) || 'N/A'}`}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-0 font-normal font-productsans text-[14px]">
            Payment Status
          </div>
        ),
        accessorKey: 'id',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-2">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.escrowStatus}
              </p>
            </div>
          );
        },
      },
    ],
    [transactionData, currency]
  );
  if (Number(userInfo?.roleId) === 3) {
    return <PaymentWalletBusiness />;
  }
  if (Number(userInfo?.roleId) === 1) {
    return <PaymentWalletAdmin />;
  }

  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById,
    }));
  };
  return (
    <>
      <Paper
        className="flex flex-col justify-between items-center p-4 min-h-screen bg-[radial-gradient(ellipse_at_bottom,_var(--tw-gradient-stops))] from-[#50347d] via-[#3B1E6D] to-[#3B1E6D]"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 ">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div className="ml-4 text-[33px] font-semibold text-[#ffffff] font-productsans">
              {getTranslation(language, 'payment', 'Payments & Wallet')}
            </div>
          </Grid.Col>
        </Grid>
        <Grid className="px-4 py-4 w-full" gutter="xl">
          <Grid.Col span={{ base: 12, md: 12, lg: 8.5 }}>
            <Grid
              gutter="xs"
              className="rounded-2xl font-semibold font-productsans text-[22px] h-[100%] text-color-white p-5 mt-1 bg-[#ffffff] bg-opacity-5 border border-[#623F9D] flex flex-col md:flex-row justify-between items-start"
            >
              <Grid.Col span={{ base: 12, md: 12, lg: 3.25 }}>
                <div className="mt-3">
                  <div className="rounded-2xl font-semibold text-[22px] text-color-white p-5 bg-opacity-5">
                    <div className="text-color-white mb-2 font-productsans font-normal text-[16px]">
                      {getTranslation(language, 'payment', 'Available Balance')}
                    </div>
                    {/* Balance Amount */}
                    <div className="text-color-white text-[28px] font-bold ">
                      {`${currency === '1' ? '$' : 'BHD ≈'} ${formatNumber(data?.balance)} `}
                    </div>
                    {/* Buttons */}
                    <div className="mt-4">
                      <button
                        className="bg-[#D83F87] text-color-white px-6 py-2 rounded-full text-[14px] font-semibold"
                        onClick={() => navigate('/payment/withdrawal')}
                      >
                        {getTranslation(
                          language,
                          'payment',
                          'Withdraw Balance'
                        )}
                      </button>
                    </div>
                    {/* <div className="mt-3">
                      <button className="bg-[#3B1E6D] text-color-white px-6 py-2 rounded-full text-[14px] font-semibold">
                        {getTranslation(
                          language,
                          'payment',
                          'Add Payment Method'
                        )}
                      </button>
                    </div> */}
                  </div>
                </div>
              </Grid.Col>
              <Grid.Col span={{ base: 12, md: 12, lg: 8.75 }}>
                <div className="">
                  <PaymentWalletChart />
                </div>
              </Grid.Col>
            </Grid>
          </Grid.Col>
          <Grid.Col span={{ base: 12, md: 12, lg: 3.5 }}>
            <div className="space-y-4">
              <div className="rounded-2xl font-semibold text-[22px] text-color-white p-5 bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
                <div className="flex justify-between">
                  <span className="text-color-white font-productsans text-[18px]">
                    {getTranslation(language, 'payment', 'Pending Payments')}
                  </span>
                </div>
                <div className="mt-1 space-y-2">
                  <span className="text-color-white font-productsans text-[28px]">
                    {`${currency === '1' ? '$' : 'BHD ≈'} ${formatNumber(data?.pendingPayments)} `}
                  </span>
                </div>
                <div className="text-color-white font-productsans font-normal underline text-[15px]">
                  <span>{data?.totalEventsPending}</span> <span></span>
                  <span>
                    {getTranslation(
                      language,
                      'payment',
                      'Events Pending Payments'
                    )}
                  </span>
                </div>
              </div>
              <div className="rounded-2xl font-semibold text-[22px] text-color-white p-5 bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
                <div className="flex justify-between">
                  <span className="text-color-white font-productsans text-[18px]">
                    {getTranslation(
                      language,
                      'payment',
                      'Estimated Gig Offers'
                    )}
                  </span>
                </div>
                <div className="mt-1 space-y-2">
                  <span className="text-color-white font-productsans text-[28px]">
                    {`${currency === '1' ? '$' : 'BHD ≈'} ${formatNumber(data?.totalEstimatedGigOffers)} `}
                  </span>
                </div>
                <div>
                  <span className="text-color-white font-productsans underline font-normal text-[15px]">
                    {getTranslation(
                      language,
                      'payment',
                      'Waiting For Approval'
                    )}
                  </span>
                </div>
              </div>
            </div>
          </Grid.Col>
        </Grid>

        <Grid className="py-5 px-4 w-full">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl font-semibold text-color-white p-6 font-['Product Sans'] bg-[#ffffff] bg-opacity-5 border border-[#623F9D] bg-transparent"
              style={{ minHeight: '339px' }}
            >
              <Grid className="mt-4 mr-7 ">
                <Grid.Col span={{ base: 12, md: 12, lg: 2.25 }}>
                  <div className="text-[22px] text-color-white font-productsans">
                    {getTranslation(language, 'payment', 'Transactions')}
                  </div>
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.5 }}>
                  <DebouncedTextInput
                    placeholder={getTranslation(
                      language,
                      'sales',
                      'Search order ID'
                    )}
                    value={searchValues.searchValue}
                    onChange={event =>
                      setValuesById({ searchValue: event.currentTarget.value })
                    }
                    styles={textAreaStyle}
                    classNames={{ input: 'text-input' }}
                    radius={'6px'}
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.5 }}>
                  <Select
                    placeholder={getTranslation(
                      language,
                      'sales',
                      'Event Type'
                    )}
                    data={organizeDropDownData(referenceData?.eventCategories)}
                    value={searchValues.type}
                    clearable
                    rightSection={<IconChevronDown size={16} color="white" />}
                    onChange={value => setValuesById({ type: value || '' })}
                    styles={filterStyle}
                    classNames={{ input: 'custom-input' }}
                    radius="6px"
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.65 }}>
                  <Select
                    placeholder={getTranslation(
                      language,
                      'sales',
                      'Amount Limit'
                    )}
                    data={AMOUNT}
                    value={searchValues.limit}
                    rightSection={<IconChevronDown size={16} color="white" />}
                    onChange={value => value && setValuesById({ limit: value })}
                    styles={filterStyle}
                    classNames={{ input: 'custom-input' }}
                    radius="6px"
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.5 }}>
                  <Select
                    placeholder={getTranslation(language, 'sales', 'Sort By')}
                    data={SORT_OPTIONS}
                    value={searchValues.sort}
                    clearable
                    rightSection={<IconChevronDown size={16} color="white" />}
                    onChange={value => setValuesById({ sort: value || '' })}
                    styles={filterStyle}
                    classNames={{ input: 'custom-input' }}
                    radius="6px"
                  />
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 2, lg: 1.25 }}>
                  <div className="text-[14px] mt-2.5 font-productsans px-2">
                    {getTranslation(language, 'refund', 'Select Date')}
                  </div>
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 3, lg: 2.25 }}>
                  <div>
                    <DatePickerInput
                      type="range"
                      placeholder={getTranslation(
                        language,
                        'refund',
                        'Pick date range'
                      )}
                      required
                      value={[
                        searchValues?.searchValue1
                          ? new Date(searchValues.searchValue1)
                          : null,
                        searchValues?.searchValue3
                          ? new Date(searchValues.searchValue3)
                          : null,
                      ]}
                      onChange={(value: any) => {
                        if (value) {
                          setValuesById({
                            searchValue1: value[0]
                              ? value[0].toISOString()
                              : '',
                            searchValue3: value[1]
                              ? value[1].toISOString()
                              : '',
                          });
                        }
                      }}
                      valueFormat="M/D/YY"
                      style={filterStyle}
                      radius={'6px'}
                      styles={{
                        input: {
                          backgroundColor: 'transparent',
                          borderColor: '#FFFFFF26',
                          color: '#ffffff',
                          fontFamily: 'Product Sans',
                          fontSize: '12px',
                          height: '48px',
                        },
                        placeholder: { color: 'white', fontWeight: 'normal' },
                      }}
                      rightSection={
                        <IconChevronDown size={16} color="#ffffff" />
                      }
                      leftSection={<IconCalendar size={20} color="#ffffff" />}
                    />
                  </div>
                </Grid.Col>
              </Grid>
              <div className="mt-5 ">
                <Table
                  data={transactionData}
                  columns={columns}
                  isLoading={isLoading}
                  paginationInfo={paginationInfo}
                  handlePagination={handlePagination}
                  isPaginated={false}
                  isWithActionButton={false}
                />
              </div>
            </div>
          </Grid.Col>
        </Grid>
      </Paper>
    </>
  );
};

export default PaymentWalletEntertainer;
