import React, { useState, useEffect } from 'react';
import Highcharts from 'highcharts';
import HighchartsReact from 'highcharts-react-official';
import { fetchData } from '../../api/api';
import { useSelector } from 'react-redux';
import { formatNumber } from '../../utils/common/function';

interface AnalyticsResult {
  period: string;
  balance: number;
  periodReceived: number;
  periodWithdrawal: number;
  periodDeposit: number;
  periodSent: number;
  periodNetChange: number;
  transactionCount: number;
}

interface AnalyticsOptions {
  periodType: 'day' | 'week' | 'month' | 'year';
  periodCount?: number;
  startDate?: string;
  endDate?: string;
  userId?: string;
  transactionType?: string[];
  status?: string;
}

interface PaymentWalletProps {
  periodType?: 'day' | 'week' | 'month' | 'year';
  periodCount?: number;
  userId?: string;
  startDate?: string;
  endDate?: string;
  transactionType?: string[];
  status?: string;
  isNeedToUpdate?: boolean;
}

const PaymentWalletChart: React.FC<PaymentWalletProps> = ({
  periodType = 'year',
  periodCount,
  userId,
  startDate = new Date().toISOString(),
  endDate = new Date().toISOString(),
  transactionType,
  status,
  isNeedToUpdate,
}) => {
  const [analyticsData, setAnalyticsData] = useState<AnalyticsResult[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  // Function to fetch analytics data
  const fetchAnalyticsData = async () => {
    try {
      setLoading(true);
      const newParams = new URLSearchParams();
      if (periodType) newParams.set('periodType', periodType);
      else newParams.set('periodType', 'week');
      if (periodCount) newParams.set('periodCount', periodCount.toString());
      if (transactionType)
        newParams.set('transactionType', transactionType.join(','));
      if (status) newParams.set('status', status);
      if (periodType !== 'year') {
        if (startDate) newParams.set('startDate', startDate);
        if (endDate) newParams.set('endDate', endDate);
      }

      window.history.pushState(null, '', `?${newParams.toString()}`);

      // Replace with your actual API endpoint
      const response: any = await fetchData(
        `/transactions/average/${userInfo?.id}${`?${newParams.toString()}`}`
      );

      if (!response) {
        throw new Error('Failed to fetch analytics data');
      }

      const data: AnalyticsResult[] = await response;
      setAnalyticsData(data);
      setError(null);
    } catch (err) {
      setError(err instanceof Error ? err.message : 'An error occurred');
      console.error('Error fetching analytics data:', err);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchAnalyticsData();
  }, [
    periodType,
    periodCount,
    userId,
    transactionType,
    status,
    isNeedToUpdate,
    currency,
  ]);

  // Function to format period labels based on period type
  const formatPeriodLabel = (period: string, type: string): string => {
    switch (type) {
      case 'day':
        return new Date(period).toLocaleDateString('en-US', {
          month: 'short',
          day: 'numeric',
        });
      case 'week':
        return period.replace('Week of ', '');
      case 'month':
        return new Date(period + ' 1, 2024').toLocaleDateString('en-US', {
          month: 'short',
        });
      case 'year':
        return period;
      default:
        return period;
    }
  };

  // Prepare chart data
  const chartCategories = analyticsData
    .map(item => formatPeriodLabel(item.period, periodType))
    .reverse(); // Reverse to show chronological order

  const chartData = analyticsData
    .map(
      item => Math.round(item.balance * 100) / 100 // Round to 2 decimal places
    )
    .reverse();

  // Calculate tick interval based on data range
  const maxValue = Math.max(...chartData);
  const tickInterval = Math.ceil(maxValue / 5);

  const chartOptions: Highcharts.Options = {
    chart: {
      type: 'spline',
      backgroundColor: 'transparent',
      height: '260px',
    },
    title: {
      text: '',
    },
    xAxis: {
      categories: chartCategories,
      gridLineColor: 'transparent',
      lineColor: '#824DDD',
      lineWidth: 2,
      labels: {
        style: {
          color: '#ABADBF',
          fontWeight: 'normal',
        },
      },
    },
    yAxis: {
      title: {
        text: '',
      },
      gridLineColor: '#824DDD',
      labels: {
        style: {
          color: '#ABADBF',
          fontWeight: 'normal',
        },
        formatter: function (): string {
          return (
            `${currency === '1' ? '$' : 'BHD ≈'}` + formatNumber(this.value)
          );
        },
      },
      tickInterval: tickInterval || 20,
    },
    plotOptions: {
      spline: {
        marker: {
          radius: 6,
          lineColor: '#000000',
          lineWidth: 1,
        },
      },
    },
    legend: {
      enabled: false,
    },
    series: [
      {
        type: 'spline',
        color: {
          linearGradient: { x1: 0, y1: 0, x2: 1, y2: 0 },
          stops: [
            [0, '#8C5DF8'],
            [1, '#C508FF'],
          ],
        },
        lineWidth: 5,
        name: 'Wallet Balance',
        marker: {
          symbol: 'circle',
          radius: 6,
          fillColor: '#418F90',
          lineColor: '#ffffff',
          lineWidth: 3,
        },
        data: chartData,
      },
    ],
  };

  if (loading) {
    return (
      <div
        style={{
          height: '260px',
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'center',
          color: '#ABADBF',
        }}
      >
        Loading analytics data...
      </div>
    );
  }

  if (error) {
    return (
      <div
        style={{
          height: '260px',
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'center',
          color: '#ff6b6b',
        }}
      >
        Error: {error}
      </div>
    );
  }

  if (analyticsData.length === 0) {
    return (
      <div
        style={{
          height: '260px',
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'center',
          color: '#ABADBF',
        }}
      >
        No data available for the selected period
      </div>
    );
  }

  return (
    <HighchartsReact
      id={JSON.stringify(chartOptions)}
      highcharts={Highcharts}
      options={chartOptions}
    />
  );
};

export default PaymentWalletChart;
