import { Grid, Paper, ScrollArea } from '@mantine/core';
import { DatePickerInput } from '@mantine/dates';
import { IconCalendar, IconChevronDown } from '@tabler/icons-react';
import { useEffect, useState } from 'react';
import { BsDownload } from 'react-icons/bs';
import { useSelector } from 'react-redux';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { fetchData } from '../../api/api';
import { Button, Table } from '../../components';
import { getTranslation } from '../../layout/languages';
import { filterStyle } from '../../theme/common.style';
import { paginationInfoValue } from '../../utils/common/constant.objects';
import { formatNumber, handleError } from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import { initialSearchValues, SearchFilter } from '../reporting/initial.values';
import PaymentWalletChart from './payment-wallet-chart';
const PaymentWalletAdmin = () => {
  const navigate = useNavigate();

  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const [loadingWallet, setLoadingWallet] = useState(true);
  const [isLoading, setIsLoading] = useState(false);
  const [walletBalance, setWalletBalance] = useState<any>();
  const [commission, setCommission] = useState([]);

  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };

  const getCommission = async () => {
    setIsLoading(true);
    const startDate = searchValues?.searchValue1;
    const endDate = searchValues?.searchValue3;

    // Construct query parameters for date range
    const queryParams = new URLSearchParams();
    if (startDate) queryParams.append('startDate', startDate);
    if (endDate) queryParams.append('endDate', endDate);
    try {
      const res: any = await fetchData(
        `/transactions/commission?${queryParams.toString()}`
      );
      setCommission(res); // Assuming res is array of event objects
    } catch (error) {
      handleError(error, 'Failed to fetch commission');
    } finally {
      setIsLoading(false);
    }
  };
  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  const [searchParams, setSearchParams] = useSearchParams();
  const [value, setValue] = useState<[Date | null, Date | null]>([null, null]);
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  const fetchWallet = () => {
    setLoadingWallet(true);
    fetchData(`wallet/${userInfo.id}`)
      .then((res: any) => {
        setWalletBalance(res);
        setLoadingWallet(false);
      })
      .catch(err => {
        console.log('Error', err);
        setLoadingWallet(false);
      });
  };

  useEffect(() => {
    fetchWallet();
  }, [currency]);

  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;
    const searchValue1 =
      searchParams.get('searchValue1') ?? initialSearchValues.searchValue1;
    const searchValue3 =
      searchParams.get('searchValue3') ?? initialSearchValues.searchValue3;
    return {
      searchValue,
      type,
      limit,
      sort,
      searchValue1,
      searchValue3,
    };
  };

  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );
  useEffect(() => {
    if (searchValues?.searchValue1 && searchValues?.searchValue3) {
      getCommission();
    }
  }, [searchValues?.searchValue1, searchValues?.searchValue3, currency]);

  const columns = [
    {
      header: (
        <div className="flex text-start ml-2 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'payment', 'ID')}
        </div>
      ),
      accessorKey: 'displayOrderId',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start ml-3">
            <p className="text-start font-normal font-productsans text-[14px]">
              {rowObject?.displayOrderId}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-center items-center ml-4 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'payment', 'Client Name')}
        </div>
      ),
      accessorKey: 'title',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start">
            <p className="text-start ml-1 font-productsans font-normal text-[14px]">
              {rowObject?.fromUser?.fullNameOrBusinessName
                ? rowObject?.fromUser?.fullNameOrBusinessName
                : 'N/A'}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-center items-center ml-3 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'payment', 'Events')}
        </div>
      ),
      accessorKey: 'title',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start">
            <div>
              {/* <img
                src={rowObject?.event?.image}
                alt="image"
                className="rounded-md w-9 h-6"
                loading="lazy"
              /> */}
              {rowObject?.event?.image ? (
                <img
                  src={rowObject.event.image}
                  alt="Image"
                  className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4"
                  loading="lazy"
                />
              ) : (
                <div className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 bg-gray-300 flex items-center justify-center text-xs font-semibold uppercase">
                  {rowObject?.event?.title?.[0] || 'E'}
                </div>
              )}
            </div>
            <p className="text-start ml-2 font-productsans font-normal text-[14px]">
              {rowObject?.event?.title ? rowObject?.event?.title : 'N/A'}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-start ml-4 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'payment', 'Commission Type')}
        </div>
      ),
      accessorKey: 'type',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start ml-2">
            <p className="text-start font-normal font-productsans text-[14px]">
              {rowObject?.type}
            </p>
          </div>
        );
      },
    },

    {
      header: (
        <div className="flex text-start ml-5 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'payment', 'Date')}
        </div>
      ),
      accessorKey: 'createdAt',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        const date = new Date(info?.row?.original?.createdAt);

        return (
          <div className="flex text-start ml-2">
            <p className="text-start font-normal font-productsans text-[14px]">
              {date.toLocaleDateString()}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-start  font-normal font-productsans text-[14px]">
          {getTranslation(language, 'payment', 'Commission Earned')}
        </div>
      ),
      accessorKey: 'amount',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start ml-2">
            <p className="text-start font-normal font-productsans text-[14px]">
              {`${currency === '1' ? '$' : 'BHD ≈'} ${currency === '1' ? formatNumber(rowObject?.amount) : formatNumber(rowObject?.convertedAmount) || 'N/A'}`}
            </p>
          </div>
        );
      },
    },
  ];
  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById, // Merge the new values with the existing state
    }));
  };

  const exportToCSV = (data: any[], filename: string) => {
    if (!data || data.length === 0) {
      alert('No data available to export.');
      return;
    }

    const header = Object.keys(data[0]).join(',');
    const csvRows = data.map(row =>
      Object.values(row)
        .map(value => `"${String(value).replace(/"/g, '""')}"`)
        .join(',')
    );

    const csvContent = [header, ...csvRows].join('\n');
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });

    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.setAttribute('download', `${filename}.csv`);
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  return (
    <>
      <Paper
        className="flex flex-col justify-between items-center min-h-screen p-4 h-auto bg-[radial-gradient(ellipse_at_bottom,_var(--tw-gradient-stops))] from-[#50347d] via-[#3B1E6D] to-[#3B1E6D]"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 ">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div className="ml-4 text-[33px] font-semibold text-[#ffffff] font-productsans">
              {getTranslation(language, 'payment', 'Payments & Wallet')}
            </div>
          </Grid.Col>
        </Grid>
        <Grid className="px-6 py-4 w-full" gutter="xs">
          {/* Available Balance and Ticket Report Section */}

          <Grid className="rounded-2xl font-semibold text-[22px] text-color-white p-5 mt-1 bg-[#ffffff] bg-opacity-5 border border-[#623F9D] justify-between items-start w-full">
            {/* Left Section - Available Balance */}

            <Grid.Col span={{ base: 12, md: 4, lg: 4 }}>
              <div className="mt-3">
                <div className="rounded-2xl font-semibold text-[22px] text-color-white p-6 bg-opacity-5">
                  <div className="text-color-white mb-2 font-productsans font-normal text-[18px]">
                    {getTranslation(language, 'payment', 'Available Balance')}
                  </div>
                  {/* Balance Amount */}
                  <div className="text-color-white text-[34px] font-bold ">
                    {loadingWallet ? (
                      <div className=" h-8 w-12 bg-gray-300 rounded-md flex items-center justify-center animate-pulse" />
                    ) : (
                      `${currency === '1' ? '$' : 'BHD ≈'}
                      ${formatNumber(walletBalance?.balance || '0')}`
                    )}
                  </div>
                  {/* Buttons */}
                  <div className="mt-4">
                    <button
                      className="bg-[#D83F87] text-color-white px-6 py-2 rounded-full text-[14px] font-semibold"
                      onClick={() => navigate('/payment/withdrawal')}
                    >
                      {getTranslation(language, 'payment', 'Withdraw Balance')}
                    </button>
                  </div>
                  {/* <div className="mt-3">
                    <button className="bg-[#3B1E6D] text-color-white px-6 py-2 rounded-full text-[14px] font-semibold">
                      {getTranslation(
                        language,
                        'payment',
                        'Add Payment Method'
                      )}
                    </button>
                  </div> */}
                </div>
              </div>
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 8, lg: 8 }}>
              <div className="">
                <PaymentWalletChart />
              </div>
            </Grid.Col>
          </Grid>
        </Grid>

        <Grid className="py-5 px-4 w-full">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl font-semibold text-color-white p-4 font-['Product Sans'] bg-[#ffffff] bg-opacity-5 border border-[#623F9D] bg-transparent"
              style={{ minHeight: '339px' }}
            >
              {/* Heading & Search Filters in One Line */}
              <Grid className="mt-3" gutter={'sm'}>
                <Grid.Col span={{ base: 12, md: 2.5, lg: 3.75 }}>
                  {/* Title */}
                  <div className="text-[22px] text-color-white font-productsans">
                    {getTranslation(language, 'payment', 'Commission Earned')}
                  </div>
                </Grid.Col>
                {/* Search Filters - Responsive */}

                <Grid.Col
                  span={{ base: 12, md: 3.75, lg: 3 }}
                  className="ml-7 mt-1"
                >
                  {/* Export CSV/PDF Button */}
                  <Button
                    style={{
                      backgroundColor: '#27B4B6',
                      color: 'white',
                      display: 'flex',
                      alignItems: 'center',

                      height: '40px',
                      width: 'auto',
                    }}
                    className="py-2 rounded-md text-[24px] font-productsans"
                  >
                    {getTranslation(language, 'payment', 'Export CSV/PDF')}
                    <BsDownload size={16} className="ml-3 font-bold" />
                  </Button>
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 2, lg: 1.25 }}>
                  <div className="text-[14px] mt-2.5 font-productsans">
                    {getTranslation(language, 'dashboard', 'Select Date')}
                  </div>
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 3.75, lg: 3.75 }}>
                  <div>
                    <DatePickerInput
                      type="range"
                      placeholder="Pick date range"
                      required
                      value={[
                        searchValues?.searchValue1
                          ? new Date(searchValues.searchValue1)
                          : null,
                        searchValues?.searchValue3
                          ? new Date(searchValues.searchValue3)
                          : null,
                      ]}
                      onChange={(value: any) => {
                        if (value) {
                          setValuesById({
                            searchValue1: value[0]
                              ? value[0].toISOString()
                              : '',
                            searchValue3: value[1]
                              ? value[1].toISOString()
                              : '',
                          });
                        }
                      }}
                      valueFormat="M/D/YY"
                      style={filterStyle}
                      radius={'6px'}
                      styles={{
                        input: {
                          backgroundColor: 'transparent',
                          borderColor: '#FFFFFF26',
                          color: '#ffffff',
                          fontFamily: 'Product Sans',
                          fontSize: '12px',
                          height: '48px',
                        },
                        placeholder: { color: 'white', fontWeight: 'normal' },
                      }}
                      rightSection={
                        <IconChevronDown size={16} color="#ffffff" />
                      }
                      leftSection={<IconCalendar size={20} color="#ffffff" />}
                    />
                  </div>
                </Grid.Col>
              </Grid>

              {/* Table Section */}
              <div className="mt-5 overflow-x-auto">
                <ScrollArea
                  h={300}
                  scrollbars="y"
                  className="custom-scroll"
                  type="scroll"
                >
                  <Table
                    data={commission}
                    columns={columns}
                    isLoading={isLoading}
                    paginationInfo={paginationInfo}
                    handlePagination={handlePagination}
                    isPaginated={false}
                    isWithActionButton={false}
                  />
                </ScrollArea>
              </div>
            </div>
          </Grid.Col>
        </Grid>
      </Paper>
    </>
  );
};

export default PaymentWalletAdmin;
