import { Button, Group, Modal, Radio, TextInput } from '@mantine/core';
import { PayPalButtons } from '@paypal/react-paypal-js';
import { useEffect, useState } from 'react';
import { useSelector } from 'react-redux';
import { useNavigate } from 'react-router-dom';
import { fetchData, postData } from '../../api/api';
import { inputStyle2 } from '../../theme/common.style';
import { TransactionType } from '../../utils/common/enums';
import { handleError, handleSuccess } from '../../utils/common/function';
import SuccessModal from '../events/success';
import PayPalCheckout from '../payment-paypal/PayPalCheckout';

interface DepositProps {
  opened: boolean;
  setOpened: (opened: boolean) => void;
}

const DepositModal = ({ opened, setOpened }: DepositProps) => {
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const [accountInfo, setAccountInfo] = useState(false);
  const [selectedPayment, setSelectedPayment] = useState('paypal');
  const [renderPayPal, setRenderPayPal] = useState(false);
  const [openCardModal, setOpenCardModal] = useState(false);
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  // Form state
  const [depositAmount, setDepositAmount] = useState('');
  const [isDepositAmountTouched, setIsDepositAmountTouched] = useState(false);
  const [serviceSetting, setServiceSetting] = useState<any>();

  useEffect(() => {
    fetchData('system-settings')
      .then((res: any) => setServiceSetting(res.data))
      .catch(err => console.log(err));
  }, []);

  // Calculate total amount (same as deposit amount since no fees)
  const totalAmount = parseFloat(depositAmount) || 0;

  useEffect(() => {
    if (selectedPayment === 'paypal') {
      const timer = setTimeout(() => setRenderPayPal(true), 300);
      return () => clearTimeout(timer);
    } else {
      setRenderPayPal(false);
    }
  }, [selectedPayment]);

  const handleDepositSubmit = () => {
    if (!depositAmount || parseFloat(depositAmount) <= 0) {
      handleError('Please enter a valid deposit amount');
      return;
    }

    if (selectedPayment === 'card') {
      setOpenCardModal(true);
    }
  };

  const transaction = {
    amount: totalAmount,
    currency: currency?.code || 'USD',
    description: 'Wallet Deposit',
  };

  const cardTransaction = {
    amount: totalAmount,
    currency: currency?.code || 'USD',
    description: 'Wallet Deposit',
  };

  return (
    <Modal
      opened={opened}
      centered
      size="37%"
      onClose={() => setOpened(false)}
      withCloseButton={false}
      styles={{
        content: {
          borderRadius: '15px',
          backgroundColor: '#452975',
          padding: 0,
        },
        body: {
          padding: 0,
        },
        inner: {
          padding: 0,
          margin: 0,
          maxWidth: '100vw',
        },
      }}
      overlayProps={{
        backgroundOpacity: 0.55,
        blur: 3,
      }}
      transitionProps={{ transition: 'fade-up', duration: 300 }}
      title=""
    >
      <div className="relative w-full flex flex-row justify-between">
        <div className="mt-4 text-[#ffffff] font-productsans ml-4 text-[18px]">
          Deposit to Wallet
        </div>
        <Modal.CloseButton
          onClick={() => setOpened(false)}
          style={{
            position: 'absolute',
            top: 10,
            right: 10,
            border: 'none',
            outline: 'none',
            boxShadow: 'none',
            color: '#ffffff',
            backgroundColor: 'transparent',
            transition: 'background-color 0.2s ease-in-out',
            width: 33,
            height: 33,
          }}
        />
      </div>
      <div className="mt-4">
        <TextInput
          className="custom-input px-4 py-4"
          label={`Deposit Amount ($)
          )`}
          placeholder="Enter Amount"
          type="number"
          min="0"
          step="0.01"
          value={depositAmount}
          onBlurCapture={() => setIsDepositAmountTouched(true)}
          onChange={e => setDepositAmount(e.target.value)}
          radius="8px"
          styles={inputStyle2}
          error={
            (isDepositAmountTouched && !depositAmount) ||
            parseFloat(depositAmount) <= 0
              ? 'Enter valid amount'
              : ''
          }
        />

        <div className="flex flex-row justify-between">
          <div className="px-4 py-2 text-[14px] text-[#ffffff]">
            Payment Method:
          </div>
          <Radio.Group
            className="text-[#ffffff] mr-3"
            value={selectedPayment}
            onChange={setSelectedPayment}
          >
            <Group mt="xs">
              <Radio
                value="paypal"
                label="Paypal"
                styles={() => ({
                  radio: {
                    backgroundColor: 'transparent',
                    borderColor: '#D83F87',
                    '&:checked': {
                      backgroundColor: '#D83F87',
                      borderColor: '#D83F87',
                    },
                  },
                  icon: {
                    color: '#ffffff',
                  },
                })}
              />

              <Radio
                value="card"
                label="Card"
                styles={() => ({
                  radio: {
                    backgroundColor: 'transparent',
                    borderColor: '#D83F87',
                    '&:checked': {
                      backgroundColor: '#D83F87',
                      borderColor: '#D83F87',
                    },
                  },
                  icon: {
                    color: '#ffffff',
                  },
                })}
              />
            </Group>
          </Radio.Group>

          <Group>
            <div className="flex flex-row items-center gap-0 -mt-1">
              {selectedPayment === 'paypal' && (
                <svg
                  width="50"
                  height="50"
                  viewBox="0 0 100 50"
                  xmlns="http://www.w3.org/2000/svg"
                >
                  <rect
                    x="10"
                    y="0"
                    width="80"
                    height="50"
                    rx="8"
                    fill="#002C8A"
                  />
                  <text
                    x="20"
                    y="29"
                    fontFamily="Arial"
                    fontSize="18"
                    fill="white"
                    fontWeight="bold"
                  >
                    PayPal
                  </text>
                </svg>
              )}

              <svg
                width="50"
                height="50"
                viewBox="0 0 100 50"
                xmlns="http://www.w3.org/2000/svg"
              >
                <rect
                  x="10"
                  y="0"
                  width="80"
                  height="50"
                  rx="8"
                  fill="#0057A0"
                />
                <text
                  x="30"
                  y="30"
                  fontFamily="Arial"
                  fontSize="18"
                  fill="white"
                  fontWeight="bold"
                >
                  VISA
                </text>
              </svg>
              <svg
                width="50"
                height="50"
                viewBox="0 0 100 50"
                xmlns="http://www.w3.org/2000/svg"
              >
                <rect
                  x="10"
                  y="0"
                  width="80"
                  height="50"
                  rx="8"
                  fill="#314A8E"
                />
                <circle cx="35" cy="25" r="15" fill="#EA001B" />
                <circle cx="55" cy="25" r="15" fill="#FFA200" />
                <rect x="45" y="12" width="10" height="26" fill="#FF6200" />
              </svg>
              <svg
                width="50"
                height="50"
                viewBox="0 0 100 50"
                xmlns="http://www.w3.org/2000/svg"
              >
                <rect
                  x="10"
                  y="0"
                  width="80"
                  height="50"
                  rx="8"
                  fill="#0074C1"
                />
                <text
                  x="20"
                  y="30"
                  fontFamily="Arial"
                  fontSize="18"
                  fill="white"
                  fontWeight="bold"
                >
                  AM EX
                </text>
              </svg>
            </div>
          </Group>
        </div>
        <div className="border-t border-[#ffffff] border-opacity-15 my-1 mx-4 border-1.5"></div>
        <div className="px-5 py-2 text-[14px] text-[#ffffff] font-productsans flex flex-row justify-between">
          <p> Deposit Amount</p>
          <p>${depositAmount || '0'}</p>
        </div>

        <div className="border-t border-[#ffffff] border-opacity-15 my-1 mx-4 border-1.5"></div>
        <div className="px-5 py-2 text-[16px] text-[#ffffff] font-productsans flex flex-row justify-between">
          <p> Total</p>
          <p>${totalAmount.toFixed(2)}</p>
        </div>

        {renderPayPal && selectedPayment === 'paypal' && (
          <div className="px-2 mx-2 mb-3">
            <PayPalButtons
              style={{
                layout: 'vertical',
                color: 'blue',
                label: 'pay',
                height: 35,
                tagline: false,
              }}
              fundingSource="paypal"
              forceReRender={[transaction.amount, transaction.currency]}
              createOrder={async () => {
                try {
                  if (totalAmount <= 0) {
                    throw new Error('Invalid deposit amount');
                  }
                  const data: any = await postData(
                    `wallet/add-funds/${userInfo?.id}`,
                    {
                      paypalOrder: {
                        amount: transaction.amount,
                        currency: transaction.currency,
                        type: TransactionType.DEPOSIT,
                        returnUrl: `${process.env.REACT_APP_HOST_URL}/payment`,
                        cancelUrl: `${process.env.REACT_APP_HOST_URL}/payment`,
                      },
                    }
                  );
                  return data?.paypalTransaction?.paypalOrderId;
                } catch (error) {
                  handleError(error);
                  throw error;
                }
              }}
              onApprove={async (data: { orderID: string }) => {
                try {
                  const details = await postData<any>('paypal/orders/capture', {
                    orderId: data.orderID,
                  });
                  if (details.status === 'COMPLETED') {
                    handleSuccess(
                      `Deposit of ${currency?.symbol || '$'}${totalAmount.toFixed(2)} was successful`
                    );
                    setOpened(false);
                  }
                } catch (error) {
                  console.error('Capture Order Failed:', error);
                  handleError(
                    'An error occurred while capturing the payment.' + error
                  );
                }
              }}
              onError={(err: any) => {
                console.error('PayPal Checkout Error:', err);
                handleError(
                  'An error occurred during the payment process.' + err
                );
              }}
            />
          </div>
        )}
        {selectedPayment === 'card' && (
          <div className="px-2 mr-2 mb-3">
            <Button
              autoContrast
              m={5}
              size="sm"
              disabled={!depositAmount || parseFloat(depositAmount) <= 0}
              styles={{
                root: {
                  background: '#D83F87',
                  '&:disabled': {
                    backgroundColor: '#D83F8760',
                  },
                },
              }}
              style={{
                height: 40,
                width: '100%',
              }}
              className="px-2"
              onClick={handleDepositSubmit}
            >
              <div className="font-productsans font-normal text-[14px] text-[#ffffff]">
                Go To Card Details
              </div>
            </Button>
          </div>
        )}

        <PayPalCheckout
          transaction={{
            amount: Math.round(totalAmount),
            currency: currency?.code || 'USD',
            description: 'Wallet Deposit',
          }}
          opened={openCardModal}
          onClose={() => setOpenCardModal(false)}
          createOrder={async () => {
            try {
              const data: any = await postData(
                `wallet/add-funds/${userInfo?.id}`,
                {
                  paypalOrder: {
                    amount: cardTransaction.amount,
                    currency: cardTransaction.currency,
                    type: TransactionType.DEPOSIT,
                    returnUrl: `${process.env.REACT_APP_HOST_URL}/payment`,
                    cancelUrl: `${process.env.REACT_APP_HOST_URL}/payment`,
                  },
                }
              );
              return data?.paypalTransaction?.paypalOrderId;
            } catch (error: any) {
              console.log('Error', error);
              handleError(error?.response?.data?.message?.message);
              // navigate('/payment/deposit');
            }
          }}
          onApprove={async (data: { orderID: string }) => {
            try {
              const details = await postData<any>('paypal/orders/capture', {
                orderId: data.orderID,
              });

              if (details.status === 'COMPLETED') {
                handleSuccess(
                  `Deposit of ${currency?.symbol || '$'}${totalAmount.toFixed(2)} was successful`
                );
                setOpened(false);
              } else {
                // Handle unsuccessful completion
                handleError('The payment was not completed. Please try again.');
              }
            } catch (error: any) {
              console.error('Capture failed:', error);

              // You can customize this check based on your backend's error format
              if (
                error?.response?.data?.details?.some(
                  (d: any) =>
                    d.field === '/payment_source/card/number' &&
                    d.issue === 'VALIDATION_ERROR'
                )
              ) {
                handleError(
                  'Invalid card number. Please check your card details.'
                );
              } else if (error?.response?.data?.message) {
                handleError(error.response.data.message); // use server message if available
              } else {
                handleError(
                  'Something went wrong while processing your payment.'
                );
              }
            }
          }}
        />
      </div>
      <SuccessModal
        opened={accountInfo}
        setOpened={setAccountInfo}
        amount={totalAmount}
        performerName="Your Wallet"
        currencySymbol={currency?.symbol || '$'}
      />
    </Modal>
  );
};

export default DepositModal;
