import { Modal } from '@mantine/core';
import {
  PayPalCardFieldsForm,
  PayPalCardFieldsProvider,
  usePayPalCardFields,
} from '@paypal/react-paypal-js';
import React, { useState } from 'react';
import {
  extractJsonFromErrorString,
  handleError,
} from '../../utils/common/function';

interface PayPalCheckoutProps {
  transaction: {
    amount: number;
    currency: string;
    description: string;
  };
  opened: boolean;
  onClose: () => void;
  onPaymentSuccess?: () => void;
  onPaymentError?: (error: any) => void;
  createOrder?: any;
  onApprove?: any;
}

const PayPalCheckout: React.FC<PayPalCheckoutProps> = ({
  transaction,
  opened,
  onClose,
  onPaymentSuccess,
  onPaymentError,
  createOrder,
  onApprove,
}) => {
  const [isPaying, setIsPaying] = useState(false);
  const SubmitPayment = ({ isPaying, setIsPaying }: any) => {
    const { cardFieldsForm } = usePayPalCardFields();

    const handleClick = async () => {
      if (!cardFieldsForm) {
        const childErrorMessage =
          'Unable to find any child components in the <PayPalCardFieldsProvider />';
        throw new Error(childErrorMessage);
      }
      const formState = await cardFieldsForm.getState();
      if (!formState.isFormValid) {
        return alert('The payment form is invalid');
      }
      setIsPaying(true);
      cardFieldsForm
        .submit()
        .catch(error => console.log(error))
        .finally(() => {
          setIsPaying(false);
        });
    };

    return (
      <div className="flex justify-center mt-6">
        <button
          className={`
          w-full 
          max-w-sm
          py-3 px-6 
          rounded-lg 
          font-medium 
          text-[#fff] 
          transition-all 
          duration-200 
          ease-in-out
          shadow-md
          focus:outline-none 
          focus:ring-2 
          focus:ring-offset-2 
          disabled:opacity-70 
          disabled:cursor-not-allowed
          bg-[#452975]
        `}
          onClick={handleClick}
          disabled={isPaying}
        >
          {isPaying ? (
            <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin mx-auto" />
          ) : (
            `Pay ${transaction.amount.toFixed(2)} ${transaction.currency}`
          )}
        </button>
      </div>
    );
  };

  return (
    <>
      <Modal
        opened={opened}
        onClose={onClose}
        centered
        size="md"
        withCloseButton={false}
        styles={{
          content: {
            borderRadius: '15px',
            padding: 0,
          },
          body: {
            padding: 0,
          },
          inner: {
            padding: 0,
            margin: 0,
            maxWidth: '100vw',
          },
        }}
        overlayProps={{
          backgroundOpacity: 0.55,
          blur: 3,
        }}
        transitionProps={{ transition: 'fade-up', duration: 300 }}
        title=""
      >
        <PayPalCardFieldsProvider
          style={{
            input: {
              'font-size': '16px',
              'font-family': 'Inter, sans-serif',
              'font-weight': 'normal',
              color: '#1f2937',
            },
            '.invalid': {
              color: '#ef4444',
            },
            ':hover': {
              color: '#3b82f6',
            },
            '.purple': {
              color: '#6b7280',
            },
          }}
          createOrder={createOrder}
          onApprove={onApprove}
          onError={err => {
            const errorObj = extractJsonFromErrorString(err.toString());
            console.log('Parsed error object:', errorObj);
            console.log('Message:', errorObj.message);
            console.log('Issue:', errorObj.details?.[0]?.issue);
            if (errorObj?.details[0]?.issue === 'VALIDATION_ERROR') {
              handleError(errorObj?.details[0]?.description);
            } else if (
              errorObj?.name === 'UNPROCESSABLE_ENTITY' &&
              errorObj?.details[0]?.description === 'UNPROCESSABLE_ENTITY'
            ) {
              handleError('Card details are incorrect');
            } else
              handleError(
                'Cannot Process Your Payment, try again with different card details'
              );
          }}
        >
          <div className="space-y-4 bg-white p-6 rounded-lg shadow-sm border border-gray-200">
            <style>
              {`
                  .paypal-card-fields-container input {
                    border-radius: 6px;
                    padding: 12px;
                    border: 1px solid #d1d5db;
                    box-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
                    width: 100%;
                    box-sizing: border-box;
                  }
                  .paypal-card-fields-container input:hover {
                    border: 1px solid #3b82f6;
                  }
                  .paypal-card-fields-container input.invalid {
                    border: 1px solid #ef4444;
                  }
                `}
            </style>
            <div className="paypal-card-fields-container">
              <PayPalCardFieldsForm />
            </div>

            <SubmitPayment isPaying={isPaying} setIsPaying={setIsPaying} />
          </div>
        </PayPalCardFieldsProvider>
      </Modal>
    </>
  );
};

export default PayPalCheckout;
