import { Box, Button, Loader, Modal } from '@mantine/core';
import { IconMapPin } from '@tabler/icons-react';
import { useState } from 'react';
import { useSelector } from 'react-redux';
import { putData } from '../../api/api';
import { getTranslation } from '../../layout/languages';
import {
  formatDate,
  handleError,
  handleSuccess,
} from '../../utils/common/function';

const NotificationModal = ({ opened, setOpened, eventData }: any) => {
  const [loading, setLoading] = useState<boolean>(false);
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currentRole = userInfo?.roleId;

  const handleAccept = async (eventId: any) => {
    try {
      setLoading(true);

      await putData(`events/${eventId}`, {
        isApproved: true,
      });

      handleSuccess('Notification accepted successfully!');
    } catch (error) {
      console.error('Error accepting notification:', error);
      handleError(error);
    } finally {
      setLoading(false);
    }
  };

  const handleReject = async (eventId: any) => {
    try {
      setLoading(true);

      await putData(`events/${eventId}`, {
        status: 'Rejected',
        isApproved: false,
      });
    } catch (error) {
      console.error('Error rejecting notification:', error);
      handleError(error);
    } finally {
      setLoading(false);
    }
  };
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  return (
    <Modal
      opened={opened}
      onClose={() => setOpened(false)}
      size="md"
      centered
      withCloseButton={false}
      styles={{
        content: {
          borderRadius: '15px',
          backgroundColor: '#6D3EBD',
          padding: 0, // Remove default padding
        },
        body: {
          padding: 0, // Ensure no padding inside modal body
          backgroundColor: '#6D3EBD',
          color: 'white',
        },
        inner: {
          padding: 0, // Ensures modal wrapper has no padding
          margin: 0,
          maxWidth: '100vw',
        },
        header: { backgroundColor: '#6D3EBD', color: 'white' },
      }}
      overlayProps={{
        backgroundOpacity: 0.55,
        blur: 3,
      }}
      transitionProps={{ transition: 'fade-up', duration: 300 }}
    >
      <div className="relative w-full">
        {eventData?.event?.eventImage ? (
          <img
            src={eventData?.event?.eventImage}
            alt="image"
            className="rounded-t-md h-[180px] w-full object-cover"
            loading="lazy"
          />
        ) : (
          <div className="flex items-center justify-center bg-gray-200 text-gray-600 rounded-md h-[160px] w-full text-6xl font-bold">
            {eventData?.title?.charAt(0) || ''}
          </div>
        )}
        <Modal.CloseButton
          onClick={() => setOpened(false)}
          style={{
            position: 'absolute',
            top: 10,
            right: 10,
            border: 'none',
            outline: 'none',
            boxShadow: 'none',
            color: '#ffffff',
            backgroundColor: 'transparent',
            transition: 'background-color 0.2s ease-in-out',
            width: 33,
            height: 33,
          }}
        />
      </div>

      {loading ? (
        <Box
          style={{
            height: '200px',
            display: 'flex',
            justifyContent: 'center',
            alignItems: 'center',
          }}
        >
          <Loader color="white" size="lg" />
        </Box>
      ) : (
        <>
          <div className="mt-4">
            <div className="flex flex-row justify-between ml-8 mr-6">
              <div className="text-[18px] font-productsans text-[#F4F4FE]">
                {eventData?.event?.title}
              </div>

              <div className="text-[14px] font-productsans text-[#F4F4FE] mt-2">
                Ticket Price
              </div>
            </div>
            <div className="flex flex-row justify-between ml-8 mr-6">
              <div className="text-[14px] font-productsans text-[#F4F4FE] flex flex-row justify-start">
                <span className="mr-1 mt-0.5">
                  <IconMapPin size={17} />
                </span>
                <p>
                  {[
                    eventData?.event?.venue,
                    eventData?.event?.city,
                    eventData?.event?.country,
                  ]
                    .filter(Boolean)
                    .join(', ') || 'N/A'}
                </p>
              </div>
              <div className="text-[18px] font-productsans text-[#F4F4FE] ml-28">
                {Array.isArray(eventData?.event?.ticketsAndPrices) &&
                eventData.event.ticketsAndPrices.length > 0
                  ? (() => {
                      const prices = eventData.event.ticketsAndPrices
                        .map((ticket: any) => Number(ticket.price)) // extract numbers
                        .filter((price: number) => !isNaN(price));

                      if (prices.length === 0) return 'N/A';

                      const minPrice = Math.min(...prices).toFixed(2);
                      const maxPrice = Math.max(...prices).toFixed(2);

                      return minPrice === maxPrice
                        ? `${currency === '1' ? '$' : 'BHD ≈'}${minPrice} / ${getTranslation(language, 'event', 'per piece')}`
                        : `${currency === '1' ? '$' : 'BHD ≈'}${minPrice} - ${currency === '1' ? '$' : 'BHD ≈'}${maxPrice} / ${getTranslation(language, 'event', 'per piece')}`;
                    })()
                  : 'N/A'}
              </div>
            </div>
            <div className="bg-[#ffffff] w-auto h-auto bg-opacity-10 rounded-lg mb-3 ml-2 mr-2 ">
              <div
                className={`flex flex-row justify-between mt-4 text-[#ffffff] ml-1 mr-1 text-[13px] font-productsans`}
              >
                <div className="flex flex-col justify-between">
                  <p className="mt-3 ml-4">Event Type </p>
                  <p className="mb-2 ml-4">
                    {eventData?.event?.category?.name}
                  </p>
                </div>
                <div className="border-r-[1px] border-r-caption border-color-light mt-4 mb-3"></div>
                <div className="flex flex-col justify-between">
                  <p className="mt-3">Date</p>
                  <p className="mb-2">
                    {formatDate(
                      eventData?.event?.startDateTime?.substring(0, 10) || ''
                    )}
                  </p>
                </div>
                <div className="border-r-[1px] border-r-caption border-color-light  mt-4 mb-3"></div>
                <div className="flex flex-col justify-between">
                  <p className="mt-3 mr-4">Event Size</p>
                  <p className='"mb-2'> {eventData?.event?.maxAttendees}</p>
                </div>
              </div>
            </div>
            <div className="text-[#BBACD7] font-productsans text-[14px] px-8 py-3">
              {eventData?.event?.description}
            </div>
            <div className="text-[#BBACD7] font-productsans text-[14px] px-8 py-3"></div>

            <div className=" text-[14px] px-8 py-3 bg-[#3B1E6D] bg-opacity-30">
              <div className="text-[#ffffff] text-[14px] font-semibold font-productsans mt-3">
                Submitted From:
              </div>
              <div className="text-[#BBACD7D9] text-[13px] font-productsans mt-1">
                <p className="text-[#ffffff]">User Name:</p>
                <p className="underline">
                  {eventData?.event?.organizedBy?.fullNameOrBusinessName}
                </p>
              </div>
            </div>
            <div className="px-2 mb-3 mr-1 mt-2 flex flex-row justify-between">
              {currentRole === '1' &&
                !eventData?.event?.isApproved &&
                !eventData?.event?.isRejected && (
                  <>
                    <Button
                      autoContrast
                      m={5}
                      size="sm"
                      styles={{
                        root: {
                          background: '#51B74B',
                        },
                      }}
                      style={{
                        height: 42,
                        width: '100%',
                      }}
                      className="px-2"
                      disabled={loading}
                      onClick={() => handleAccept(eventData?.event?.id)}
                    >
                      <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                        {getTranslation(
                          language,
                          'gig',
                          `${loading ? 'Loading' : 'Approve'}`
                        )}
                      </div>
                    </Button>

                    <Button
                      autoContrast
                      m={5}
                      size="sm"
                      styles={{
                        root: {
                          background: '#DD3033',
                        },
                      }}
                      style={{
                        height: 42,
                        width: '100%',
                      }}
                      className="px-2"
                      onClick={() => handleReject(eventData?.event?.id)}
                    >
                      <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                        {getTranslation(
                          language,
                          'dashboard',
                          `${loading ? 'Loading' : 'Reject'}`
                        )}
                      </div>
                    </Button>
                  </>
                )}
            </div>
          </div>
        </>
      )}
    </Modal>
  );
};

export default NotificationModal;
