import { Group, Loader, Paper, Text, Title } from '@mantine/core';
import dayjs from 'dayjs';
import relativeTime from 'dayjs/plugin/relativeTime';
import { useEffect, useState } from 'react';
import { AiOutlineArrowLeft } from 'react-icons/ai';
import { useSelector } from 'react-redux';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { fetchData, postData, putData } from '../../api/api';
import OnlyPagination from '../../components/table/onlypagination';
import { getTranslation } from '../../layout/languages';
import { paginationInfoValue } from '../../utils/common/constant.objects';
import {
  extractPageInfo,
  handleError,
  handleSuccess,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import ReportEvents from '../events/report-event';
import SuccessModal from '../events/success';
import NotificationCard from './NotificationCard';
import NotificationModal from './notificationModal';
import ProfileModal from './profileModal';

dayjs.extend(relativeTime);

const Notification = () => {
  const navigate = useNavigate();
  const [notifications, setNotifications] = useState<any[]>([]);
  const [accountInfo, setAccountInfo] = useState(false);
  const [eventId, setEventId] = useState(null);
  const [againstUserId, setAgainstUserId] = useState(null);
  const [isSuccessModal, setIsSuccessModal] = useState(false);
  const [notificationId, setNotificationId] = useState(null);
  const [gigId, setGigId] = useState(null);

  const [searchParams, setSearchParams] = useSearchParams();
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );

  const initialSearchValues = {
    searchValue: '',
    type: '',
    limit: '',
    sort: '',
  };
  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;

    return {
      searchValue,
      type,
      limit,
      sort,
    };
  };
  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );

  const [eventInfo, setEventInfo] = useState({
    isModalOpen: false,
    event: {},
  });
  const [profileInfo, setProfileInfo] = useState({
    isModalOpen: false,
    profile: {},
  });

  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );

  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(false);

  // Extracted reusable fetch function
  const fetchNotifications = async () => {
    if (!userInfo?.id) return;

    setLoading(true);
    setError(false);

    try {
      const filters = removeEmptyValueFilters([
        {
          field: 'event.title',
          operator: 'eq',
          value: searchValues.searchValue,
        },
      ]);

      const filterObject = JSON.stringify({ filter: filters });
      const queryParams = new URLSearchParams({
        rpp: paginationInfo.rowPerPage.toString(),
        page: (paginationInfo.currentPage === 0
          ? 1
          : paginationInfo.currentPage
        ).toString(),
        filter: filterObject,
      });

      const url = `notifications/${userInfo?.id}?${queryParams.toString()}`;
      const response: any = await fetchData(url);

      if (!response.data || response.data.length === 0) {
        setNotifications(response.data || []);
        if (response.total > 0) {
          setPaginationInfo({
            ...paginationInfo,
            totalRecords: response.total,
          });
        }
      } else {
        setNotifications(response.data);
        const getPages = extractPageInfo(response.pages);
        setPaginationInfo({
          ...paginationInfo,
          totalRecords: response.total,
          totalPages: getPages?.totalPages ?? 0,
        });
      }
    } catch (err) {
      handleError(err, 'Failed to fetch table data');
      setError(true);
    } finally {
      setLoading(false);
    }
  };

  // Auto re-fetch on mount & when pagination changes
  useEffect(() => {
    fetchNotifications();
  }, [userInfo, paginationInfo?.currentPage]);

  const handlePagination = (actionType: string, value?: any) => {
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  };

  // ---------------- ACTION HANDLERS ----------------

  const handleAccept = async (notification: any) => {
    try {
      setLoading(true);

      if (notification.type === 'Profile') {
        await Promise.all([
          putData(`users/approve/${notification.profileId}`, {}),
          putData(`notifications/user-notification/${notification.id}`, {
            actionPerformed: true,
            userId: userInfo?.id,
          }),
        ]);
        handleSuccess('Notification accepted successfully!');
      } else if (notification.type === 'Event') {
        await Promise.all([
          putData(`events/${notification.eventId}`, { isApproved: true }),
          putData(`notifications/user-notification/${notification.id}`, {
            actionPerformed: true,
            userId: userInfo?.id,
          }),
        ]);
        handleSuccess('Notification accepted successfully!');
      } else if (notification.type === 'Payment Release') {
        await Promise.all([
          putData(`notifications/user-notification/${notification.id}`, {
            actionPerformed: true,
            userId: userInfo?.id,
          }),
          postData(`transactions/${notification.gigId}/release`, {}),
        ]);
        setEventId(notification.eventId);
        setAgainstUserId(notification?.gig?.entertainerId);
        setIsSuccessModal(true);
      }

      // Always refresh after action
      fetchNotifications();
    } catch (error) {
      console.error('Error accepting notification:', error);
      handleError(error);
    } finally {
      setLoading(false);
    }
  };

  const handleReject = async (notification: any) => {
    try {
      setLoading(true);

      if (notification.type === 'Profile') {
        await Promise.all([
          putData(`profiles/${notification.profileId}`, { isRejected: true }),
          putData(
            `notifications/${notification.id}/mark-as-read/${userInfo.id}`,
            {}
          ),
          putData(`notifications/user-notification/${notification.id}`, {
            actionPerformed: true,
            userId: userInfo?.id,
          }),
        ]);
      } else if (notification.type === 'Event') {
        await Promise.all([
          putData(`events/${notification.eventId}`, {
            status: 'Rejected',
            isApproved: false,
          }),
          putData(
            `notifications/${notification.id}/mark-as-read/${userInfo.id}`,
            {}
          ),
          putData(`notifications/user-notification/${notification.id}`, {
            actionPerformed: true,
            userId: userInfo?.id,
          }),
        ]);
      } else if (notification.type === 'Payment Release') {
        setEventId(notification.eventId);
        setAgainstUserId(notification?.gig?.entertainerId);
        setGigId(notification?.gig?.id);
        setNotificationId(notification?.id);
        setAccountInfo(true);
      }

      // Refresh notifications
      fetchNotifications();
    } catch (error) {
      console.error('Error rejecting notification:', error);
      handleError(error);
    } finally {
      setLoading(false);
    }
  };

  const handleBookmark = async (notificationId: string) => {
    try {
      await putData(
        `notifications/${notificationId}/mark-as-bookmarked/${userInfo.id}`,
        {}
      );
      handleSuccess('Bookmark status updated successfully!');

      // Refresh after bookmarking
      fetchNotifications();
    } catch (error) {
      console.error('Error bookmarking notification:', error);
      handleError(error);
    }
  };

  return (
    <Paper
      radius={0}
      className="flex justify-between items-center p-4 min-h-screen "
      style={{
        backgroundColor: '#3B1E6D',
        background: 'radial-gradient(#6534BA 43%, #3B1E6D)',
      }}
    >
      <Group
        style={{
          justifyContent: 'space-between',
          alignItems: 'center',
          marginBottom: '30px',
        }}
      >
        <Group>
          <AiOutlineArrowLeft
            size={22}
            color="white"
            onClick={() => navigate('/dashboard')}
          />
          <Title
            style={{
              color: 'white',
              fontSize: '28px',
              fontWeight: 'bold',
              fontFamily: 'Product Sans',
            }}
          >
            {getTranslation(language, 'angelwing', 'Notifications')}
          </Title>
        </Group>
      </Group>
      <div className="rounded-2xl font-semibold text-[#ffffff] p-4 font-productsans  bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
        {/* Dynamic Notifications */}
        {loading ? (
          <div className="flex items-center justify-center w-full h-[256px]">
            <Loader size="lg" color="#C28CFF" />
          </div>
        ) : notifications.length === 0 ? (
          <div className="flex items-center justify-center w-full h-[300px]">
            <Text
              style={{
                color: 'white',
                fontSize: '16px',
                fontFamily: 'Product Sans',
                textAlign: 'center',
              }}
            >
              No notifications found.
            </Text>
          </div>
        ) : (
          notifications.map(notification => (
            <NotificationCard
              key={notification.id}
              notification={notification}
              userInfo={userInfo}
              language={language}
              handleAccept={handleAccept}
              handleReject={handleReject}
              handleBookmark={handleBookmark}
              setEventInfo={setEventInfo}
              setProfileInfo={setProfileInfo}
            />
          ))
        )}
      </div>

      {!loading && (
        <div className="ml-9">
          <OnlyPagination
            data={notifications}
            columns={[]}
            isLoading={false}
            paginationInfo={paginationInfo}
            handlePagination={handlePagination}
          />
        </div>
      )}

      {/* Notification Modal */}
      <ReportEvents
        opened={accountInfo}
        setOpened={setAccountInfo}
        eventId={eventId}
        againstUserId={againstUserId}
        notificationId={notificationId}
        setResetTable={fetchNotifications} // replaced old resetTable toggle
        gigId={gigId}
      />

      <NotificationModal
        opened={eventInfo?.isModalOpen}
        setOpened={setEventInfo}
        eventData={eventInfo?.event}
      />

      <ProfileModal
        opened={profileInfo?.isModalOpen}
        setOpened={setProfileInfo}
        profileData={profileInfo?.profile}
      />

      <SuccessModal
        opened={isSuccessModal}
        setOpened={setIsSuccessModal}
        isPaymentRelease
        eventId={eventId}
        againstUserId={againstUserId}
        setResetTable={fetchNotifications} // replaced old resetTable toggle
      />
    </Paper>
  );
};

export default Notification;
