import { Avatar, Badge, Box, Grid, Group, Paper, Text } from '@mantine/core';
import { IconCalendar, IconClock } from '@tabler/icons-react';
import dayjs from 'dayjs';
import { getTranslation } from '../../layout/languages';
import NotificationAction from './NotificationActions';

interface NotificationCardProps {
  viewDetails?: boolean;
  notification: any;
  userInfo: any;
  language: string;
  handleAccept: (notification: any) => void;
  handleReject: (notification: any) => void;
  handleBookmark: (notificationId: string) => void;
  setEventInfo?: (val: any) => void;
  setProfileInfo?: (val: any) => void;
}

const NotificationCard = ({
  viewDetails = true,
  notification,
  userInfo,
  language,
  handleAccept,
  handleReject,
  handleBookmark,
  setEventInfo,
  setProfileInfo,
}: NotificationCardProps) => {
  const target =
    notification.type === 'Profile' ? notification.profile : notification.event;

  const isApproved = target?.isApproved;
  const isRejected = target?.isRejected;
  const date = new Date(notification.createdAt).toLocaleDateString();
  const userNotification = notification.userNotifications?.[0];
  const isBookmarked = userNotification?.isBookedMarked || false;
  const isUnread = userNotification?.isRead === false;

  return (
    <Paper
      withBorder
      shadow="md"
      p="xl"
      mb="2.5%"
      mt="2%"
      style={{
        borderRadius: '10px',
        background: isUnread
          ? 'linear-gradient(135deg, rgba(66,133,244,0.15) 0%, rgba(255,255,255,0.05) 100%)'
          : 'rgba(255, 255, 255, 0.08)',
        backdropFilter: 'blur(4px)',
        padding: '29px',
        border: isUnread
          ? '1.5px solid rgba(66,133,244,0.6)'
          : '1px solid rgba(255, 255, 255, 0.2)',
        minHeight: '150px',
        width: '98%',
        margin: '0 auto',
        display: 'flex',
        flexDirection: 'column',
        justifyContent: 'space-between',
        fontFamily: 'Product Sans',
        transition: 'all 0.25s ease',
      }}
      className="px-4 py-4 hover:shadow-lg hover:scale-[1.01]"
    >
      <div
        style={{
          display: 'flex',
          flexDirection: 'row',
          justifyContent: 'space-between',
          alignItems: 'center',
        }}
      >
        <Grid className="flex flex-row justify-between ">
          <Grid.Col
            span={{
              base: 12,
              md: 6,
              lg: 2,
            }}
          >
            <Box>
              <Avatar
                src={
                  notification?.profile?.profilePicture ||
                  notification?.event?.eventImage ||
                  null
                }
                alt="User"
                size={50}
                radius="xl"
                color="gray"
              >
                {notification?.profile?.fullNameOrBusinessName?.[0] ||
                  notification?.event?.title?.[0] ||
                  notification.type?.[0] ||
                  ''}
              </Avatar>

              {isUnread && (
                <Badge
                  color="blue"
                  variant="filled"
                  radius="xl"
                  size="xs"
                  style={{
                    position: 'absolute',
                    top: 0,
                    right: 0,
                    transform: 'translate(35%, -35%)',
                    boxShadow: '0 0 6px rgba(66,133,244,0.8)',
                  }}
                >
                  New
                </Badge>
              )}
            </Box>
          </Grid.Col>

          <Grid.Col
            span={{
              base: 12,
              md: 6,
              lg: 11,
            }}
            className="py-5"
          >
            <Group
              gap="md"
              align="center"
              style={{ justifyContent: 'flex-start' }}
            >
              <Text
                style={{
                  color: 'white',
                  fontSize: '18px',
                  flex: 1,
                  fontWeight: isUnread ? 700 : 500,
                }}
                className="font-productsans"
              >
                {notification.title}
              </Text>
            </Group>

            {/* Notification Description */}
            <Text
              style={{
                color: 'white',
                fontSize: '14px',
                opacity: isUnread ? 1 : 0.85,
              }}
              className="font-productsans"
            >
              {notification.message}
            </Text>

            {/* Date & Time */}
            <Box mt="xs">
              <Group
                style={{
                  color: 'white',
                  fontSize: '14px',
                  fontWeight: 'bold',
                  fontFamily: 'Product Sans',
                  justifyContent: 'flex-start',
                }}
              >
                <div className="flex flex-row gap-2">
                  <IconClock className="mt-1" size={14} />
                  <Text>{dayjs(notification.createdAt).fromNow()}</Text>
                </div>
                <div className="flex flex-row gap-2 font-normal">|</div>
                <div className="flex flex-row gap-2">
                  <IconCalendar className="mt-1" size={14} />
                  <Text>{date}</Text>
                </div>
                {viewDetails && (
                  <>
                    {(notification?.type?.toLowerCase() === 'event' ||
                      notification?.type?.toLowerCase() === 'profile') && (
                      <>
                        <div className="flex flex-row gap-2 font-normal">|</div>
                        <Text
                          className="font-productsans text-white text-[14px] cursor-pointer underline"
                          onClick={() => {
                            if (
                              notification?.type?.toLowerCase() === 'event' &&
                              setEventInfo
                            ) {
                              setEventInfo({
                                isModalOpen: true,
                                event: notification,
                              });
                            } else if (
                              notification?.type === 'Profile' &&
                              setProfileInfo
                            ) {
                              setProfileInfo({
                                isModalOpen: true,
                                profile: notification,
                              });
                            }
                          }}
                        >
                          {getTranslation(
                            language,
                            'angelwing',
                            'View Details'
                          )}
                        </Text>
                      </>
                    )}
                  </>
                )}
              </Group>
            </Box>
          </Grid.Col>
        </Grid>
        {/* Action Buttons */}

        <NotificationAction
          notification={notification}
          isApproved={isApproved}
          isRejected={isRejected}
          isBookmarked={isBookmarked}
          handleAccept={handleAccept}
          handleReject={handleReject}
          handleBookmark={handleBookmark}
          userInfo={userInfo}
          language={language}
          getTranslation={getTranslation}
        />
      </div>
    </Paper>
  );
};

export default NotificationCard;
