import { Button, Chip, Group } from '@mantine/core';
import { IconCheck, IconFlag3Filled, IconX } from '@tabler/icons-react';
import React from 'react';
import { RiFlag2Line } from 'react-icons/ri';
import NotificationPerformed from './ActionPerformed';

interface NotificationActionProps {
  notification: any;
  userInfo: { roleId: string };
  language: string;
  isApproved: boolean;
  isRejected: boolean;
  isBookmarked: boolean;
  handleAccept: (notification: any) => void;
  handleReject: (notification: any) => void;
  handleBookmark: (id: string) => void;
  getTranslation: (lang: string, section: string, key: string) => string;
}

// 🔹 Shared styles
const buttonBase: React.CSSProperties = {
  color: 'white',
  fontWeight: 400,
  fontFamily: "'Segoe UI', 'Roboto', Product Sans",
  fontSize: '14px',
  letterSpacing: '0.5px',
  cursor: 'pointer',
  transition: 'all 0.3s ease-in-out',
};

const NotificationAction: React.FC<NotificationActionProps> = ({
  notification,
  userInfo,
  language,
  isApproved,
  isRejected,
  isBookmarked,
  handleAccept,
  handleReject,
  handleBookmark,
  getTranslation,
}) => {
  const shouldRender =
    ((notification.type === 'Profile' ||
      notification.type === 'Event' ||
      (notification.type === 'Payment Release' && userInfo.roleId !== '1')) &&
      notification.title !== 'Gig Cancelled' &&
      notification.title !== 'Event Completed Successfully!' &&
      notification.title !== 'Event Performance Completed!') ||
    (notification.type === 'Dispute' && userInfo.roleId === '1');

  if (!shouldRender) return null;

  return (
    <Group
      style={{
        display: 'flex',
        justifyContent: 'center',
        alignItems: 'center',
      }}
    >
      {notification.userNotifications?.[0]?.actionPerformed &&
      notification.type !== 'Profile' ? (
        <NotificationPerformed
          language={language}
          getTranslation={getTranslation}
        />
      ) : isApproved && notification.type !== 'Payment Release' ? (
        <Chip
          checked
          icon={<IconCheck size={16} color="white" />}
          color="#418F90"
          variant="filled"
          readOnly
          styles={{
            root: {
              color: '#418F90',
              borderRadius: '16px',
              padding: '8px 14px',
              fontSize: '13px',
              fontWeight: 500,
              cursor: 'default',
            },
            label: { padding: 8 },
          }}
        >
          {getTranslation(language, 'dashboard', 'Approved by Admin')}
        </Chip>
      ) : isRejected && notification.type !== 'Payment Release' ? (
        <Chip
          checked
          icon={<IconX size={16} color="white" />}
          color="red"
          variant="filled"
          readOnly
          styles={{
            root: {
              color: 'white',
              borderRadius: '16px',
              padding: '6px 14px',
              fontSize: '13px',
              fontWeight: 500,
              cursor: 'default',
            },
            label: { padding: 8 },
          }}
        >
          {getTranslation(language, 'dashboard', 'Rejected by Admin')}
        </Chip>
      ) : (
        <>
          {/* Approve / Release Payment */}
          <Button
            style={{
              ...buttonBase,
              backgroundColor: '#418F90',
            }}
            onClick={() => handleAccept(notification)}
          >
            {notification.type === 'Payment Release'
              ? getTranslation(language, 'dashboard', 'Release Payment')
              : getTranslation(language, 'dashboard', 'Approve')}
          </Button>

          {/* Reject / Dispute */}
          <Button
            style={{
              ...buttonBase,
              backgroundColor: '#f44336',
            }}
            onClick={() => handleReject(notification)}
          >
            {notification.type === 'Payment Release'
              ? getTranslation(language, 'dashboard', 'Dispute')
              : getTranslation(language, 'dashboard', 'Reject')}
          </Button>

          {/* Bookmark */}
          {isBookmarked ? (
            <IconFlag3Filled
              color="#ffffff"
              size={24}
              style={{ cursor: 'pointer' }}
              onClick={() => handleBookmark(notification.id)}
            />
          ) : (
            <RiFlag2Line
              color="white"
              size={22}
              style={{ cursor: 'pointer' }}
              onClick={() => handleBookmark(notification.id)}
            />
          )}
        </>
      )}
    </Group>
  );
};

export default NotificationAction;
