import { Button, Grid, Modal } from '@mantine/core';
import { IconMapPin } from '@tabler/icons-react';
import { useState } from 'react';
import { useSelector } from 'react-redux';
import { putData } from '../../api/api';
import { getTranslation } from '../../layout/languages';
import { GigStatus } from '../../utils/common/enums';
import {
  formatDate,
  handleError,
  handleSuccess,
} from '../../utils/common/function';

const ViewDetails = ({ opened, setOpened, event, onStatusChange }: any) => {
  const [isLoading, setIsLoading] = useState(false);
  const [loadingButton, setLoadingButton] = useState<
    'accept' | 'reject' | null
  >(null);

  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );

  const gigData = event;
  const handleGigStatusUpdate = async (
    gigId: string,
    newStatus: GigStatus.ACCEPTED | GigStatus.REJECTED
  ) => {
    setIsLoading(true);
    try {
      if (newStatus === GigStatus.ACCEPTED) {
        await putData(`gig/${gigId}/accept`, {});
      } else if (newStatus === GigStatus.REJECTED) {
        await putData(`gig/${gigId}/reject`, {});
      }
      if (onStatusChange) {
        onStatusChange();
      }

      handleSuccess(`Gig ${newStatus} successfully`);

      setTimeout(() => {
        setOpened(false);
      }, 1000);
    } catch (error) {
      console.error('Error updating gig status:', error);
      handleError(error);
    }
  };

  return (
    <Modal
      opened={opened}
      centered
      size="md"
      onClose={() => setOpened(false)}
      withCloseButton={false}
      styles={{
        content: {
          borderRadius: '15px',
          backgroundColor: '#6D3EBD',
          padding: 0, // Remove default padding
        },
        body: {
          padding: 0, // Ensure no padding inside modal body
        },
        inner: {
          padding: 0, // Ensures modal wrapper has no padding
          margin: 0,
          maxWidth: '100vw',
        },
      }}
      overlayProps={{
        backgroundOpacity: 0.55,
        blur: 3,
      }}
      transitionProps={{ transition: 'fade-up', duration: 300 }}
      title=""
    >
      <div className="relative w-full">
        {gigData?.event?.eventImage ? (
          <img
            src={gigData?.event?.eventImage}
            alt="image"
            className="rounded-t-md h-[180px] w-full object-cover"
            loading="lazy"
          />
        ) : (
          <div className="rounded-t-md h-[30px] w-full object-cover"> </div>
        )}
        <Modal.CloseButton
          onClick={() => setOpened(false)}
          style={{
            position: 'absolute',
            top: 10,
            right: 10,
            border: 'none',
            outline: 'none',
            boxShadow: 'none',
            color: '#ffffff',
            backgroundColor: 'transparent',
            transition: 'background-color 0.2s ease-in-out',
            width: 33,
            height: 33,
          }}
        />
      </div>
      <div className="mt-4">
        <div className="flex flex-row justify-between ml-8 mr-6">
          <div className="text-[18px] font-productsans text-[#F4F4FE]">
            {gigData?.event?.title || 'N/A'}
          </div>

          <div className="text-[14px] font-productsans text-[#F4F4FE] mt-2">
            {getTranslation(language, 'event', 'Ticket Price')}
          </div>
        </div>
        <div className="flex flex-row justify-between ml-8 mr-6">
          <div className="text-[14px] font-productsans text-[#F4F4FE] flex flex-row justify-start">
            <span className="mr-1 mt-0.5">
              <IconMapPin size={17} />
            </span>
            <p>
              {[
                gigData?.event?.venue,
                gigData?.event?.city,
                gigData?.event?.country,
              ]
                .filter(Boolean)
                .join(', ') || 'N/A'}
            </p>
          </div>
          <div className="text-[18px] font-productsans text-[#F4F4FE] ml-16">
            {Array.isArray(gigData?.event?.ticketsAndPrices) &&
            gigData.event.ticketsAndPrices.length > 0
              ? (() => {
                  const prices = gigData.event.ticketsAndPrices
                    .map((ticket: any) => Number(ticket.price))
                    .filter((price: any) => !isNaN(price));

                  if (prices.length === 0) return 'N/A';

                  const minPrice = Math.min(...prices).toFixed(2);
                  const maxPrice = Math.max(...prices).toFixed(2);

                  return minPrice === maxPrice
                    ? `${currency === '1' ? '$' : 'BHD ≈'}${minPrice} / ${getTranslation(language, 'event', 'per piece')}`
                    : `${currency === '1' ? '$' : 'BHD ≈'}${minPrice} - ${currency === '1' ? '$' : 'BHD ≈'}${maxPrice} / ${getTranslation(language, 'event', 'per piece')}`;
                })()
              : 'N/A'}
          </div>
        </div>
        <div className="bg-[#ffffff] w-auto h-auto bg-opacity-10 rounded-lg mb-3 ml-2 mr-2 ">
          <div
            className={`flex flex-row justify-between mt-4 text-[#ffffff] ml-1 mr-1 text-[13px] font-productsans`}
          >
            <div className="flex flex-col justify-between">
              <p className="mt-3 ml-4 rtl:mr-4 rtl:ml-0">
                {getTranslation(language, 'event', 'Event Type')}
              </p>
              <p className="mb-2 ml-4 rtl:mr-4 rtl:ml-0">
                {gigData?.event?.category?.name || 'N/A'}
              </p>
            </div>
            <div className="border-r-[1px] border-r-caption border-color-light mt-4 mb-3"></div>
            <div className="flex flex-col justify-between">
              <p className="mt-3">
                {getTranslation(language, 'event', 'Date')}
              </p>
              <p className="mb-2">
                {formatDate(
                  gigData?.event?.startDateTime?.substring(0, 10) || ''
                )}
              </p>
            </div>
            <div className="border-r-[1px] border-r-caption border-color-light  mt-4 mb-3"></div>
            <div className="flex flex-col justify-between">
              <p className="mt-3 mr-4 rtl:mr-0 rtl:ml-4">
                {getTranslation(language, 'event', 'Event Size')}
              </p>
              <p className="mr-4 mb-2">
                {gigData?.event?.maxAttendees || 'N/A'}
              </p>
            </div>
          </div>
        </div>
        <div className="text-[#BBACD7] font-productsans text-[14px] px-8 py-3">
          {gigData?.event?.description || 'N/A'}
        </div>
        <div className=" text-[14px] px-8 py-3 bg-[#3B1E6D] bg-opacity-30">
          <div className="text-[#ffffff] text-[14px] font-semibold font-productsans">
            {getTranslation(language, 'gig', 'Gig Offer')}:
          </div>
          <Grid>
            <Grid.Col span={{ base: 12, md: 6, lg: 8 }}>
              <div className="flex flex-row justify-between mt-4 font-semibold">
                <div className="text-[12px] font-productsans text-[#ffffff]">
                  {getTranslation(language, 'gig', 'Performance Offer')}
                </div>
                <div className="text-[12px] font-productsans text-[#ffffff] ml-7 ">
                  {getTranslation(language, 'gig', 'Payment Method')}
                </div>
              </div>
            </Grid.Col>
          </Grid>
          <Grid>
            <Grid.Col span={{ base: 12, md: 6, lg: 7.5 }}>
              <div className="flex flex-row justify-between -mt-1 ">
                <div className="text-[12px] font-productsans text-[#ffffff]">
                  {currency === '1' ? '$' : 'BHD ≈'}{' '}
                  {gigData?.proposedPerformanceFee?.toFixed(2) || 'N/A'}
                </div>
                <div className="text-[12px] font-productsans text-[#ffffff] ml-20 rtl:mr-20 rtl:ml-0">
                  {gigData?.paymentType || 'N/A'}
                </div>
              </div>
            </Grid.Col>
          </Grid>
          <div className="text-[#ffffff] text-[14px] font-semibold font-productsans mt-3">
            {getTranslation(language, 'gig', 'Message')}
          </div>
          <div className="text-[#BBACD7] font-productsans text-[12px] py-2">
            {gigData?.message}
          </div>
          <div className="text-[#ffffff] text-[14px] font-semibold font-productsans mt-3">
            {getTranslation(language, 'gig', 'Organizer Details')}
          </div>
          <div className="text-[#BBACD7D9] text-[13px] font-productsans mt-1">
            <p>
              {gigData?.event?.organizedBy?.fullNameOrBusinessName || 'N/A'}
            </p>
            <p>{gigData?.event?.organizedBy?.email || 'N/A'}</p>
            <p>{gigData?.event?.organizedBy?.phoneNumber || 'N/A'}</p>
          </div>
        </div>
        <div className="px-2 mb-3 mr-1 mt-2 flex flex-row justify-between">
          {gigData?.status === GigStatus.PENDING && (
            <>
              <Button
                autoContrast
                m={5}
                size="sm"
                styles={{
                  root: {
                    background: '#51B74B',
                  },
                }}
                style={{
                  height: 42,
                  width: '100%',
                }}
                className="px-2"
                onClick={async () => {
                  setLoadingButton('accept');
                  await handleGigStatusUpdate(gigData.id, GigStatus.ACCEPTED);
                  setLoadingButton(null);
                }}
                disabled={loadingButton !== null}
              >
                <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                  {getTranslation(
                    language,
                    'gig',
                    `${loadingButton === 'accept' ? 'Loading' : 'Accept Now'}`
                  )}
                </div>
              </Button>
              <Button
                autoContrast
                m={5}
                size="sm"
                styles={{
                  root: {
                    background: '#DD3033',
                  },
                }}
                style={{
                  height: 42,
                  width: '100%',
                }}
                className="px-2"
                onClick={async () => {
                  setLoadingButton('reject');
                  await handleGigStatusUpdate(gigData.id, GigStatus.REJECTED);
                  setLoadingButton(null);
                }}
                disabled={loadingButton !== null}
              >
                <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                  {getTranslation(
                    language,
                    'dashboard',
                    `${loadingButton === 'reject' ? 'Loading' : 'Reject'}`
                  )}
                </div>
              </Button>
            </>
          )}
        </div>
      </div>
    </Modal>
  );
};

export default ViewDetails;
