import { Grid, Paper, Select, TextInput } from '@mantine/core';
import { IconChevronDown } from '@tabler/icons-react';
import { useEffect, useMemo, useState } from 'react';
import { useSearchParams } from 'react-router-dom';
import { Table } from '../../components';
import { filterStyle, textAreaStyle } from '../../theme/common.style';
import {
  AMOUNT_LIMIT_OPTIONS,
  paginationInfoValue,
  SORT_OPTIONS,
} from '../../utils/common/constant.objects';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import { fetchData } from '../../api/api';
import useScreenSize from '../../hooks/useScreenSize';
import {
  extractPageInfo,
  handleError,
  handleSetParams,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import ViewDetails from './view-details';
import { useSelector } from 'react-redux';
import { DebouncedTextInput } from '../../components/DebouncedTextInput';
import { getTranslation } from '../../layout/languages';

interface SearchFilter {
  searchValue: string;
  type: string;
  limit: string;
  sort: string;
}

const initialSearchValues: SearchFilter = {
  searchValue: '',
  type: '',
  limit: '',
  sort: '',
};

const Gigs = () => {
  const { isSmallScreen, isMediumScreen, isLargeScreen } = useScreenSize();
  const [searchParams, setSearchParams] = useSearchParams();
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const [accountInfo, setAccountInfo] = useState({
    isModalOpen: false,
    id: {},
  });
  const [isLoading, setIsLoading] = useState(false);
  const [tableData, setTableData] = useState([]);
  const [resetTable, setResetTable] = useState(false);

  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);

  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );

  useEffect(() => {
    handleFetchDataByFilter();
  }, [currency]);
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const handlePagination = (actionType: string, value?: any) => {
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  };

  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;

    return {
      searchValue,
      type,
      limit,
      sort,
    };
  };

  const [searchValues, setSearchValues] = useState<SearchFilter>(
    initializeStateFromQueryParams()
  );

  const handleFetchDataByFilter = () => {
    setIsLoading(true);

    const filters = removeEmptyValueFilters([
      {
        field: 'event.title',
        operator: 'like',
        value: searchValues.searchValue,
      },
      {
        field: 'gig.entertainerId',
        operator: 'eq',
        value: userInfo?.id,
      },
      {
        field: 'gig.status',
        operator: 'eq',
        value: searchValues.type,
      },
    ]);

    const filterObject = JSON.stringify({ filter: filters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });

    if (searchValues.sort) {
      queryParams.append(
        'orderBy',
        JSON.stringify({
          createdAt: searchValues.sort,
        })
      );
    }

    const fetchUrl = `gig?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        if (!response.data || response.data.length === 0) {
          setTableData([]);
          if (response.total > 0) {
            setPaginationInfo({
              ...paginationInfo,
              totalRecords: response.total,
            });
          }
        } else {
          setTableData(response.data);
          const getPages = extractPageInfo(response.pages);
          setPaginationInfo({
            ...paginationInfo,
            totalRecords: response.total,
            totalPages: getPages?.totalPages ?? 0,
          });
        }
      })
      .catch(error => {
        handleError(error, 'Failed to fetch table data');
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };

  useEffect(() => {
    handleSearchButtonClick();
  }, [
    resetTable,
    searchValues.searchValue,
    searchValues.type,
    searchValues.limit,
    searchValues.sort,
    paginationInfo?.currentPage,
    paginationInfo?.rowPerPage,
  ]);

  useEffect(() => {
    const newSearchValues = initializeStateFromQueryParams();
    if (JSON.stringify(newSearchValues) !== JSON.stringify(searchValues)) {
      setSearchValues(newSearchValues);
    }
  }, [searchParams]);

  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById,
    }));
  };

  const columns = useMemo(
    () => [
      {
        header: (
          <div className="flex text-start font-normal font-productsans text-[14px] rtl:mr-3">
            {getTranslation(language, 'dashboard', 'ID')}
          </div>
        ),
        accessorKey: 'id',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start rtl:mr-2">
              <p className="text-start font-normal font-productsans text-[14px]">
                {rowObject?.orderId}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center ml-4 rtl:mr-4 rtl:ml-0 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'dashboard', 'Events')}
          </div>
        ),
        accessorKey: 'event',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex items-center text-start mr-5 rtl:mr-0">
              {rowObject?.event?.eventImage ? (
                <img
                  src={rowObject.event.eventImage}
                  alt="Image"
                  className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 object-cover flex-shrink-0"
                  loading="lazy"
                />
              ) : (
                <div className="rounded-md w-9 h-6 rtl:ml-6 rtl:-mr-4 bg-gray-300 flex items-center justify-center text-xs font-semibold uppercase flex-shrink-0">
                  {rowObject?.event?.title?.[0] || 'E'}
                </div>
              )}
              <p className="text-start ml-2 font-productsans font-normal text-[14px] leading-snug">
                {rowObject?.event?.title || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-start ml-4 rtl:mr-8 rtl:ml-0 font-normal font-productsans text-[14px]">
            {getTranslation(language, 'dashboard', 'Date')}
          </div>
        ),
        accessorKey: 'date',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start ml-1 rtl:mr-4 rtl:ml-0">
              <p className="text-start font-productsans font-normal text-[14px] ">
                {rowObject?.event?.startDateTime?.substring(0, 10) || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex ml-4 rtl:mr-28 rtl:ml-0 font-normal font-productsans text-[14px]' : 'flex ml-16 rtl:mr-28 rtl:ml-0 font-normal font-productsans text-[14px]'}`}
          >
            {getTranslation(language, 'sales', 'Details')}
          </div>
        ),
        accessorKey: 'action',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div
              className={` ${isSmallScreen ? 'flex items-start justify-start font-normal text-[13px] mr-36 font-productsans underline  rtl:ml-4 rtl:mr-0' : 'flex items-center justify-center font-normal text-[13px] font-productsans underline mr-11 rtl:ml-4 rtl:mr-0'}`}
            >
              <span
                onClick={() =>
                  setAccountInfo({
                    isModalOpen: true,
                    id: rowObject,
                  })
                }
              >
                {getTranslation(language, 'sales', 'View More')}
              </span>
            </div>
          );
        },
      },
    ],
    [tableData, language, currency]
  );

  return (
    <Paper
      className="flex justify-between items-center p-4 min-h-screen"
      radius={0}
      style={{
        backgroundColor: '#3B1E6D',
        background: 'radial-gradient(#6534BA 43%, #3B1E6D)',
      }}
    >
      <Grid className="mt-6 mr-7 ml-4">
        <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
          <div className="text-[33px] font-semibold text-[#ffffff] font-productsans ">
            {getTranslation(language, 'navbar', 'Gigs')}
          </div>
        </Grid.Col>
        {(isLargeScreen || isMediumScreen) && (
          <Grid.Col span={{ base: 12, md: 3, lg: 2 }}></Grid.Col>
        )}
        <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
          <DebouncedTextInput
            placeholder={getTranslation(
              language,
              'event',
              'Search by Event Name'
            )}
            value={searchValues.searchValue}
            onDebouncedChange={e =>
              setValuesById({
                searchValue: e || '',
              })
            }
            styles={textAreaStyle}
            classNames={{ input: 'text-input' }}
            radius="6px"
          />
        </Grid.Col>
        <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
          <Select
            placeholder={getTranslation(language, 'event', 'Select Status')}
            data={[
              { value: 'Pending', label: 'Pending' },
              { value: 'Accepted', label: 'Accepted' },
              { value: 'Rejected', label: 'Rejected' },
              { value: 'Cancelled', label: 'Cancelled' },
            ]}
            value={searchValues.type}
            clearable
            onChange={value => setValuesById({ type: value || '' })}
            rightSection={<IconChevronDown size={16} color="white" />}
            styles={filterStyle}
            classNames={{ input: 'custom-input' }}
            radius="6px"
          />
        </Grid.Col>
        <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
          <Select
            placeholder={getTranslation(language, 'event', 'Amount Limit 1-10')}
            data={AMOUNT_LIMIT_OPTIONS}
            value={paginationInfo.rowPerPage}
            onChange={value => value && handlePagination('rowPerPage', value)}
            rightSection={<IconChevronDown size={16} color="white" />}
            styles={filterStyle}
            classNames={{ input: 'custom-input' }}
            radius="6px"
          />
        </Grid.Col>
        <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
          <Select
            placeholder={getTranslation(language, 'event', 'Sort By')}
            data={SORT_OPTIONS}
            value={searchValues.sort}
            clearable
            onChange={value => setValuesById({ sort: value || '' })}
            rightSection={<IconChevronDown size={16} color="white" />}
            styles={filterStyle}
            classNames={{ input: 'custom-input' }}
            radius="6px"
          />
        </Grid.Col>
      </Grid>
      <Grid className="py-5 px-4">
        <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
          <div
            className="rounded-2xl mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
            style={{ height: 'auto' }}
          >
            <div className="mt-4">
              <Table
                data={tableData}
                columns={columns}
                isLoading={isLoading}
                paginationInfo={paginationInfo}
                handlePagination={handlePagination}
                isWithActionButton={false}
              />
            </div>
          </div>
        </Grid.Col>
      </Grid>
      <ViewDetails
        opened={accountInfo?.isModalOpen}
        setOpened={setAccountInfo}
        event={accountInfo?.id}
        onStatusChange={() => handleFetchDataByFilter()}
      />
    </Paper>
  );
};

export default Gigs;
