import { Grid, Paper, Select } from '@mantine/core';
import {
  IconCalendar,
  IconChevronDown,
  IconCirclePlus,
  IconEye,
  IconPencil,
} from '@tabler/icons-react';
import img from '../../assets/images/rocket (1).png';

import { useEffect, useMemo, useState } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { Button, Table } from '../../components';
import { filterStyle, textAreaStyle } from '../../theme/common.style';
import {
  paginationInfoValue,
  SORT_OPTIONS,
} from '../../utils/common/constant.objects';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';

import { FiTrash } from 'react-icons/fi';
import { RiFlag2Fill, RiFlag2Line } from 'react-icons/ri';
import { useSelector } from 'react-redux';
import { deleteData, fetchData, putData } from '../../api/api';
import { DebouncedTextInput } from '../../components/DebouncedTextInput';
import useScreenSize from '../../hooks/useScreenSize';
import { getTranslation } from '../../layout/languages';
import {
  extractPageInfo,
  formatDate,
  handleError,
  handleSetParams,
  handleSuccess,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import ViewDetails from './event-view';
import { SearchFilter } from './initial.values';

import { DatePickerInput } from '@mantine/dates';
import BoostListingDetails from './boost-listing.modal';
import ViewPaymentDetails from './view-paymentDetails';
const UpcomingEvents = () => {
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const handlePagination = (actionType: string, value?: any) => {
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  };
  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  //const [boostingInfo, setBoostingInfo] = useState(false);
  const { isSmallScreen } = useScreenSize();
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);
  const navigate = useNavigate();
  const [boostingInfo, setBoostingInfo] = useState({
    isModalOpen: false,
    event: {},
  });
  const [eventInfo, seteventInfo] = useState({
    isModalOpen: false,
    event: {},
  });
  const [paymentInfo, setpaymentInfo] = useState({
    isModalOpen: false,
    event: {},
  });
  const handleCreateEvent = () => {
    navigate('/upcomingEvents/create-event');
  };
  const { userInfo } = useSelector((state: any) => state?.userInfo);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) =>
      state.userInfo?.preferences?.currency?.preferredCurrencyId ?? '1'
  );
  const currentRole = userInfo?.roleId;

  useEffect(() => {
    handleFetchDataByFilter();
  }, [currency]);

  const refetch = () => {
    handleFetchDataByFilter();
  };

  const handleEditClick = (id: string) => {
    navigate(`/upcomingEvents/create-event?eventId=${id}`);
  };

  const handleViewClick = (event: any) => {
    if (currentRole === '1') {
      seteventInfo({
        isModalOpen: true,
        event,
      });
    } else {
      navigate(`/upcomingEvents/view-details/${event.id}`);
    }
  };

  const initialSearchValues = {
    searchValue: '',
    type: '',
    limit: '',
    sort: '',
    searchValue1: new Date().toISOString(),
    searchValue3: new Date(
      new Date().setDate(new Date().getDate() + 7)
    ).toISOString(),
  };
  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;
    const searchValue1 =
      searchParams.get('searchValue1') ?? initialSearchValues.searchValue1;
    const searchValue3 =
      searchParams.get('searchValue3') ?? initialSearchValues.searchValue3;
    return {
      searchValue,
      type,
      limit,
      sort,
      searchValue1,
      searchValue3,
    };
  };
  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );

  const [isLoading, setIsLoading] = useState(false);

  // State for table data10
  const [tableData, setTableData] = useState([]);

  // State for reset button
  const [resetTable, setResetTable] = useState(false);
  const handleFetchDataByFilter = () => {
    setIsLoading(true);

    const baseFilters = [
      {
        field: 'event.title',
        operator: 'like',
        value: searchValues.searchValue,
      },
      {
        field: 'event.status',
        operator: 'eq',
        value: searchValues.type,
      },
    ];

    const filters = [...baseFilters]; // Start with baseFilters

    if (currentRole === '3') {
      filters.push({
        field: 'event.organizedById',
        operator: 'eq',
        value: userInfo.id,
      });
    }

    if (currentRole === '2') {
      filters.push({
        field: 'event.startDateTime',
        operator: 'gte',
        value: new Date().toISOString(),
      });
    }

    if (currentRole === '1') {
      if (searchValues.searchValue1) {
        filters.push({
          field: 'event.createdAt',
          operator: 'gte',
          value: new Date(
            new Date(searchValues.searchValue1).setDate(
              new Date(searchValues.searchValue1).getDate() - 1
            )
          ).toISOString(),
        });
      }

      if (searchValues.searchValue3) {
        filters.push({
          field: 'event.createdAt',
          operator: 'lte',
          value: new Date(
            new Date(searchValues.searchValue3).setDate(
              new Date(searchValues.searchValue3).getDate() + 2
            )
          ).toISOString(),
        });
      }
    }

    const finalFilters = removeEmptyValueFilters(filters);

    const filterObject = JSON.stringify({ filter: finalFilters });
    const queryParams = new URLSearchParams({
      rpp: paginationInfo.rowPerPage.toString(),
      page: (paginationInfo.currentPage === 0
        ? 1
        : paginationInfo.currentPage
      ).toString(),
      filter: filterObject,
    });

    if (searchValues.sort) {
      queryParams.append(
        'orderBy',
        JSON.stringify({
          title: searchValues.sort,
          displayOrderId: searchValues.sort,
        })
      );
    }

    const fetchUrl = `events${currentRole === '2' ? `/forEntertainer/${userInfo?.id}` : ''}?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        if (!response.data || response.data.length === 0) {
          setTableData(response.data);
          if (response.total > 0) {
            setPaginationInfo({
              ...paginationInfo,
              totalRecords: response.total,
            });
          }
        } else {
          setTableData(response.data);
          const getPages = extractPageInfo(response.pages);
          setPaginationInfo({
            ...paginationInfo,
            totalRecords: response.total,
            totalPages: getPages?.totalPages ?? 0,
          });
        }
      })
      .catch(error => {
        handleError(error, 'Failed to fetch table data');
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  const handleSearchButtonClick = () => {
    const filteredSearchValues =
      currentRole === 1
        ? searchValues
        : {
            ...searchValues,
            searchValue1: undefined,
            searchValue3: undefined,
          };

    handleSetParams(
      searchParams,
      filteredSearchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };

  // Effect for handling search button click

  useEffect(() => {
    if (currentRole) {
      handleSearchButtonClick();
    }
  }, [
    currentRole,
    resetTable,
    searchValues.searchValue,
    searchValues.type,
    searchValues.limit,
    searchValues.sort,
    searchValues.searchValue1,
    searchValues.searchValue3,
    paginationInfo?.currentPage,
    paginationInfo?.rowPerPage,
  ]);

  // Effect to update state when searchParams change
  useEffect(() => {
    const newSearchValues = initializeStateFromQueryParams();
    if (JSON.stringify(newSearchValues) !== JSON.stringify(searchValues)) {
      setSearchValues(newSearchValues);
    }
  }, [searchParams]);

  const handleDeleteById = async (id: string) => {
    try {
      setIsLoading(true);
      await deleteData(`/events/${id}`);

      handleSuccess('Event is deleted successfully');
      setResetTable(prev => !prev);
    } catch (error: any) {
      handleError(error);
    } finally {
      setIsLoading(false);
    }
  };

  ////////////////////////////////
  const restrictedRoles = ['3'];

  const handleBookmarkToggle = (eventId: string, status: boolean) => {
    putData(`events/${eventId}`, {
      isBookedMarked: status,
    })
      .then(() => {
        setResetTable(prev => !prev);
        // Optional success notification
        handleSuccess(`Bookmark ${status ? 'added' : 'removed'} successfully.`);
      })
      .catch(error => {
        // Optional error notification
        handleError(error);
      });
  };

  const columns = [
    {
      header: (
        <div className="flex text-start font-normal font-productsans text-[14px] rtl:mr-4">
          {getTranslation(language, 'event', 'Order ID')}
        </div>
      ),
      accessorKey: 'displayOrderId',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start  rtl:mr-3 rtl:ml-0 flex-row ml-2">
            <p className="text-start font-normal font-productsans text-[14px]">
              {rowObject?.displayOrderId}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-center items-center font-normal ml-3 font-productsans text-[14px]">
          {getTranslation(language, 'event', 'Events')}
        </div>
      ),
      accessorKey: 'event',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex items-center text-start">
            {rowObject?.eventImage ? (
              <img
                src={rowObject.eventImage}
                alt={rowObject?.title || 'Event Image'}
                className="rounded-md w-9 h-6 object-cover flex-shrink-0 rtl:ml-6 rtl:-mr-4"
                loading="lazy"
              />
            ) : (
              <div className="rounded-md w-9 h-6 bg-gray-300 flex items-center justify-center text-xs font-semibold uppercase flex-shrink-0 rtl:ml-6 rtl:-mr-4">
                {rowObject?.title?.[0] || 'E'}
              </div>
            )}

            <span className="ml-2 rtl:ml-0 rtl:mr-2 font-productsans font-normal text-[14px] truncate max-w-[160px]">
              {rowObject?.title || 'N/A'}
            </span>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-start ml-3 rtl:mr-4 rtl:ml-0 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'event', 'Location')}
        </div>
      ),
      accessorKey: 'location',
      cell: (info: any) => {
        const row = info?.row?.original ?? {};
        const city = row.city?.trim();
        const country = row.country?.trim();
        const venue = row.venue?.trim();

        const location =
          [venue, city, country].filter(Boolean).join(', ') || 'N/A';

        return (
          <div
            className={` ${isSmallScreen ? 'flex text-start truncate' : 'flex text-start'}`}
          >
            <p className="text-start font-productsans font-normal text-[14px]">
              {location}
            </p>
          </div>
        );
      },
    },
  ];

  if (restrictedRoles.includes(currentRole)) {
    columns.push({
      header: (
        <div className="flex text-start items-start font-normal font-productsans text-[14px] rtl:mr-6">
          {getTranslation(language, 'event', 'Entertainers Payment')}
        </div>
      ),
      accessorKey: 'gigs',
      cell: (info: any) => {
        const rowObject = info?.row?.original;

        return (
          <div className="flex flex-col text-start gap-1 rtl:mr-4">
            <div
              className="font-productsans text-sm font-normal"
              onClick={() => {
                setpaymentInfo({
                  isModalOpen: true,
                  event: rowObject,
                });
              }}
            >
              View Payment Details
            </div>
          </div>
        );
      },
    });
  }

  // Helper functions
  function getStatusColor(status: string) {
    switch (status.toLowerCase()) {
      case 'pending':
        return '#FF7900';
      case 'accepted':
        return '#20E2E4';
      case 'rejected':
        return '#FF3D00';
      case 'completed':
        return '#4CAF50';
      default:
        return '#9E9E9E';
    }
  }

  function formatStatus(status: string) {
    return status.charAt(0).toUpperCase() + status.slice(1).toLowerCase();
  }

  columns.push({
    header: (
      <div className="flex text-center items-center ml-3 rtl:mr-4 rtl:ml-0 font-normal font-productsans text-[14px]">
        {getTranslation(language, 'event', 'Date')}
      </div>
    ),
    accessorKey: 'date',
    cell: (info: any) => {
      const rowObject = info?.row?.original;
      return (
        <div className="flex text-start">
          <p className="text-start font-productsans font-normal text-[14px]">
            {formatDate(rowObject?.startDateTime?.substring(0, 10) || '')}
          </p>
        </div>
      );
    },
  });

  if (currentRole === '1' || restrictedRoles.includes(currentRole)) {
    columns.push({
      header: (
        <div className="flex text-center items-center ml-3 rtl:mr-3 rtl:ml-0 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'event', 'Status')}
        </div>
      ),
      accessorKey: 'status',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div
            className="flex text-start"
            style={{
              color: rowObject?.status === 'Pending' ? '#FF7900' : '#20E2E4',
            }}
          >
            <p
              className="text-start font-productsans font-normal text-[14px] ]"
              style={{
                color: rowObject?.status === 'Pending' ? '#FF7900' : '#20E2E4',
              }}
            >
              {rowObject?.status === 'Pending' && rowObject?.isApproved
                ? 'Approved'
                : rowObject?.status}
            </p>
          </div>
        );
      },
    });
  } else {
    columns.splice(3, 0, {
      header: (
        <div className="flex text-start ml-2 rtl:mr-5 rtl:ml-0 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'event', 'Speakers')}
        </div>
      ),
      accessorKey: 'clientName',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start rtl:mr-4 ">
            <p className="text-start font-productsans font-normal text-[14px]">
              {rowObject?.entertainers?.length
                ? rowObject.entertainers
                    .map(
                      (entertainer: any) => entertainer.fullNameOrBusinessName
                    )
                    .join(', ')
                : 'N/A'}
            </p>
          </div>
        );
      },
    });
  }

  columns.push(
    {
      header: (
        <div className="flex text-center items-center ml-3 rtl:mr-4 rtl:ml-0 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'event', 'Time')}
        </div>
      ),
      accessorKey: 'time',
      cell: (info: any) => {
        const rowObject = info?.row?.original;

        return (
          <div className="flex text-start">
            <p className="text-start font-productsans font-normal text-[14px]">
              {rowObject?.startDateTime
                ? new Date(rowObject.startDateTime).toLocaleTimeString(
                    'en-US',
                    {
                      hour: '2-digit',
                      minute: '2-digit',
                      hour12: true,
                    }
                  )
                : ''}
              {' - '}
              {rowObject?.endDateTime
                ? new Date(rowObject.endDateTime).toLocaleTimeString('en-US', {
                    hour: '2-digit',
                    minute: '2-digit',
                    hour12: true,
                  })
                : ''}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex text-start font-normal font-productsans text-[14px] rtl:mr-4">
          {'Boost Detail'}
        </div>
      ),
      accessorKey: 'boostDetail',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div>
            {rowObject?.isBoosted ? (
              <img
                src={img}
                alt="Boosted"
                className="w-9 h-9 ml-4"
                onClick={() => {
                  setBoostingInfo({
                    isModalOpen: true,
                    event: rowObject,
                  });
                }}
              />
            ) : (
              <div
                style={{
                  display: 'inline-block',
                  borderRadius: '28px',
                  padding: '4px 14px',
                  fontSize: '12px',
                  fontWeight: 500,
                  cursor: 'pointer',
                  backgroundColor: '#D83F87',
                  color: 'white',
                  whiteSpace: 'normal',
                  maxWidth: '100%',
                  textAlign: 'center',
                }}
                onClick={() => {
                  if (currentRole === '1') {
                    setBoostingInfo({
                      isModalOpen: true,
                      event: rowObject,
                    });
                  }
                }}
              >
                {getTranslation(language, 'dashboard', 'Not Boosted')}
              </div>
            )}
          </div>
        );
      },
    },
    {
      header: (
        <div className="flex items-end justify-end mr-16 rtl:ml-20 rtl:mr-0 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'event', 'Action')}
        </div>
      ),
      accessorKey: 'action',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        const { isBookedMarked, id } = rowObject;
        return (
          <div
            className={` ${isSmallScreen ? 'flex items-center justify-center font-semibold  mt-2' : 'flex items-center justify-center font-semibold ml-3 -mt-2 -mb-2'}`}
          >
            <div className="flex flex-row justify-between">
              <IconEye
                onClick={() => handleViewClick(rowObject)}
                className="align-middle select-none bg-transparent text-center transition-all cursor-pointer rtl:ml-3 rtl:mr-0 mr-3"
                size={22}
              />
              <div className="border-r-[1px] border-r-caption border-color-light h-5 mt-0.5"></div>
              {currentRole === '3' && (
                <>
                  <IconPencil
                    onClick={() => handleEditClick(rowObject?.id)}
                    className="ml-3 mr-2 mt-1 align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                    size={15}
                  />
                  <div className="border-r-[1px] border-r-caption border-color-light h-5 mt-0.5"></div>
                </>
              )}
              <FiTrash
                onClick={() => handleDeleteById(rowObject?.id)}
                className="ml-3 mr-2 mt-1 align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                size={15}
              />

              {currentRole === '1' && (
                <>
                  <div className="border-r-[1px] border-r-caption border-color-light h-5 mt-0.5"></div>

                  {isBookedMarked ? (
                    <RiFlag2Fill
                      className="ml-2 mr-2 mt-1 cursor-pointer transition-all"
                      size={18}
                      title="Remove Bookmark"
                      onClick={() => handleBookmarkToggle(id, false)}
                    />
                  ) : (
                    <RiFlag2Line
                      className="ml-2 mr-2 mt-1 cursor-pointer transition-all"
                      size={18}
                      title="Add Bookmark"
                      onClick={() => handleBookmarkToggle(id, true)}
                    />
                  )}
                </>
              )}
            </div>
          </div>
        );
      },
    }
  );
  const commoncolumns = useMemo(() => {
    return [...columns];
  }, [resetTable, tableData, language, currency]);

  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById, // Merge the new values with the existing state
    }));
  };

  return (
    <>
      <Paper
        className="flex justify-between items-center p-4 min-h-screen "
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 mb-4">
          <Grid.Col
            span={{
              base: 12,
              md: 12,
              lg: currentRole === '1' || currentRole === '3' ? 4 : 4.5,
            }}
          >
            {currentRole === '2' && (
              <div className=" text-[33px] font-semibold text-[#ffffff] font-productsans">
                {getTranslation(language, 'navbar', 'Upcoming Events')}
              </div>
            )}
            {currentRole !== '2' && (
              <div className=" text-[33px] font-semibold text-[#ffffff] font-productsans">
                {getTranslation(language, 'event', 'Events')}
              </div>
            )}
          </Grid.Col>

          <Grid.Col
            span={{
              base: 12,
              md: 3,
              lg: currentRole === '1' || currentRole === '3' ? 2 : 2.5,
            }}
          >
            <DebouncedTextInput
              placeholder={getTranslation(
                language,
                'event',
                'Search By Event Title'
              )}
              value={searchValues.searchValue ?? ''}
              onDebouncedChange={e =>
                setValuesById({
                  searchValue: e || '',
                })
              }
              styles={textAreaStyle}
              classNames={{ input: 'text-input' }}
              radius="6px"
            />
          </Grid.Col>
          <Grid.Col
            span={{
              base: 12,
              md: 3,
              lg: currentRole === '1' || currentRole === '3' ? 2 : 2.5,
            }}
          >
            <Select
              placeholder={getTranslation(language, 'event', 'Select Status')}
              data={[
                'Pending',
                'Waiting for Entertainer',
                'Pending Payment',
                'Rejected',
                'Live',
                'Ended',
              ]}
              rightSection={<IconChevronDown size={16} color="white" />}
              value={searchValues.type ?? ''}
              onChange={e =>
                setValuesById({
                  type: e ?? '',
                })
              }
              styles={filterStyle}
              classNames={{ input: 'custom-input' }}
              radius="6px"
            />
          </Grid.Col>

          <Grid.Col
            span={{
              base: 12,
              md: 3,
              lg: currentRole === '1' || currentRole === '3' ? 2 : 2.5,
            }}
          >
            <Select
              placeholder={getTranslation(language, 'event', 'Sort By')}
              data={SORT_OPTIONS}
              value={searchValues.sort ?? ''}
              rightSection={<IconChevronDown size={16} color="white" />}
              onChange={value =>
                value
                  ? setValuesById({
                      sort: value,
                    })
                  : setValuesById({
                      sort: '',
                    })
              }
              styles={filterStyle}
              radius="6px"
              classNames={{ input: 'custom-input' }}
            />
          </Grid.Col>
          {currentRole === '1' && (
            <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
              <DatePickerInput
                type="range"
                placeholder="Pick date range"
                value={[
                  searchValues?.searchValue1
                    ? new Date(searchValues.searchValue1)
                    : null,
                  searchValues?.searchValue3
                    ? new Date(searchValues.searchValue3)
                    : null,
                ]}
                onChange={(value: any) => {
                  if (value) {
                    setValuesById({
                      searchValue1: value[0] ? value[0].toISOString() : '',
                      searchValue3: value[1] ? value[1].toISOString() : '',
                    });
                  }
                }}
                valueFormat="M/D/YY"
                styles={{
                  input: {
                    backgroundColor: 'transparent',
                    borderColor: '#FFFFFF',
                    color: '#ffffff',
                    fontFamily: 'Product Sans',
                    fontSize: '12px',
                    height: '48px',
                    borderRadius: '5px',
                  },
                  placeholder: {
                    color: 'white',
                    fontWeight: 'normal',
                  },
                }}
                rightSection={<IconChevronDown size={16} color="#ffffff" />}
                leftSection={<IconCalendar size={20} color="#ffffff" />}
              />
            </Grid.Col>
          )}
          {currentRole === '3' && (
            <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
              <div className="mb-3">
                <Button
                  autoContrast
                  size="sm"
                  styles={{
                    root: {
                      background: '#D83F87',
                    },
                  }}
                  style={{
                    height: 48,
                    width: '100%',
                  }}
                  className="px-2"
                  onClick={handleCreateEvent}
                >
                  <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                    <IconCirclePlus size={17} />
                    <div className="ml-1 mt-0.5">
                      {getTranslation(language, 'event', 'Create Event')}
                    </div>
                  </div>
                </Button>
              </div>
            </Grid.Col>
          )}
        </Grid>
        <Grid className="py-5 px-4">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl  mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
              style={{ height: 'auto' }}
            >
              <div className="mt-4">
                <Table
                  data={tableData}
                  columns={commoncolumns}
                  isLoading={isLoading}
                  paginationInfo={paginationInfo}
                  handlePagination={handlePagination}
                />
              </div>
            </div>
          </Grid.Col>
        </Grid>
        <ViewDetails
          opened={eventInfo?.isModalOpen}
          setOpened={(value: boolean) =>
            seteventInfo(prev => ({ ...prev, isModalOpen: value }))
          }
          eventData={eventInfo?.event}
          refetch={refetch}
        />

        <ViewPaymentDetails
          opened={paymentInfo?.isModalOpen}
          setOpened={setpaymentInfo}
          eventData={paymentInfo?.event}
        />
        <BoostListingDetails
          opened={boostingInfo?.isModalOpen}
          setOpened={setBoostingInfo}
          eventData={boostingInfo?.event}
        />
      </Paper>
    </>
  );
};

export default UpcomingEvents;
