import { Box, Button, Modal, Stack, Textarea, TextInput } from '@mantine/core';
import { Typography } from '@mui/material';
import Rating from '@mui/material/Rating';
import { IconStar } from '@tabler/icons-react';
import { useState } from 'react';
import img from '../../assets/images/send.gif';
import paymentimg from '../../assets/images/payment.gif';
import { inputStyle2 } from '../../theme/common.style';
import { postData } from '../../api/api';
import { handleError, handleSuccess } from '../../utils/common/function';
import { useSelector } from 'react-redux';
import { useNavigate } from 'react-router-dom';

const SuccessModal = ({
  opened,
  setOpened,
  isPublished = false,
  isPaymentRelease = false,
  isReport = false,
  eventId,
  againstUserId,
  setResetTable,
}: any) => {
  const [feedbackOpened, setFeedbackOpened] = useState(false);
  const [feedback, setFeedback] = useState('');
  const [rating, setRating] = useState<number | null>(0);
  const [businessName, setBusinessName] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { userInfo } = useSelector((state: any) => state?.userInfo);
  const navigate = useNavigate();

  const handleSubmitFeedback = async () => {
    setIsSubmitting(true);
    try {
      await postData('feedback/business', {
        comment: feedback,
        rating: rating,
        entertainerId: againstUserId,
        businessId: userInfo?.id,
        eventId,
      });

      handleSuccess('Feedback submitted successfully!');
      setFeedbackOpened(false);

      setFeedback('');
      setRating(0);
      setBusinessName('');
      setTimeout(() => {
        setOpened(false);
        setResetTable();
      }, 3000);
    } catch (error) {
      handleError(error);
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <>
      {/* Success Modal */}
      <Modal
        opened={opened}
        centered
        size="sm"
        onClose={() => setOpened(false)}
        withCloseButton={false}
        styles={{
          content: {
            borderRadius: '15px',
            backgroundColor: '#452975',
            padding: 0,
          },
          body: { padding: 0 },
          inner: { padding: 0, margin: 0, maxWidth: '100vw' },
        }}
        overlayProps={{
          backgroundOpacity: 0.55,
          blur: 3,
        }}
        transitionProps={{ transition: 'fade-up', duration: 300 }}
        title=""
      >
        <div className="mt-8 pb-6">
          <div className="flex justify-center">
            <div className="bg-[#A46FFE] rounded-full w-20 h-20">
              {isPaymentRelease ? (
                <img
                  src={paymentimg}
                  alt="Success"
                  className="w-16 h-16 mt-2 ml-2"
                />
              ) : (
                <img src={img} alt="Success" className="w-16 h-16 mt-2 ml-2" />
              )}
            </div>
          </div>
          <div className="text-[20px] text-[#fff] font-productsans font-semibold text-center mt-4">
            {isPublished ? 'Successfully Published' : 'Successfully Sent!'}
          </div>
          <div className="text-[14px] text-[#BBACD7] font-productsans text-center px-8 mt-2">
            {isPublished
              ? 'Your event has been successfully published.'
              : isPaymentRelease
                ? 'The payment has been released to the entertainer'
                : isReport
                  ? 'Your report has been successfully submitted.'
                  : 'The gig has been successfully offered and the payment has been made.'}
            <div className="mt-4 flex justify-center">
              <Button
                size="sm"
                styles={{
                  root: {
                    backgroundColor: '#D83F87',
                    color: '#ffffff',
                    padding: '10px 20px',
                    width: '100%',
                    height: '60%',
                  },
                }}
                onClick={() => {
                  setOpened(false);
                  if (isPublished) {
                    navigate('/upcomingEvents');
                  }
                }}
              >
                Close
              </Button>
            </div>
          </div>

          {/* Only show buttons if not a report */}
          {!isPublished && !isReport && (
            <div className="px-2 mr-2 mb-3 mt-4 space-y-3">
              {isPaymentRelease && (
                <div className="px-2 mr-2 mb-3">
                  <Button
                    autoContrast
                    m={5}
                    size="sm"
                    styles={{
                      root: {
                        background: '#D83F87',
                      },
                    }}
                    style={{
                      height: 50,
                      width: '100%',
                    }}
                    className="px-2"
                    type="submit"
                    onClick={() => {
                      setFeedbackOpened(true);
                    }}
                  >
                    <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                      FeedBack
                    </div>
                  </Button>
                </div>
              )}
            </div>
          )}
        </div>
      </Modal>

      {/* Feedback Modal - Only shown for payment release */}
      {!isPublished && !isReport && (
        <Modal
          opened={feedbackOpened}
          centered
          size="sm"
          onClose={() => setFeedbackOpened(false)}
          withCloseButton={false}
          styles={{
            content: {
              borderRadius: '15px',
              backgroundColor: '#452975',
              padding: 0,
            },
            body: { padding: 0 },
            inner: { padding: 0, margin: 0, maxWidth: '100vw' },
          }}
          overlayProps={{
            backgroundOpacity: 0.55,
            blur: 3,
          }}
          transitionProps={{ transition: 'fade-up', duration: 300 }}
          title=""
        >
          <div className="relative w-full flex flex-row justify-between">
            <div className="mt-4 text-[#ffffff] font-productsans ml-4 text-[18px]">
              Entertainer Feedback
            </div>
            <Modal.CloseButton
              onClick={() => setFeedbackOpened(false)}
              style={{
                position: 'absolute',
                top: 10,
                right: 10,
                border: 'none',
                outline: 'none',
                boxShadow: 'none',
                color: '#ffffff',
                backgroundColor: 'transparent',
                transition: 'background-color 0.2s ease-in-out',
                width: 33,
                height: 33,
              }}
            />
          </div>

          <div className="mt-8 px-6">
            <TextInput
              label="Business Name"
              className="custom-input px-2 py-2"
              value={businessName}
              onChange={e => setBusinessName(e.currentTarget.value)}
              styles={inputStyle2}
            />

            <div className="my-4 px-2 flex flex-row gap-2">
              <div className="text-[#fff] font-productsans mb-2">
                Rating & Reviews
              </div>
              <Stack gap={2}>
                <Rating
                  name="feedback-rating"
                  value={rating}
                  size="small"
                  getLabelText={value => `${value}`}
                  onChange={(_, newValue) => {
                    setRating(newValue);
                  }}
                  IconContainerComponent={({ value, ...props }) => (
                    <Box
                      {...props}
                      style={{
                        border: '1px solid white',
                        borderRadius: '4px',
                        padding: '3px',
                        display: 'flex',
                        flexDirection: 'row',
                        alignItems: 'center',
                        justifyContent: 'center',
                        gap: 'px',
                        width: '40px',
                        height: '30px',
                        marginRight: '8px',
                        backgroundColor:
                          rating && value <= rating ? '#D83F87' : 'transparent',
                        color: 'white',
                      }}
                    >
                      <Typography variant="body1">{value}</Typography>
                      <IconStar fontSize="inherit" size={20} />
                    </Box>
                  )}
                  emptyIcon={
                    <Box
                      style={{
                        border: '1px solid white',
                        borderRadius: '4px',
                        padding: '6px',
                        display: 'flex',
                        flexDirection: 'row',
                        alignItems: 'center',
                        justifyContent: 'center',
                        gap: '6px',
                        width: '50px',
                        height: '40px',
                        backgroundColor: 'transparent',
                        color: 'white',
                      }}
                    >
                      <Typography variant="body1">0</Typography>
                    </Box>
                  }
                />
              </Stack>
            </div>

            <Textarea
              label="Feedback"
              className="custom-input px-2 py-2"
              value={feedback}
              onChange={e => setFeedback(e.currentTarget.value)}
              autosize
              minRows={4}
              styles={inputStyle2}
            />

            <div className="px-2 mr-2 mb-3">
              <Button
                autoContrast
                m={5}
                size="sm"
                styles={{
                  root: {
                    background: '#D83F87',
                  },
                }}
                style={{
                  height: 50,
                  width: '100%',
                }}
                className="px-2"
                onClick={handleSubmitFeedback}
                disabled={
                  !feedback.trim() ||
                  !businessName.trim() ||
                  rating === 0 ||
                  isSubmitting
                }
                loading={isSubmitting}
              >
                <div className="font-productsans font-normal text-[14px] text-[#ffffff]">
                  {isSubmitting ? 'Submitting...' : 'Submit Feedback'}
                </div>
              </Button>
            </div>
          </div>
        </Modal>
      )}
    </>
  );
};

export default SuccessModal;
