import {
  Button,
  Loader,
  Modal,
  NumberInput,
  Select,
  Textarea,
  TextInput,
} from '@mantine/core';
import { IconChevronDown } from '@tabler/icons-react';
import { useFormik } from 'formik';
import { ReactNode, useEffect, useState } from 'react';
import { useSelector } from 'react-redux';
import { useParams } from 'react-router-dom';
import * as Yup from 'yup';
import { fetchData, postData, putData } from '../../api/api';
import { inputStyle2, inputStyle3 } from '../../theme/common.style';
import { initialNotification } from '../../utils/common/constant.objects';
import { handleError, handleSuccess } from '../../utils/common/function';
import SuccessModal from './success';

const ReportEvents = ({
  opened,
  setOpened,
  eventId,
  againstUserId,
  notificationId,
  setResetTable,
  gigId,
}: any) => {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isFetchingData, setIsFetchingData] = useState(false);
  const [isSuccessModal, setIsSuccessModal] = useState(false);

  const { userInfo } = useSelector((state: any) => state?.userInfo);
  const { id: routeId } = useParams();

  // Determine which ID to use (route param takes precedence over prop)
  const effectiveEventId = routeId || eventId;

  const formik = useFormik({
    enableReinitialize: true,
    initialValues: {
      entertainerId: userInfo?.id,
      eventId: effectiveEventId,
      name: '',
      reason: '',
      description: '',
      isResolved: false,
    },
    validationSchema: Yup.object().shape({
      name: Yup.string().required('Name is required'),
      description: Yup.string().required('Description is required'),
      reason: Yup.string().required('Reason is required'),
    }),
    onSubmit: async values => {
      setIsSubmitting(true);

      try {
        if (eventId) {
          await postData('disputes/', {
            name: values.name,
            description: values.description,
            disputeReason: values.reason,
            raisedById: userInfo?.id,
            againstUserId,
            eventId: values.eventId,
            gigId,
          });
          await putData(
            `notifications/${notificationId}/mark-as-read/${userInfo.id}`,
            {}
          );
          handleSuccess('Dispute Reported Successfully');
          setIsSuccessModal(true);
          setTimeout(() => {
            setIsSuccessModal(false);
            setResetTable();
          }, 3000);
        } else {
          await postData('/report-event', values);
          handleSuccess('Reported Event Successfully');
        }

        setTimeout(() => {
          setOpened(false);
        }, 1000);
      } catch (error: any) {
        handleError(error);
      } finally {
        setIsSubmitting(false);
      }
    },
  });

  const resetAndClose = () => {
    formik.resetForm();
    setOpened(false);
  };

  const handleClose = () => {
    resetAndClose();
  };

  useEffect(() => {
    if (!effectiveEventId) return;

    setIsFetchingData(true);

    fetchData(`/events/${effectiveEventId}`)
      .then((response: any) => {
        if (response?.id) {
          formik.setFieldValue('eventId', response.id);
          if (eventId)
            formik.setFieldValue(
              'name',
              response?.organizedBy?.fullNameOrBusinessName
            );
        }
      })
      .catch(error => {
        handleError(error, 'Failed to fetch event data');
      })
      .finally(() => {
        setIsFetchingData(false);
      });
  }, [effectiveEventId]);

  if (isFetchingData) {
    return (
      <div className="flex justify-center items-center h-64">
        <Loader size="xl" />
      </div>
    );
  }

  return (
    <>
      <Modal
        opened={opened}
        centered
        size="md"
        onClose={() => handleClose()}
        withCloseButton={false}
        styles={{
          content: {
            borderRadius: '15px',
            backgroundColor: '#452975',
            padding: 0,
          },
          body: {
            padding: 0,
          },
          inner: {
            padding: 0,
            margin: 0,
            maxWidth: '100vw',
          },
        }}
        overlayProps={{
          backgroundOpacity: 0.55,
          blur: 3,
        }}
        transitionProps={{ transition: 'fade-up', duration: 300 }}
        title=""
      >
        <div className="relative w-full flex flex-row justify-between">
          <div className="mt-4 text-[#ffffff] font-productsans ml-4 text-[18px]">
            {eventId ? 'Service Not Delivered' : 'Report Events'}
          </div>
          <Modal.CloseButton
            onClick={handleClose}
            style={{
              position: 'absolute',
              top: 10,
              right: 10,
              border: 'none',
              outline: 'none',
              boxShadow: 'none',
              color: '#ffffff',
              backgroundColor: 'transparent',
              transition: 'background-color 0.2s ease-in-out',
              width: 33,
              height: 33,
            }}
          />
        </div>

        <div className="mt-8 ">
          <form onSubmit={formik.handleSubmit}>
            <TextInput
              className="custom-input px-4 py-4"
              label={eventId ? 'Business Name' : 'Your Name'}
              disabled={eventId || isFetchingData}
              placeholder="Enter your name"
              value={formik.values?.name}
              onChange={e => formik.setFieldValue('name', e.target.value)}
              error={
                formik.errors.name &&
                (formik.touched.name || formik.submitCount > 0)
                  ? formik.errors.name
                  : null
              }
              radius="8px"
              styles={inputStyle2}
            />
            <NumberInput
              className="custom-input px-4 py-4"
              label="Event ID"
              placeholder="Enter event ID"
              hideControls
              value={
                formik.values?.eventId
                  ? String(formik.values.eventId).padStart(8, '0')
                  : ''
              }
              onChange={value => {
                const numericValue = Number(value);
                formik.setFieldValue(
                  'eventId',
                  isNaN(numericValue) ? '' : numericValue
                );
              }}
              error={
                formik.errors.eventId &&
                (formik.touched.eventId || formik.submitCount > 0)
                  ? (formik.errors.eventId as ReactNode)
                  : null
              }
              radius="8px"
              styles={inputStyle2}
              disabled
            />
            <Select
              className="custom-input px-4 py-4"
              label="Dispute Reason"
              placeholder="Enter reason"
              rightSection={<IconChevronDown size={20} color="white" />}
              radius="8px"
              styles={inputStyle2}
              data={
                eventId
                  ? ['Service not delivered']
                  : ['Payment did not receive']
              }
              onChange={value => formik.setFieldValue('reason', value)}
              value={formik.values?.reason}
              error={
                formik.errors.reason &&
                (formik.touched.reason || formik.submitCount > 0)
                  ? formik.errors.reason
                  : null
              }
              disabled={isFetchingData}
            />
            <Textarea
              className="custom-input px-4 py-4"
              label="Description"
              placeholder="Enter Your Issue Here"
              radius="8px"
              styles={inputStyle3}
              onChange={e =>
                formik.setFieldValue('description', e.target.value)
              }
              value={formik.values?.description}
              error={
                formik.errors.description &&
                (formik.touched.description || formik.submitCount > 0)
                  ? formik.errors.description
                  : null
              }
              disabled={isFetchingData}
            />
          </form>
          <div className="px-2 mr-2 mb-3">
            <Button
              autoContrast
              m={5}
              size="sm"
              styles={{
                root: {
                  background: '#D83F87',
                },
              }}
              style={{
                height: 50,
                width: '100%',
              }}
              className="px-2"
              type="submit"
              onClick={() => {
                formik.handleSubmit();
              }}
              disabled={isSubmitting || isFetchingData || !effectiveEventId}
              loading={isSubmitting}
            >
              <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                {isSubmitting ? 'Submitting...' : 'Send Report'}
              </div>
            </Button>
          </div>
        </div>
      </Modal>

      {/* Success Modal - only shown when eventId is present */}
      {eventId && (
        <SuccessModal
          opened={isSuccessModal}
          setOpened={setIsSuccessModal}
          isReport
        />
      )}
    </>
  );
};

export default ReportEvents;
