import {
  Button,
  Modal,
  NumberInput,
  Textarea,
  useMantineTheme,
} from '@mantine/core';

import { inputStyle2, inputStyle3 } from '../../theme/common.style';

import { useEffect, useState } from 'react';
import SuccessModal from './success';
import { fetchData, postData } from '../../api/api';
import * as Yup from 'yup';
import { useFormik } from 'formik';
import { useNavigate, useParams } from 'react-router-dom';
import { initialNotification } from '../../utils/common/constant.objects';

import { handleError, handleSuccess } from '../../utils/common/function';
const OfferGigs = ({ opened, setOpened, eventid }: any) => {
  const theme = useMantineTheme();
  const [accountInfo, setaccountInfo] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [notification, setNotification] = useState(initialNotification);
  const [userData, setUserData] = useState<any>();
  const { id } = useParams();
  const navigate = useNavigate();
  useEffect(() => {
    if (!id) return;

    setIsLoading(true);

    fetchData(`/users/${id}`)
      .then((response: any) => {
        setUserData(response);
      })
      .catch((error: any) => {
        handleError(error, 'Failed to fetch user data');
      })
      .finally(() => {
        setIsLoading(false);
      });
  }, [id]);

  const formik = useFormik({
    enableReinitialize: true,
    initialValues: {
      entertainerId: userData?.id ?? '',
      eventId: eventid,
      currencyId: userData?.currencyId ?? '',
      proposedPerformanceFee: '',
      paymentType: 'Wallet',
      message: '',
    },
    validationSchema: Yup.object().shape({
      proposedPerformanceFee: Yup.string().required('Fee is required'),

      message: Yup.string().required('message is required'),
    }),
    onSubmit: values => {
      setIsLoading(true);

      postData('/gig', values)
        .then(() => {
          handleSuccess('Gig added successfully');

          setTimeout(() => {
            setOpened(false);

            navigate(-1);
          }, 1000);
        })
        .catch((error: any) => {
          if (
            error?.response?.data?.message?.message ===
            'Unexpected database error: Transaction failed: Insufficient balance'
          ) {
            setNotification({
              ...notification,
              message: error?.response?.data?.message,
            });
            handleError(
              'You have not enough balance, please add balance in your wallet to proceed'
            );
          } else {
            handleError(error);
          }
        })
        .finally(() => {
          setIsLoading(false);
          setOpened(false);
        });
    },
  });

  const resetAndClose = () => {
    formik.resetForm();
    setOpened(false);
  };

  const handleClose = () => {
    resetAndClose();
  };

  return (
    <Modal
      opened={opened}
      centered
      size="md"
      onClose={() => setOpened(false)}
      withCloseButton={false}
      closeOnClickOutside={false} // prevents clicking outside from closing
      closeOnEscape={false}
      styles={{
        content: {
          borderRadius: '15px',
          backgroundColor: '#452975',
          padding: 0, // Remove default padding
        },
        body: {
          padding: 0, // Ensure no padding inside modal body
        },
        inner: {
          padding: 0, // Ensures modal wrapper has no padding
          margin: 0,
          maxWidth: '100vw',
        },
      }}
      overlayProps={{
        backgroundOpacity: 0.55,
        blur: 3,
      }}
      transitionProps={{ transition: 'fade-up', duration: 300 }}
      title=""
    >
      <div className="relative w-full flex flex-row justify-between">
        <div className="mt-4 text-[#ffffff] font-productsans ml-4  text-[18px]">
          Entertainer Offers
        </div>
        <Modal.CloseButton
          onClick={() => setOpened(false)}
          style={{
            position: 'absolute',
            top: 10,
            right: 10,
            border: 'none',
            outline: 'none',
            boxShadow: 'none',
            color: '#ffffff',
            backgroundColor: 'transparent', // Ensure initial state is transparent
            transition: 'background-color 0.2s ease-in-out',
            width: 33,
            height: 33,
          }}
        />
      </div>
      <div className="mt-8 ">
        <form onSubmit={formik.handleSubmit}>
          <NumberInput
            className="custom-input px-4 py-4"
            label="Proposed Performance Fee"
            hideControls
            placeholder="Enter Your Price Offer 250-350$"
            value={formik.values?.proposedPerformanceFee}
            onChange={value =>
              formik.setFieldValue('proposedPerformanceFee', value)
            }
            error={
              formik.errors.proposedPerformanceFee &&
              (formik.touched.proposedPerformanceFee || formik.submitCount > 0)
                ? formik.errors.proposedPerformanceFee
                : null
            }
            radius="8px"
            styles={inputStyle2}
          />

          <Textarea
            className="custom-input px-4 py-4"
            label="Message"
            placeholder="Enter Your Message For Entertainer"
            value={formik.values?.message}
            onChange={e => formik.setFieldValue('message', e.target.value)}
            error={
              formik.errors.message &&
              (formik.touched.message || formik.submitCount > 0)
                ? formik.errors.message
                : null
            }
            radius="8px"
            styles={inputStyle3}
          />
        </form>

        <div className="px-2 mr-2 mb-3">
          <Button
            autoContrast
            m={5}
            size="sm"
            styles={{
              root: {
                background: '#D83F87',
              },
            }}
            style={{
              height: 50,
              width: '100%',
            }}
            className="px-2"
            onClick={() => {
              setaccountInfo(false);

              formik.handleSubmit();
            }}
            disabled={isLoading}
          >
            <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
              {isLoading ? 'Loading' : 'Send Offer'}
            </div>
          </Button>
        </div>
      </div>
      <SuccessModal opened={accountInfo} setOpened={setaccountInfo} />
    </Modal>
  );
};

export default OfferGigs;
