import {
  GoogleMap,
  LoadScript,
  Autocomplete,
  Marker,
} from '@react-google-maps/api';
import { useRef, useState } from 'react';

interface GoogleMapSearchProps {
  onSelect: (data: { address: string; lat: number; lng: number }) => void;
}

function GoogleMapSearch({ onSelect }: GoogleMapSearchProps) {
  const [map, setMap] = useState<google.maps.Map | null>(null);
  const [marker, setMarker] = useState<{ lat: number; lng: number } | null>(
    null
  );
  const [selectedAddress, setSelectedAddress] = useState('');
  const [selectedLatLng, setSelectedLatLng] = useState<{
    lat: number;
    lng: number;
  } | null>(null);
  const autoRef = useRef<google.maps.places.Autocomplete | null>(null);

  const onPlaceChanged = () => {
    const place = autoRef.current?.getPlace();
    if (place && place.geometry?.location && place.formatted_address) {
      const lat = place.geometry.location.lat();
      const lng = place.geometry.location.lng();
      setMarker({ lat, lng });
      setSelectedAddress(place.formatted_address);
      setSelectedLatLng({ lat, lng });
      map?.panTo({ lat, lng });
    }
  };

  const onMapClick = async (e: google.maps.MapMouseEvent) => {
    if (!e.latLng) return;

    const lat = e.latLng.lat();
    const lng = e.latLng.lng();
    setMarker({ lat, lng });
    setSelectedLatLng({ lat, lng });
    map?.panTo({ lat, lng });

    const geocoder = new google.maps.Geocoder();
    const result = await geocoder.geocode({ location: { lat, lng } });

    if (result.results.length > 0) {
      setSelectedAddress(result.results[0].formatted_address || '');
    }
  };

  const handleSave = () => {
    if (selectedAddress && selectedLatLng) {
      onSelect({
        address: selectedAddress,
        lat: selectedLatLng.lat,
        lng: selectedLatLng.lng,
      });
    }
  };

  const apiKey = process.env.REACT_APP_GOOGLE_MAP_KEY;

  return (
    <LoadScript googleMapsApiKey={apiKey!} libraries={['places']}>
      <div className="space-y-4">
        {/* Search Box */}
        <Autocomplete
          onLoad={(ref: any) => (autoRef.current = ref)}
          onPlaceChanged={onPlaceChanged}
        >
          <input
            type="text"
            placeholder="Search location"
            className="w-full p-3 border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
        </Autocomplete>

        {/* Map */}
        <div className="rounded-xl overflow-hidden shadow-lg border border-gray-200">
          <GoogleMap
            mapContainerStyle={{ width: '100%', height: '400px' }}
            center={marker || { lat: 26.2235, lng: 50.5876 }}
            zoom={12}
            onLoad={(m: any) => setMap(m)}
            onClick={onMapClick}
          >
            {marker && <Marker position={marker} />}
          </GoogleMap>
        </div>

        {/* Address + Save Button */}
        {selectedAddress && (
          <div className="space-y-3">
            <p className="text-sm text-gray-600">
              <span className="font-semibold text-gray-800">
                Selected Address:
              </span>
              <br />
              {selectedAddress}
            </p>

            <div className="text-right">
              <button
                onClick={handleSave}
                className="px-6 py-2 border border-white text-black rounded-md hover:bg-white hover:text-black transition text-color-black"
              >
                Save Location
              </button>
            </div>
          </div>
        )}
      </div>
    </LoadScript>
  );
}

export default GoogleMapSearch;
