import { Button, Grid, Modal } from '@mantine/core';
import { IconMapPin } from '@tabler/icons-react';
import { useMemo, useState } from 'react';
import { useSelector } from 'react-redux';
import { useNavigate } from 'react-router-dom';
import { putData } from '../../api/api';
import { getTranslation } from '../../layout/languages';
import {
  formatDate,
  handleError,
  handleSuccess,
} from '../../utils/common/function';
import BoostListingDetails from './boost-listing.modal';

interface ViewDetailsProps {
  opened: boolean;
  setOpened: (opened: boolean) => void;
  eventData: any;
  refetch: () => void;
}

const ViewDetails = ({
  opened,
  setOpened,
  eventData,
  refetch,
}: ViewDetailsProps) => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [boostingInfo, setBoostingInfo] = useState<{
    isModalOpen: boolean;
    event: any;
  }>({
    isModalOpen: false,
    event: {},
  });

  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );

  // ==== Derived Values ====
  const priceDisplay = useMemo(() => {
    const prices =
      eventData?.ticketsAndPrices
        ?.map((t: any) => Number(t.price))
        .filter((p: any) => !isNaN(p)) ?? [];

    if (!prices.length) return 'N/A';

    const minPrice = Math.min(...prices).toFixed(2);
    const maxPrice = Math.max(...prices).toFixed(2);
    const currencySymbol = currency === '1' ? '$' : 'BHD ≈';

    const label = getTranslation(language, 'event', 'per piece');

    return minPrice === maxPrice
      ? `${currencySymbol}${minPrice} / ${label}`
      : `${currencySymbol}${minPrice} - ${currencySymbol}${maxPrice} / ${label}`;
  }, [eventData?.ticketsAndPrices, currency, language]);

  // ==== Handlers ====
  const handleAccept = async (eventId: string) => {
    try {
      setLoading(true);
      await putData(`events/${eventId}`, {
        isApproved: true,
        status: 'Approved',
        isCancelled: false,
      });
      handleSuccess('Event approved successfully!');
    } catch (error) {
      console.error('Error approving event:', error);
      handleError(error);
    } finally {
      setOpened(false);
      refetch();
      setLoading(false);
    }
  };

  const handleCancelEvent = async (eventId: string) => {
    try {
      setLoading(true);
      await putData(`events/cancel-event/${eventId}`, {});
      handleSuccess('Event cancelled successfully!');
    } catch (error) {
      console.error('Error cancelling event:', error);
      handleError(error);
    } finally {
      setOpened(false);
      refetch();
      setLoading(false);
    }
  };

  const handleBoostClick = () => {
    setOpened(false);
    setBoostingInfo({ isModalOpen: true, event: eventData });
  };

  // ==== Render ====
  return (
    <>
      <Modal
        opened={opened}
        centered
        size="md"
        onClose={() => setOpened(false)}
        withCloseButton={false}
        styles={{
          content: {
            borderRadius: '15px',
            backgroundColor: '#6D3EBD',
            padding: 0,
          },
          body: { padding: 0 },
          inner: { padding: 0, margin: 0, maxWidth: '100vw' },
        }}
        overlayProps={{ backgroundOpacity: 0.55, blur: 3 }}
        transitionProps={{ transition: 'fade-up', duration: 300 }}
      >
        <div className="relative w-full">
          {eventData?.eventImage ? (
            <img
              src={eventData.eventImage}
              alt="image"
              className="rounded-t-md h-[180px] w-full object-cover"
            />
          ) : (
            <div className="rounded-t-md h-[30px] w-full bg-gray-300" />
          )}
          <Modal.CloseButton
            onClick={() => setOpened(false)}
            style={{
              position: 'absolute',
              top: 10,
              right: 10,
              border: 'none',
              outline: 'none',
              boxShadow: 'none',
              color: '#fff',
              backgroundColor: 'transparent',
              width: 33,
              height: 33,
            }}
          />
        </div>

        <div className="mt-4">
          {/* Status */}
          <div className="flex flex-row justify-between ml-8 mr-6 text-[14px] font-productsans text-[#d5d5e3]">
            <div className="text-[#F4F4FE]">
              {eventData?.isPublished
                ? getTranslation(language, 'event', 'Published')
                : getTranslation(language, 'event', 'Unpublished')}
            </div>
          </div>

          {/* Title + Ticket price */}
          <div className="flex flex-row justify-between ml-8 mr-6">
            <div className="text-[18px] font-productsans text-[#F4F4FE]">
              {eventData?.title}
            </div>
            <div className="text-[14px] font-productsans text-[#F4F4FE] mt-2">
              {getTranslation(language, 'event', 'Ticket Price')}
            </div>
          </div>

          {/* Location + Prices */}
          <div className="flex flex-row justify-between ml-8 mr-6">
            <div className="text-[14px] font-productsans text-[#F4F4FE] flex flex-row justify-start">
              <span className="mr-1 mt-0.5">
                <IconMapPin size={17} />
              </span>
              <p>
                {[eventData?.venue, eventData?.city, eventData?.country]
                  .filter(Boolean)
                  .join(', ') || 'N/A'}
              </p>
            </div>
            <div className="text-[18px] font-productsans text-[#F4F4FE] ml-32">
              {priceDisplay}
            </div>
          </div>

          {/* Info Box */}
          <div className="bg-[#ffffff] w-auto bg-opacity-10 rounded-lg mb-3 ml-2 mr-2">
            <div className="flex flex-row justify-between mt-4 text-[#ffffff] ml-1 mr-1 text-[13px] font-productsans">
              <div className="flex flex-col justify-between">
                <p className="mt-3 ml-4">
                  {getTranslation(language, 'event', 'Event Type')}
                </p>
                <p className="mb-2 ml-4">{eventData?.category?.name}</p>
              </div>
              <div className="border-r border-color-light mt-4 mb-3"></div>
              <div className="flex flex-col justify-between">
                <p className="mt-3">
                  {getTranslation(language, 'event', 'Date')}
                </p>
                <p className="mb-2">
                  {formatDate(eventData?.startDateTime?.substring(0, 10) || '')}
                </p>
              </div>
              <div className="border-r border-color-light mt-4 mb-3"></div>
              <div className="flex flex-col justify-between">
                <p className="mt-3 mr-4">
                  {getTranslation(language, 'event', 'Event Size')}
                </p>
                <p className="mr-4 mb-2">{eventData?.maxAttendees}</p>
              </div>
            </div>
          </div>

          {/* Description */}
          <div className="text-[#BBACD7] font-productsans text-[14px] px-8 py-3">
            {eventData?.event?.description}
          </div>

          {/* Submitted From */}
          <div className="text-[14px] px-8 py-3 bg-[#3B1E6D] bg-opacity-30">
            <div className="text-[#fff] text-[14px] font-semibold font-productsans mt-3">
              {getTranslation(language, 'event', 'Submitted From')}:
            </div>
            <div className="text-[#BBACD7D9] text-[13px] font-productsans mt-1">
              <p className="text-[#fff]">
                {getTranslation(language, 'event', 'User Name')}:
              </p>
              <p
                className="underline cursor-pointer"
                onClick={() =>
                  navigate('/user/business', {
                    state: { userData: eventData?.organizedBy },
                  })
                }
              >
                {eventData?.organizedBy?.fullNameOrBusinessName}
              </p>
            </div>
          </div>

          {/* Action Buttons */}
          <Grid className="px-2 mb-3 mr-1 mt-2 flex flex-col">
            {/* Approve */}
            {eventData?.isPublished && !eventData?.isApproved && (
              <Button
                m={5}
                size="sm"
                styles={{ root: { background: '#51B74B' } }}
                style={{ height: 42, width: '100%' }}
                className="px-2"
                disabled={loading}
                onClick={() => handleAccept(eventData.id)}
              >
                <div className="font-productsans font-normal text-[14px] text-white">
                  {getTranslation(
                    language,
                    'gig',
                    loading ? 'Loading' : 'Approve'
                  )}
                </div>
              </Button>
            )}

            {/* Boost Listing */}
            {eventData?.isApproved && (
              <Button
                m={5}
                size="sm"
                styles={{
                  root: {
                    background: eventData?.isBoosted ? '#888' : '#CD5F03',
                  },
                }}
                style={{ height: 42, width: '100%' }}
                className="px-4"
                disabled={loading}
                onClick={handleBoostClick}
              >
                <div className="font-productsans font-semibold text-[14px] text-white">
                  {getTranslation(
                    language,
                    'gig',
                    loading
                      ? 'Loading'
                      : eventData?.isBoosted
                        ? 'Boosted'
                        : 'Boost Listing'
                  )}
                </div>
              </Button>
            )}

            {/* Cancel */}
            {eventData?.status !== 'Canceled' && (
              <Button
                m={5}
                size="sm"
                styles={{ root: { background: '#DD3033' } }}
                style={{ height: 42, width: '100%' }}
                className="px-2"
                disabled={loading}
                onClick={() => handleCancelEvent(eventData.id)}
              >
                <div className="font-productsans font-normal text-[14px] text-white">
                  {getTranslation(
                    language,
                    'dashboard',
                    loading ? 'Loading' : 'Cancel Event'
                  )}
                </div>
              </Button>
            )}
          </Grid>
        </div>
      </Modal>

      <BoostListingDetails
        opened={boostingInfo.isModalOpen}
        setOpened={setBoostingInfo}
        eventData={boostingInfo.event}
      />
    </>
  );
};

export default ViewDetails;
