import {
  Avatar,
  Grid,
  Loader,
  Paper,
  ScrollArea,
  Select,
  TextInput,
} from '@mantine/core';
import {
  IconArrowLeft,
  IconChevronDown,
  IconMapPin,
  IconPencil,
  IconPlus,
} from '@tabler/icons-react';

import { useEffect, useMemo, useState } from 'react';
import { useNavigate, useParams, useSearchParams } from 'react-router-dom';
import { Button, Table } from '../../components';
import { filterStyle, textAreaStyle } from '../../theme/common.style';
import {
  AMOUNT_LIMIT_OPTIONS,
  paginationInfoValue,
  SORT_OPTIONS,
} from '../../utils/common/constant.objects';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';

import { useSelector } from 'react-redux';
import { fetchData } from '../../api/api';
import { DebouncedTextInput } from '../../components/DebouncedTextInput';
import useScreenSize from '../../hooks/useScreenSize';
import { getTranslation } from '../../layout/languages';
import { GigStatus, TransactionType } from '../../utils/common/enums';
import {
  formatDate,
  formatNumber,
  handleError,
  organizeDropDownData,
} from '../../utils/common/function';
import CancelBooking from './cancel-booking';
import ReportEvents from './report-event';
const UpcomingEventsDetail = () => {
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  const [accountInfo, setAccountInfo] = useState(false);
  const [bookingInfo, setBookingInfo] = useState(false);

  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const { userInfo } = useSelector((state: any) => state?.userInfo);
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const currentUserRole = userInfo?.roleId;
  const { isSmallScreen, isMediumScreen, isLargeScreen } = useScreenSize();
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);
  const navigate = useNavigate();
  const { id } = useParams();

  const initialSearchValues = {
    searchValue: '',
    type: '',
    limit: '',
    sort: '',
  };
  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const limit = searchParams.get('limit') ?? initialSearchValues.limit;
    const sort = searchParams.get('sort') ?? initialSearchValues.sort;

    return {
      searchValue,
      type,
      limit,
      sort,
    };
  };
  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );
  const [isLoading, setIsLoading] = useState(false);
  const [Data, setData] = useState<any>();
  // State for table data
  const [tableData, setTableData] = useState([]);
  const [gigData, setgigData] = useState([]);
  const [attendeesData, setattendeesData] = useState([]);

  const [entertainerFilters, setEntertainerFilters] = useState({
    searchValue: '',
    type: '',
    limit: '',
    sort: '',
  });

  const [attendeeFilters, setAttendeeFilters] = useState({
    searchValue: '',
    type: '',
    limit: '',
    sort: '',
  });

  const filterEntertainers = (data: any[]) => {
    let filtered = data?.filter(item => {
      const matchesSearch =
        entertainerFilters.searchValue === '' ||
        item.entertainer?.fullNameOrBusinessName
          ?.toLowerCase()
          .includes(entertainerFilters.searchValue.toLowerCase()) ||
        item.entertainer?.email
          ?.toLowerCase()
          .includes(entertainerFilters.searchValue.toLowerCase());

      const matchesType =
        entertainerFilters.type === '' ||
        item.entertainer?.entertainerTypeId === entertainerFilters.type;

      return matchesSearch && matchesType;
    });

    // Sort by price
    if (entertainerFilters.sort === 'ASC') {
      filtered.sort((a, b) =>
        (a.entertainer?.fullNameOrBusinessName || '').localeCompare(
          b.entertainer?.fullNameOrBusinessName || ''
        )
      );
    } else if (entertainerFilters.sort === 'DESC') {
      filtered.sort((a, b) =>
        (b.entertainer?.fullNameOrBusinessName || '').localeCompare(
          a.entertainer?.fullNameOrBusinessName || ''
        )
      );
    }

    // Limit
    if (entertainerFilters.limit) {
      const limit = parseInt(entertainerFilters.limit, 10);
      if (!isNaN(limit)) {
        filtered = filtered.slice(0, limit);
      }
    }

    return filtered;
  };

  const filterAttendees = (data: any[]) => {
    let filtered = data?.filter(item => {
      const matchesSearch =
        attendeeFilters.searchValue === '' ||
        item.fullNameOrBusinessName
          ?.toLowerCase()
          .includes(attendeeFilters.searchValue.toLowerCase()) ||
        item.email
          ?.toLowerCase()
          .includes(attendeeFilters.searchValue.toLowerCase());

      const matchesType =
        attendeeFilters.type === '' ||
        item.tickets?.some(
          (ticket: any) => ticket.ticket?.name === attendeeFilters.type
        );

      return matchesSearch && matchesType;
    });

    // Sort by amount
    if (attendeeFilters.sort === 'ASC') {
      filtered.sort((a, b) =>
        (a.fullNameOrBusinessName || '').localeCompare(
          b.fullNameOrBusinessName || ''
        )
      );
    } else if (attendeeFilters.sort === 'DESC') {
      filtered.sort((a, b) =>
        (b.fullNameOrBusinessName || '').localeCompare(
          a.fullNameOrBusinessName || ''
        )
      );
    }

    // Limit
    if (attendeeFilters.limit) {
      const limit = parseInt(attendeeFilters.limit, 10);
      if (!isNaN(limit)) {
        filtered = filtered.slice(0, limit);
      }
    }

    return filtered;
  };

  const handleEntertainerSearchChange = (value: string) => {
    setEntertainerFilters(prev => ({ ...prev, searchValue: value }));
  };

  const handleEntertainerTypeChange = (value: string | null) => {
    setEntertainerFilters(prev => ({ ...prev, type: value || '' }));
  };

  const handleAttendeeTypeChange = (value: string | null) => {
    setAttendeeFilters(prev => ({ ...prev, type: value || '' }));
  };

  const handleAttendeeSearchChange = (value: string) => {
    setAttendeeFilters(prev => ({ ...prev, searchValue: value }));
  };

  const handleEntertainerSortChange = (value: string | null) => {
    setEntertainerFilters(prev => ({ ...prev, sort: value || '' }));
  };

  const handleEntertainerLimitChange = (value: string | null) => {
    setEntertainerFilters(prev => ({ ...prev, limit: value || '' }));
  };

  const handleAttendeeSortChange = (value: string | null) => {
    setAttendeeFilters(prev => ({ ...prev, sort: value || '' }));
  };

  const handleAttendeeLimitChange = (value: string | null) => {
    setAttendeeFilters(prev => ({ ...prev, limit: value || '' }));
  };

  // For entertainers table
  const filteredEntertainers = filterEntertainers(tableData);

  // For attendees table
  const filteredAttendees = filterAttendees(attendeesData);

  useEffect(() => {
    if (!id) return;

    setIsLoading(true);

    fetchData(`/events/${id}`)
      .then((response: any) => {
        setData(response);

        const gigs = response?.gigs ?? [];
        setTableData(gigs);

        const attendees = response?.attendees ?? [];
        setattendeesData(attendees);
      })
      .catch((error: any) => {
        handleError(error, 'Failed to fetch event data');
      })
      .finally(() => {
        setIsLoading(false);
      });
  }, [id, currency]);

  const AddEntertainer = () => {
    navigate(`/upcomingEvents/${id}/add-entertainer`);
  };
  //////////////////////////////////
  const columns = [
    {
      header: (
        <div
          className={` flex text-start font-normal font-productsans text-[14px] rtl:mr-4 `}
        >
          {getTranslation(language, 'event', 'Entertainer Name')}
        </div>
      ),
      accessorKey: 'entertainers',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start">
            <div>
              <Avatar
                src={rowObject?.entertainer?.profilePicture || null}
                alt="User"
                size={40}
                radius="xl"
                color="gray"
              >
                {rowObject?.entertainer?.fullNameOrBusinessName[0] || ''}
              </Avatar>
            </div>
            <p className="text-start ml-2 mt-2.5 font-productsans font-normal text-[14px]">
              {rowObject?.entertainer?.fullNameOrBusinessName}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div
          className={` ${isSmallScreen ? 'flex text-center items-center font-normal font-productsans text-[14px]' : 'flex text-center items-center font-normal ml-7 rtl:mr-4 rtl:ml-0 font-productsans text-[14px] '}`}
        >
          {getTranslation(language, 'event', 'Entertainer Type')}
        </div>
      ),
      accessorKey: 'type',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div
            className={` ${isSmallScreen ? 'flex text-start -mt-3.5 -ml-4' : 'flex text-start ml-5 rtl:mr-2 rtl:ml-0'}`}
          >
            <p className="text-start font-normal font-productsans text-[14px]">
              {referenceData?.entertainerTypes?.find(
                (e: { id: any }) =>
                  e.id === rowObject?.entertainer?.entertainerTypeId
              )?.name || 'N/A'}
            </p>
          </div>
        );
      },
    },

    {
      header: (
        <div
          className={` ${isSmallScreen ? 'flex text-start  font-normal font-productsans text-[14px]' : 'flex text-start ml-8 rtl:mr-2 rtl:ml-0 font-normal font-productsans text-[14px] '}`}
        >
          {getTranslation(language, 'event', 'Email')}
        </div>
      ),
      accessorKey: 'email',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div
            className={` ${isSmallScreen ? 'flex text-start -ml-4 rtl:-mr-8 rtl:ml-0' : 'flex text-start ml-6'}`}
          >
            <p className="text-start font-productsans font-normal text-[14px]">
              {rowObject?.entertainer?.email}
            </p>
          </div>
        );
      },
    },
  ];

  const restrictedRoles = ['3'];

  if (currentUserRole === '1' || restrictedRoles.includes(currentUserRole)) {
    columns.push(
      {
        header: (
          <div
            className={` ${
              isSmallScreen
                ? 'flex text-start font-normal font-productsans text-[14px]'
                : 'flex text-start ml-9 font-normal font-productsans text-[14px] rtl:mr-2 rtl:ml-0'
            }`}
          >
            {getTranslation(language, 'event', 'Amount')}
          </div>
        ),
        accessorKey: 'amount',
        cell: (info: any) => {
          const rowObject = info?.row?.original;

          const performanceTransaction = rowObject?.transactions?.find(
            (t: any) => t.type === 'Performance Fee'
          );

          const actualAmount =
            currency === '1'
              ? (performanceTransaction?.actualAmount.toFixed(2) ?? 'N/A')
              : (performanceTransaction?.convertedActualAmount.toFixed(3) ??
                'N/A');

          return (
            <div className="flex text-start">
              <p
                className={` ${
                  isSmallScreen
                    ? 'text-start -ml-4 font-productsans font-normal text-[14px]'
                    : 'text-start ml-7 font-productsans font-normal text-[14px]'
                }`}
              >
                {`${currency === '1' ? '$' : 'BHD ≈'} ${formatNumber(actualAmount)}`}
              </p>
            </div>
          );
        },
      },

      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-start  font-normal font-productsans text-[14px]' : 'flex text-start ml-9 font-normal font-productsans text-[14px] '}`}
          >
            {getTranslation(language, 'event', 'Offer Status')}
          </div>
        ),
        accessorKey: 'offerStatus',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const gigStatus = rowObject?.status;
          return (
            <div
              className="flex text-start"
              style={{
                color: gigStatus === GigStatus.PENDING ? '#FF7900' : '#20E2E4',
              }}
            >
              <p
                className={` ${isSmallScreen ? 'text-start -ml-4 font-productsans font-normal text-[14px]' : 'text-start ml-7 font-productsans font-normal text-[14px] '}`}
              >
                {rowObject?.status}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-center items-center font-normal font-productsans text-[14px]' : 'flex text-center items-center font-normal ml-3 font-productsans text-[14px]'}`}
          >
            {getTranslation(language, 'event', 'Payment Status')}
          </div>
        ),
        accessorKey: 'paymentStatus',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const paymentStatus =
            rowObject?.gigs?.[0]?.paymentStatus?.toLowerCase();
          const transaction = rowObject?.transactions?.filter(
            (transaction: any) =>
              transaction?.type === TransactionType.PERFORMANCE_FEE
          )[0];

          return (
            <div className="flex text-start">
              <p
                className={` ${isSmallScreen ? 'text-start -ml-4 font-productsans font-normal text-[14px]' : 'text-start ml-6 font-productsans font-normal text-[14px] '}`}
                style={{
                  color:
                    paymentStatus === GigStatus.PENDING ? '#FF7900' : '#20E2E4',
                }}
              >
                {transaction?.escrowStatus
                  ? transaction?.escrowStatus?.toUpperCase()
                  : 'N/A'}
              </p>
            </div>
          );
        },
      }
    );
  } else {
    columns.push(
      {
        header: (
          <div
            className={` ${isSmallScreen ? 'flex text-start  font-normal font-productsans text-[14px]' : 'flex text-start ml-14 rtl:mr-3 rtl:ml-0 font-normal font-productsans text-[14px] '}`}
          >
            {getTranslation(language, 'event', 'Location')}
          </div>
        ),
        accessorKey: 'location',
        cell: (info: any) => {
          const rowObject = info?.row?.original;

          return (
            <div
              className={` ${isSmallScreen ? 'flex text-start -ml-4' : 'flex text-start ml-12 '}`}
            >
              <p className="text-start  font-productsans font-normal text-[14px]">
                {rowObject?.address || 'N/A'}
              </p>
            </div>
          );
        },
      },

      {
        header: (
          <div className="flex items-start mr-36 rtl:mr-0 rtl:ml-36 justify-end  font-normal  font-productsans text-[14px]">
            {getTranslation(language, 'event', 'Status')}
          </div>
        ),
        accessorKey: 'status',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const gigStatus = rowObject?.status;
          return (
            <div
              className={` ${isSmallScreen ? 'flex text-start -ml-4' : 'flex text-start ml-10 rtl:mr-6 rtl:ml-0'}`}
            >
              <p
                className="flex text-start font-productsans font-normal text-[14px]"
                style={{
                  color:
                    gigStatus === GigStatus.PENDING ? '#FF7900' : '#20E2E4',
                }}
              >
                {gigStatus?.toUpperCase() || 'N/A'}
              </p>
            </div>
          );
        },
      }
    );
  }
  const entertainercolumns = useMemo(() => {
    return [...columns];
  }, [tableData, gigData, language, currency]);

  const commonColumns = [
    {
      header: (
        <div className="flex text-start  font-normal font-productsans text-[14px]">
          {getTranslation(language, 'dashboard', 'Customer Name')}
        </div>
      ),
      accessorKey: 'name',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start">
            <div>
              <Avatar
                src={rowObject?.profilePicture || null}
                alt="User"
                size={40}
                radius="xl"
                color="gray"
              >
                {rowObject?.fullNameOrBusinessName[0] || ''}
              </Avatar>
            </div>
            <p className="text-start ml-2 mt-1.5 font-productsans font-normal text-[14px]">
              {rowObject?.fullNameOrBusinessName}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div
          className={` ${isSmallScreen ? 'flex text-start ml-3 font-normal font-productsans text-[14px]' : 'flex text-start ml-8 font-normal font-productsans text-[14px] '}`}
        >
          {getTranslation(language, 'event', 'Email')}
        </div>
      ),
      accessorKey: 'email',
      cell: (info: any) => {
        const rowObject = info?.row?.original;

        return (
          <div
            className={` ${isSmallScreen ? 'flex text-start -mt-3.5' : 'flex text-start ml-6 '}`}
          >
            <p className="text-start  font-productsans font-normal text-[14px]">
              {rowObject?.email}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div
          className={` ${isSmallScreen ? 'flex text-center items-center font-normal ml-3 font-productsans text-[14px]' : 'flex text-center items-center font-normal ml-7 font-productsans text-[14px] '}`}
        >
          {getTranslation(language, 'dashboard', 'Ticket Type')}
        </div>
      ),
      accessorKey: 'type',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        const ticketNames =
          rowObject?.tickets?.length > 0
            ? (() => {
                const names = rowObject.tickets
                  .map((ticket: any) => ticket?.ticket?.name)
                  .filter(Boolean);

                if (names.length > 5) {
                  const visible = names.slice(0, 5).join(', ');
                  const remaining = names.length - 5;
                  return `${visible} +${remaining} more`;
                }

                return names.join(', ');
              })()
            : 'N/A';

        return (
          <div
            className={` ${isSmallScreen ? 'flex text-start' : 'flex text-start ml-5 '}`}
          >
            {ticketNames || 'N/A'}
          </div>
        );
      },
    },

    {
      header: (
        <div
          className={` ${isSmallScreen ? 'flex text-start ml-3 font-normal font-productsans text-[14px]' : 'flex text-start ml-10 font-normal font-productsans text-[14px] '}`}
        >
          {getTranslation(language, 'event', 'Price')}
        </div>
      ),
      accessorKey: 'price',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        const price =
          rowObject?.tickets?.length > 0
            ? (() => {
                const prices = rowObject.tickets
                  .map((ticket: any) => {
                    const value = ticket?.ticket?.price;
                    return typeof value === 'number'
                      ? currency === '1'
                        ? value.toFixed(2)
                        : value.toFixed(3)
                      : null;
                  })
                  .filter(Boolean);

                if (prices.length > 5) {
                  const visible = prices.slice(0, 5).join(', ');
                  const remaining = prices.length - 5;
                  return `${visible} +${remaining} more`;
                }

                return prices.join(', ');
              })()
            : 'N/A';

        return (
          <div
            className={` ${isSmallScreen ? 'flex text-start ' : 'flex text-start ml-12 '}`}
          >
            <p className="text-start font-productsans font-normal text-[14px]">
              {`${currency === '1' ? '$' : 'BHD ≈'} ${price || 'N/A'}`}
            </p>
          </div>
        );
      },
    },
  ];
  const attendeescolumns = useMemo(() => {
    return [...commonColumns];
  }, [attendeesData, language, currency]);
  return (
    <>
      <Paper
        className="flex justify-between items-center p-4 min-h-screen "
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 mb-5">
          <Grid.Col span={{ base: 6, md: 12, lg: 12 }}>
            <div className="ml-4 text-[26px] font-semibold text-[#ffffff] font-productsans flex flex-row justify-start">
              <span className="rtl:ml-2 rtl:mr-0 mr-2 mt-1">
                <div
                  className={`rounded-md border-none shadow-sm border bg-color-white bg-opacity-5 w-8 h-8  flex items-center justify-center`}
                >
                  <IconArrowLeft
                    className=" align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                    color="#ffffff"
                    onClick={() => navigate(-1)}
                    size={20}
                  />
                </div>
              </span>
              {`${getTranslation(language, 'navbar', 'Upcoming Events')}
 >`}
              <span className="text-[26px] text-[#D83F87] font-normal  ml-2 rtl:mr-2 rtl:ml-0">
                {getTranslation(language, 'event', 'Event Details')}
              </span>
            </div>
          </Grid.Col>
        </Grid>
        <div
          className="rounded-2xl ml-4 mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
          style={{ height: 'auto' }}
        >
          <Grid>
            <Grid.Col span={{ base: 12, md: 12, lg: 6.5 }}>
              <div className="relative w-full px-4 py-4">
                {isLoading ? (
                  <div className="flex items-center justify-center mt-18 w-full h-[256px]">
                    <Loader size="lg" color="#C28CFF" />
                  </div>
                ) : Data?.eventImage ? (
                  <img
                    src={Data?.eventImage}
                    className="rounded-md h-[320px] w-full object-cover"
                    loading="lazy"
                  />
                ) : (
                  <div className="rounded-md h-[320px] text-[24px] w-full object-cover flex items-center justify-center bg-[#ffffff] bg-opacity-20">
                    {Data?.title?.charAt(0) || ''}
                  </div>
                )}
              </div>
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 12, lg: 5.5 }}>
              <div className="mt-8  mr-8">
                <div className="flex flex-row justify-between ml-2 mr-6">
                  <div className="flex flex-col justify-between ml-2 mr-6">
                    <div className="text-[20px] font-productsans font-semibold text-[#F4F4FE]">
                      {Data?.title}
                    </div>
                    <div className="text-[14px] font-productsans text-[#F4F4FE] font-normal flex flex-row justify-start">
                      <span className="mr-1 mt-0.5">
                        <IconMapPin size={17} />
                      </span>
                      <p>
                        {Data?.city && Data?.venue && Data?.country
                          ? `${Data.city},${Data?.venue},  ${Data.country}`
                          : Data?.city
                            ? Data.city
                            : Data?.venue
                              ? Data.venue
                              : Data?.country
                                ? Data.country
                                : 'N/A'}
                      </p>
                    </div>
                  </div>
                  <div className="flex flex-col justify-between">
                    {currentUserRole === '1' && (
                      <div className="text-[14px] font-productsans text-[#F4F4FE] font-normal">
                        Ticket Price
                      </div>
                    )}

                    <div className="text-[16px] font-productsans text-[#F4F4FE] mt-2 bg-[#D83F87] px-3 py-2 rounded-full inline-flex items-center justify-center whitespace-nowrap">
                      {Array.isArray(Data?.ticketsAndPrices) &&
                      Data.ticketsAndPrices.length > 0
                        ? (() => {
                            const prices = Data.ticketsAndPrices
                              .map((ticket: any) => Number(ticket.price))
                              .filter((price: any) => !isNaN(price));

                            if (prices.length === 0) return 'N/A';

                            const minPrice = Math.min(...prices).toFixed(2);
                            const maxPrice = Math.max(...prices).toFixed(2);

                            return minPrice === maxPrice
                              ? `${currency === '1' ? '$' : 'BHD ≈'}${minPrice} / ${getTranslation(
                                  language,
                                  'event',
                                  'per piece'
                                )}`
                              : `${currency === '1' ? '$' : 'BHD ≈'}${minPrice} - ${
                                  currency === '1' ? '$' : 'BHD ≈'
                                }${maxPrice} / ${getTranslation(
                                  language,
                                  'event',
                                  'per piece'
                                )}`;
                          })()
                        : 'N/A'}
                    </div>
                  </div>
                  {currentUserRole === '1' && (
                    <div className="underline font-productsans  flex items-start justify-start mt-4">
                      <IconPencil
                        className="ml-2 mr-2 align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                        size={24}
                      />
                      <div className="text-[18px]">
                        {getTranslation(language, 'event', 'Edit Details')}
                      </div>
                    </div>
                  )}
                </div>

                <div className="bg-[#ffffff] w-auto h-auto bg-opacity-10 rounded-lg mb-3 ml-2 mr-2 ">
                  <div
                    className={`flex flex-row justify-between mt-4 text-[#ffffff] ml-6 mr-6 text-[14px] font-productsans`}
                  >
                    <div className="flex flex-col justify-between">
                      <p className="mt-3 font-semibold">
                        {getTranslation(language, 'event', 'Date')}
                      </p>
                      <p className="mb-2">
                        {Data?.startDateTime &&
                          formatDate(
                            Data?.startDateTime?.substring(0, 10) || ''
                          )}
                      </p>
                    </div>
                    <div className="border-r-[1px] border-r-caption border-color-light mt-4 mb-3"></div>
                    <div className="flex flex-col justify-between">
                      <p className="mt-3 font-semibold">
                        {getTranslation(language, 'event', 'Event Time')}
                      </p>
                      <p className="mb-2">
                        {Data?.startDateTime
                          ? new Date(Data.startDateTime).toLocaleTimeString(
                              'en-US',
                              {
                                hour: '2-digit',
                                minute: '2-digit',
                                hour12: true,
                              }
                            )
                          : ''}
                        {' - '}
                        {Data?.endDateTime
                          ? new Date(Data.endDateTime).toLocaleTimeString(
                              'en-US',
                              {
                                hour: '2-digit',
                                minute: '2-digit',
                                hour12: true,
                              }
                            )
                          : ''}
                      </p>
                    </div>
                  </div>
                </div>
                <div className="text-[#ffffff] font-productsans font-normal text-[16px] px-2 py-3">
                  {Data?.description}
                </div>
                {currentUserRole === '2' && (
                  <Grid className="px-2 mb-3 mr-2 flex flex-row justify-between">
                    <Grid.Col span={{ base: 12, md: 5.5, lg: 5.5 }}>
                      <Button
                        autoContrast
                        m={5}
                        size="sm"
                        styles={{
                          root: {
                            background: '#FF6166',
                          },
                        }}
                        style={{
                          height: 42,
                          width: '100%',
                        }}
                        className="px-2"
                        onClick={() => setBookingInfo(true)}
                      >
                        <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                          {getTranslation(language, 'sales', 'Cancel Booking')}
                        </div>
                      </Button>
                    </Grid.Col>
                    <Grid.Col span={{ base: 12, md: 5.5, lg: 5.5 }}>
                      <Button
                        autoContrast
                        m={5}
                        size="sm"
                        styles={{
                          root: {
                            background: '#D83F87',
                          },
                        }}
                        style={{
                          height: 42,
                          width: '100%',
                        }}
                        className="px-2"
                        onClick={() => setAccountInfo(true)}
                      >
                        <div className="font-productsans font-normal text-[14px] text-[#ffffff] ">
                          {getTranslation(language, 'sales', 'Report Event')}
                        </div>
                      </Button>
                    </Grid.Col>
                  </Grid>
                )}
              </div>
            </Grid.Col>
          </Grid>
          <Grid className="mt-1 mr-7">
            <Grid.Col span={{ base: 12, md: 12, lg: 3.25 }}>
              <div className="text-[#ffffff] font-productsans text-[17px] px-4 py-5 font-semibold">
                {getTranslation(
                  language,
                  'event',
                  'Entertainer / Performer List'
                )}
              </div>
            </Grid.Col>
            {currentUserRole !== '2' && (
              <>
                <Grid.Col span={{ base: 12, md: 3, lg: 1.75 }}>
                  <TextInput
                    placeholder={getTranslation(
                      language,
                      'event',
                      'Search By Name'
                    )}
                    value={entertainerFilters.searchValue}
                    onChange={e =>
                      handleEntertainerSearchChange(e.target.value)
                    }
                    styles={textAreaStyle}
                    classNames={{ input: 'text-input' }}
                    radius="6px"
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.75 }}>
                  <Select
                    placeholder={getTranslation(
                      language,
                      'event',
                      'Entertainer Type'
                    )}
                    data={
                      referenceData?.entertainerTypes?.map((type: any) => ({
                        value: type.id,
                        label: type.name,
                      })) || []
                    }
                    value={entertainerFilters.type}
                    rightSection={<IconChevronDown size={16} color="white" />}
                    onChange={handleEntertainerTypeChange}
                    styles={textAreaStyle}
                    classNames={{ input: 'custom-input' }}
                    radius="6px"
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.75 }}>
                  <Select
                    placeholder={getTranslation(
                      language,
                      'event',
                      'Amount Limit'
                    )}
                    data={AMOUNT_LIMIT_OPTIONS}
                    value={entertainerFilters.limit}
                    rightSection={<IconChevronDown size={16} color="white" />}
                    onChange={handleEntertainerLimitChange}
                    styles={filterStyle}
                    classNames={{ input: 'custom-input' }}
                    radius="6px"
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.75 }}>
                  <Select
                    placeholder={getTranslation(language, 'event', 'Sort By')}
                    data={SORT_OPTIONS}
                    value={entertainerFilters.sort}
                    rightSection={<IconChevronDown size={16} color="white" />}
                    onChange={handleEntertainerSortChange}
                    styles={filterStyle}
                    classNames={{ input: 'custom-input' }}
                    radius="6px"
                  />
                </Grid.Col>

                <Grid.Col span={{ base: 12, md: 3, lg: 1.5 }}>
                  {currentUserRole === '3' && Data?.isApproved && (
                    <div className=" mb-3 mr-4 rtl:mr-0">
                      <Button
                        autoContrast
                        m={5}
                        size="sm"
                        styles={{
                          root: {
                            background: '#27B4B6',
                          },
                        }}
                        style={{
                          height: 42,
                          width: 'auto',
                        }}
                        className="px-2"
                        onClick={() => AddEntertainer()}
                      >
                        <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                          <div className="mr-1 ">
                            {getTranslation(language, 'event', 'Add More')}
                          </div>
                          <IconPlus size={17} />
                        </div>
                      </Button>
                    </div>
                  )}
                </Grid.Col>
              </>
            )}
          </Grid>

          <div className="mt-4 ml-3">
            <Table
              data={filteredEntertainers}
              columns={entertainercolumns}
              isLoading={isLoading}
              isPaginated={false}
              paginationInfo={paginationInfo}
              handlePagination={handlePagination}
              isWithActionButton={false}
            />
          </div>

          {currentUserRole !== '2' && (
            <Grid className="mt-8 mr-7 ">
              <Grid.Col span={{ base: 12, md: 12, lg: 4 }}>
                <div className="text-[#ffffff] font-productsans text-[20px] px-4 py-5 font-semibold">
                  {getTranslation(language, 'event', 'Attendees List')}
                </div>
              </Grid.Col>

              <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
                <DebouncedTextInput
                  placeholder={getTranslation(
                    language,
                    'user',
                    'Search By Name'
                  )}
                  value={attendeeFilters.searchValue}
                  onDebouncedChange={(e: string) =>
                    handleAttendeeSearchChange(e)
                  }
                  styles={textAreaStyle}
                  classNames={{ input: 'text-input' }}
                  radius="6px"
                />
              </Grid.Col>

              <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
                <Select
                  placeholder={getTranslation(
                    language,
                    'sales',
                    'Select Ticket Type'
                  )}
                  data={organizeDropDownData(referenceData?.ticketTypes)}
                  value={attendeeFilters.type}
                  rightSection={<IconChevronDown size={16} color="white" />}
                  onChange={handleAttendeeTypeChange}
                  styles={filterStyle}
                  classNames={{ input: 'custom-input' }}
                  radius="6px"
                />
              </Grid.Col>

              <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
                <Select
                  placeholder={getTranslation(
                    language,
                    'event',
                    'Amount Limit 1-10'
                  )}
                  data={AMOUNT_LIMIT_OPTIONS}
                  value={attendeeFilters.limit}
                  rightSection={<IconChevronDown size={16} color="white" />}
                  onChange={handleAttendeeLimitChange}
                  styles={filterStyle}
                  classNames={{ input: 'custom-input' }}
                  radius="6px"
                />
              </Grid.Col>

              <Grid.Col span={{ base: 12, md: 3, lg: 2 }}>
                <Select
                  placeholder={getTranslation(language, 'event', 'Sort By')}
                  data={SORT_OPTIONS}
                  value={attendeeFilters.sort}
                  rightSection={<IconChevronDown size={16} color="white" />}
                  onChange={handleAttendeeSortChange}
                  styles={filterStyle}
                  classNames={{ input: 'custom-input' }}
                  radius="6px"
                />
              </Grid.Col>
            </Grid>
          )}
          {currentUserRole !== '2' && (
            <>
              {isLargeScreen ? (
                <ScrollArea h={210} scrollbars="y" className="custom-scroll">
                  <div className="mt-4 ml-2">
                    <Table
                      data={filteredAttendees}
                      columns={attendeescolumns}
                      isLoading={isLoading}
                      isPaginated={false}
                      paginationInfo={paginationInfo}
                      handlePagination={handlePagination}
                      isWithActionButton={false}
                    />
                  </div>
                </ScrollArea>
              ) : (
                <div className="mt-4 ml-2">
                  <Table
                    data={filteredAttendees}
                    columns={attendeescolumns}
                    isLoading={isLoading}
                    isPaginated={false}
                    paginationInfo={paginationInfo}
                    handlePagination={handlePagination}
                    isWithActionButton={false}
                  />
                </div>
              )}
            </>
          )}
        </div>
        <ReportEvents opened={accountInfo} setOpened={setAccountInfo} />
        <CancelBooking opened={bookingInfo} setOpened={setBookingInfo} />
      </Paper>
    </>
  );
};

export default UpcomingEventsDetail;
