import {
  ActionIcon,
  FileInput,
  Grid,
  Group,
  Image,
  Modal,
  NumberInput,
  Paper,
  Radio,
  Select,
  Textarea,
  TextInput,
  Tooltip,
} from '@mantine/core';
import {
  IconArrowLeft,
  IconChevronDown,
  IconClock,
  IconCurrentLocation,
  IconPlus,
  IconTrash,
} from '@tabler/icons-react';

import { useEffect, useState } from 'react';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { Button } from '../../components';
import { fileInputStyle, inputStyle4 } from '../../theme/common.style';
import {
  initialNotification,
  paginationInfoValue,
} from '../../utils/common/constant.objects';

import { DateTimePicker } from '@mantine/dates';
import { FieldArray, FormikProvider, useFormik } from 'formik';
import { useSelector } from 'react-redux';
import * as Yup from 'yup';
import { fetchData, postData, putData, uploadFile } from '../../api/api';
import useDisclosure from '../../hooks/useDisclosure';
import { getTranslation } from '../../layout/languages';
import {
  handleError,
  handleSuccess,
  organizeDropDownData,
} from '../../utils/common/function';
import GoogleMapSearch from './google-map';
import StepperComponent from './stepper';

interface TicketPrice {
  ticketName: string;
  price: number;
  quantity: number;
}

interface EventFormValues {
  title: string;
  description: string;
  categoryId: string;
  startDateTime: string;
  endDateTime: string;
  lat: string;
  lng: string;
  venue: string;
  maxAttendees: string | number;
  organizedById: string;
  isPublic: boolean;
  ticketPricing: number;
  eventImage: string;
  ticketsAndPrices: TicketPrice[];
}

const CreateEvent = () => {
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const [searchParams, setSearchParams] = useSearchParams();
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  const [value, setValue] = useState<[Date | null, Date | null]>([null, null]);
  const [isLoading, setIsLoading] = useState(false);
  const [notification, setNotification] = useState(initialNotification);

  const navigate = useNavigate();
  const [opened, { open, close, toggle }] = useDisclosure(false);

  const [activeStep, setActiveStep] = useState(1); // Initial step

  const handleNext = () => {
    setActiveStep(prevStep => prevStep + 1);
    navigate('/upcomingEvents/Search-Entertainer', {
      state: { step: activeStep + 1 },
    });
  };
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const { userInfo } = useSelector((state: any) => state?.userInfo);
  const eventId = searchParams.get('eventId');
  const currentRole = userInfo?.roleId;

  const formik = useFormik({
    enableReinitialize: true,
    initialValues: {
      title: '',
      description: '',
      categoryId: '',
      startDateTime: '',
      endDateTime: '',
      venue: '',
      lat: '',
      lng: '',
      maxAttendees: '',
      organizedById: userInfo?.id,
      isPublic: true,
      ticketPricing: 0,
      eventImage: '',
      ticketsAndPrices: [{ ticketName: '', price: '', quantity: 0 }],
    },

    validationSchema: Yup.object().shape({
      title: Yup.string().required('Name is required'),
      venue: Yup.string().required('Venue is required'),
      description: Yup.string().required('Description is required'),
      maxAttendees: Yup.string().required('Event Size is required'),
      categoryId: Yup.string().required('Category is required'),
      startDateTime: Yup.string().required('Start Date is required'),
      endDateTime: Yup.string().required('End Date is required'),
    }),

    onSubmit: values => {
      setIsLoading(true);

      const endpoint = eventId ? `/events/${eventId}` : '/events';
      const submitAction = eventId ? putData : postData;
      const payload = {
        ...values,
        maxAttendees: Number(values.maxAttendees),
        ticketsAndPrices: values.ticketsAndPrices.map(ticket => ({
          ...ticket,
          price: Number(ticket.price),
        })),
      };
      submitAction(endpoint, payload)
        .then((response: any) => {
          const id = response?.id || eventId;
          handleSuccess(
            eventId
              ? 'Event updated successfully.'
              : 'Event added successfully.'
          );
          if (id) {
            navigate(`/upcomingEvents/Search-Entertainer?eventId=${id}`);
          }
        })
        .catch(handleError)
        .finally(() => setIsLoading(false));
    },
  });

  const handleFileChange = async (file: File | null) => {
    if (!file || !file.type.startsWith('image/')) {
      console.warn('Only image files are allowed.');
      return;
    }

    try {
      const response = await uploadFile<{ url: string }>('file/upload', file);
      if (response?.url) {
        formik.setFieldValue('eventImage', response?.url);
      }
    } catch (error) {
      console.error('Upload failed:', error);
    }
  };

  useEffect(() => {
    const fetchEvent = async () => {
      if (!eventId) return;
      setIsLoading(true);
      try {
        const response = await fetchData<EventFormValues>(`/events/${eventId}`);
        if (response) {
          formik.setValues({
            title: response.title ?? '',
            description: response.description ?? '',
            categoryId: response.categoryId ?? '',
            startDateTime: response.startDateTime ?? '',
            endDateTime: response.endDateTime ?? '',
            venue: response.venue ?? '',
            lat: response.lat ?? '',
            lng: response.lng ?? '',
            maxAttendees: String(response.maxAttendees ?? ''),
            organizedById: response.organizedById ?? userInfo?.id ?? '',
            isPublic: response.isPublic ?? true,
            ticketPricing: response.ticketPricing ?? 0,
            eventImage: response.eventImage ?? '',
            ticketsAndPrices:
              response.ticketsAndPrices?.length > 0
                ? response.ticketsAndPrices.map(ticket => ({
                    ticketName: ticket.ticketName ?? '',
                    price: String(ticket.price ?? 0),
                    quantity: ticket.quantity ?? 0,
                  }))
                : [{ ticketName: '', price: '0', quantity: 0 }],
          });
        }
      } catch (error) {
        handleError(error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchEvent();
  }, [eventId]);

  return (
    <>
      <Paper
        className="flex justify-between items-center p-4  min-h-screen "
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7 mb-5">
          <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
            <div className="ml-4 text-[26px] font-semibold text-[#ffffff] font-productsans flex flex-row justify-start">
              <span className="mr-2 mt-1">
                <div
                  className={`rounded-md border-none shadow-sm border bg-color-white bg-opacity-5 w-8 h-8  flex items-center justify-center`}
                >
                  <IconArrowLeft
                    className=" align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                    onClick={() => navigate(-1)}
                    size={20}
                  />
                </div>
              </span>

              {getTranslation(language, 'event', 'Create Event')}
            </div>
          </Grid.Col>
          <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
            <StepperComponent />
          </Grid.Col>
        </Grid>
        <div
          className="rounded-2xl ml-4 mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
          style={{ height: 'auto' }}
        >
          <FormikProvider value={formik}>
            <form onSubmit={formik.handleSubmit}>
              <Grid>
                <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
                  {formik.values.eventImage ? (
                    <>
                      <Image
                        src={formik.values.eventImage}
                        alt="Preview"
                        style={{
                          height: '300px',
                          cursor: 'pointer',
                          borderRadius: '4px',
                          marginBottom: '16px',
                        }}
                        onClick={() =>
                          document.getElementById('fileInput')?.click()
                        }
                      />
                      <FileInput
                        id="fileInput"
                        placeholder="Change Image"
                        styles={fileInputStyle}
                        className="px-4"
                        onChange={handleFileChange}
                        accept="image/*"
                        style={{ display: 'none' }}
                      />
                    </>
                  ) : (
                    <FileInput
                      placeholder={getTranslation(
                        language,
                        'event',
                        'Upload Your Event Image'
                      )}
                      styles={fileInputStyle}
                      className="px-4"
                      onChange={handleFileChange}
                      accept="image/*"
                    />
                  )}
                </Grid.Col>
              </Grid>
              <Grid className="mt-4">
                <Grid.Col span={{ base: 12, md: 6, lg: 4 }}>
                  <TextInput
                    className="custom-input ml-4 py-4"
                    label={getTranslation(language, 'event', 'Event Name')}
                    placeholder={getTranslation(
                      language,
                      'event',
                      'Enter Your Event Name'
                    )}
                    value={formik.values?.title}
                    onChange={e =>
                      formik.setFieldValue('title', e.target.value)
                    }
                    error={
                      formik.errors.title &&
                      (formik.touched.title || formik.submitCount > 0)
                        ? formik.errors.title
                        : null
                    }
                    radius="8px"
                    styles={inputStyle4}
                  />
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 6, lg: 4 }} className="mt-4">
                  <DateTimePicker
                    minDate={new Date()}
                    valueFormat="DD MMM YYYY hh:mm A"
                    label={getTranslation(
                      language,
                      'event',
                      'Event Start Date & Time'
                    )}
                    radius="8px"
                    value={
                      formik.values?.startDateTime
                        ? new Date(formik.values.startDateTime)
                        : null
                    }
                    onChange={(value: Date | null) => {
                      formik.setFieldValue('startDateTime', value || '');
                    }}
                    placeholder={getTranslation(
                      language,
                      'event',
                      'Start date and time'
                    )}
                    clearable
                    style={inputStyle4}
                    styles={{
                      label: {
                        fontWeight: '400',
                        fontSize: '16px',
                        color: '#ffffff',
                        marginBottom: '0',
                        fontFamily: 'Product Sans',
                        position: 'absolute',
                        top: '-12px',
                        left: '30px',
                        backgroundColor: 'rgba(103, 63, 170, 0.4)',
                        backdropFilter: 'blur(3px)',
                        padding: '0 8px',
                        zIndex: 1,
                      },
                      input: {
                        backgroundColor: 'transparent',
                        color: '#ffffff',
                        border: '1px solid #ffffff',
                        height: '50.84px',
                        paddingLeft: '22px',
                        width: '100%',
                        fontSize: '14px',
                        fontWeight: 'normal',
                        fontFamily: 'Product Sans',
                      },
                      root: {
                        position: 'relative',
                        maxWidth: '100%',
                      },
                    }}
                    rightSection={<IconClock size={20} color="white" />}
                  />
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 6, lg: 4 }} className="mt-4">
                  <DateTimePicker
                    minDate={new Date()}
                    valueFormat="DD MMM YYYY hh:mm A"
                    label={getTranslation(
                      language,
                      'event',
                      'Event End Date & Time'
                    )}
                    className="mr-4"
                    radius="8px"
                    value={
                      formik.values?.endDateTime
                        ? new Date(formik.values.endDateTime)
                        : null
                    }
                    onChange={(value: Date | null) => {
                      formik.setFieldValue('endDateTime', value);
                    }}
                    placeholder={getTranslation(
                      language,
                      'event',
                      'End date and time'
                    )}
                    clearable
                    style={inputStyle4}
                    styles={{
                      label: {
                        fontWeight: '400',
                        fontSize: '16px',
                        color: '#ffffff',
                        marginBottom: '0',
                        fontFamily: 'Product Sans',
                        position: 'absolute',
                        top: '-12px',
                        left: '30px',
                        backgroundColor: 'rgba(103, 63, 170, 0.4)',
                        backdropFilter: 'blur(3px)',
                        padding: '0 8px',
                        zIndex: 1,
                      },
                      input: {
                        backgroundColor: 'transparent',
                        color: '#ffffff',
                        border: '1px solid #ffffff',
                        height: '50.84px',
                        paddingLeft: '22px',
                        width: '100%',
                        fontSize: '14px',
                        fontWeight: 'normal',
                        fontFamily: 'Product Sans',
                      },
                      root: {
                        position: 'relative',
                        maxWidth: '100%',
                      },
                    }}
                    rightSection={<IconClock size={20} color="white" />}
                  />
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 6, lg: 4 }}>
                  <Select
                    className="custom-input ml-4 py-4"
                    label={getTranslation(language, 'event', 'Event Type')}
                    placeholder={getTranslation(
                      language,
                      'event',
                      'Select Your Event Type'
                    )}
                    data={organizeDropDownData(referenceData?.eventCategories)}
                    value={formik.values?.categoryId}
                    onChange={value =>
                      formik.setFieldValue('categoryId', value)
                    }
                    error={
                      formik.errors.categoryId &&
                      (formik.touched.categoryId || formik.submitCount > 0)
                        ? formik.errors.categoryId
                        : null
                    }
                    radius="8px"
                    styles={inputStyle4}
                    rightSection={<IconChevronDown size={20} color="white" />}
                  />
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 6, lg: 4 }}>
                  <TextInput
                    className="custom-input py-4"
                    label={getTranslation(language, 'event', 'Event Location')}
                    placeholder={getTranslation(
                      language,
                      'event',
                      'Enter Your Address'
                    )}
                    value={formik.values?.venue}
                    onChange={e =>
                      formik.setFieldValue('venue', e.target.value)
                    }
                    error={
                      formik.errors.venue &&
                      (formik.touched.venue || formik.submitCount > 0)
                        ? formik.errors.venue
                        : null
                    }
                    radius="8px"
                    styles={inputStyle4}
                    rightSection={
                      <div onClick={open} className="cursor-pointer">
                        <IconCurrentLocation size={20} color="white" />
                      </div>
                    }
                  />
                </Grid.Col>
                <Grid.Col span={{ base: 12, md: 6, lg: 3.85 }}>
                  <NumberInput
                    className="custom-input py-4"
                    label={getTranslation(language, 'event', 'Event Size')}
                    placeholder="50-100"
                    hideControls
                    value={formik.values?.maxAttendees}
                    onChange={value =>
                      formik.setFieldValue('maxAttendees', value)
                    }
                    error={
                      formik.errors.maxAttendees &&
                      (formik.touched.maxAttendees || formik.submitCount > 0)
                        ? formik.errors.maxAttendees
                        : null
                    }
                    radius="8px"
                    styles={inputStyle4}
                  />
                </Grid.Col>
                <FieldArray name="ticketsAndPrices">
                  {({ push, remove }) => (
                    <>
                      {formik.values.ticketsAndPrices?.map((ticket, index) => (
                        <Grid.Col key={index} span={12}>
                          <Grid style={{ marginLeft: '15px' }}>
                            {/* Ticket Type */}
                            <Grid.Col span={{ base: 12, md: 4, lg: 3.5 }}>
                              <Select
                                className="custom-input py-4"
                                label={getTranslation(
                                  language,
                                  'event',
                                  'Ticket Type'
                                )}
                                placeholder={getTranslation(
                                  language,
                                  'event',
                                  'Select Ticket Type'
                                )}
                                data={organizeDropDownData(
                                  referenceData?.ticketTypes
                                )?.map(ele => ele.label)}
                                value={ticket.ticketName || ''}
                                onChange={value =>
                                  formik.setFieldValue(
                                    `ticketsAndPrices.${index}.ticketName`,
                                    value
                                  )
                                }
                                radius="8px"
                                styles={inputStyle4}
                                rightSection={
                                  <IconChevronDown size={20} color="white" />
                                }
                              />
                            </Grid.Col>

                            {/* Ticket Price */}
                            <Grid.Col span={{ base: 12, md: 4, lg: 3.5 }}>
                              <NumberInput
                                className="custom-input ml-1 py-4"
                                label={getTranslation(
                                  language,
                                  'event',
                                  'Ticket Price'
                                )}
                                placeholder={getTranslation(
                                  language,
                                  'event',
                                  'Enter Price'
                                )}
                                hideControls
                                value={ticket.price ?? ''}
                                onChange={value =>
                                  formik.setFieldValue(
                                    `ticketsAndPrices.${index}.price`,
                                    value
                                  )
                                }
                                radius="8px"
                                styles={inputStyle4}
                              />
                            </Grid.Col>

                            {/* Ticket Quantity */}
                            <Grid.Col span={{ base: 12, md: 4, lg: 3 }}>
                              <NumberInput
                                className="custom-input ml-1 py-4"
                                label={getTranslation(
                                  language,
                                  'event',
                                  'Ticket Quantity'
                                )}
                                placeholder={getTranslation(
                                  language,
                                  'event',
                                  'Enter Quantity'
                                )}
                                hideControls
                                value={ticket.quantity ?? ''}
                                onChange={value =>
                                  formik.setFieldValue(
                                    `ticketsAndPrices.${index}.quantity`,
                                    value
                                  )
                                }
                                radius="8px"
                                styles={inputStyle4}
                              />
                            </Grid.Col>

                            {/* Add / Remove Icons */}
                            <Grid.Col
                              span={{ base: 12, md: 12, lg: 2 }}
                              className="flex items-end"
                            >
                              <Group>
                                {formik.values.ticketsAndPrices.length < 3 &&
                                  index ===
                                    formik.values.ticketsAndPrices.length -
                                      1 && (
                                    <Tooltip label="Add Ticket Type">
                                      <ActionIcon
                                        variant="filled"
                                        color="pink"
                                        size="lg"
                                        onClick={() =>
                                          push({
                                            ticketName: '',
                                            price: '',
                                            quantity: '',
                                          })
                                        }
                                        className="ml-2 mb-4"
                                      >
                                        <IconPlus size={20} />
                                      </ActionIcon>
                                    </Tooltip>
                                  )}

                                {formik.values.ticketsAndPrices.length > 1 && (
                                  <Tooltip label="Remove Ticket Type">
                                    <ActionIcon
                                      variant="filled"
                                      color="pink"
                                      size="lg"
                                      onClick={() => remove(index)}
                                      className="mb-4"
                                    >
                                      <IconTrash size={20} />
                                    </ActionIcon>
                                  </Tooltip>
                                )}
                              </Group>
                            </Grid.Col>
                          </Grid>
                        </Grid.Col>
                      ))}
                    </>
                  )}
                </FieldArray>

                <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
                  <Textarea
                    className="custom-input px-4 py-4  Profile-textArea"
                    label={getTranslation(language, 'event', 'Description')}
                    placeholder={getTranslation(
                      language,
                      'event',
                      'Enter Description'
                    )}
                    value={formik.values?.description}
                    onChange={e =>
                      formik.setFieldValue('description', e.target.value)
                    }
                    error={
                      formik.errors.description &&
                      (formik.touched.description || formik.submitCount > 0)
                        ? formik.errors.description
                        : null
                    }
                    radius="8px"
                    styles={inputStyle4}
                    style={{ minHeight: '160 !important' }}
                  />
                </Grid.Col>
                <div className="flex flex-row justify-between">
                  <div className="px-6 py-2 text-[14px] text-[#ffffff]">
                    {getTranslation(language, 'event', 'Event Privacy')}
                  </div>
                  <Radio.Group
                    className="text-[#ffffff] px-6 "
                    value={formik.values.isPublic ? 'public' : 'private'}
                    onChange={value =>
                      formik.setFieldValue('isPublic', value === 'public')
                    }
                  >
                    <Group mt="xs">
                      <Radio
                        value="private"
                        label={getTranslation(language, 'event', 'Private')}
                        styles={theme => ({
                          radio: {
                            backgroundColor: 'transparent',
                            borderColor: '#D83F87', // Default border color
                            '&:checked': {
                              backgroundColor: '#D83F87', // Background when checked
                              borderColor: '#D83F87',
                            },
                          },
                          icon: {
                            color: '#ffffff', // Icon color when checked
                          },
                        })}
                      />
                      <Radio
                        value="public"
                        label={getTranslation(language, 'event', 'Public')}
                        styles={theme => ({
                          radio: {
                            backgroundColor: 'transparent',
                            borderColor: '#D83F87', // Default border color
                            '&:checked': {
                              backgroundColor: '#D83F87', // Background when checked
                              borderColor: '#D83F87',
                            },
                          },
                          icon: {
                            color: '#ffffff', // Icon color when checked
                          },
                        })}
                      />
                    </Group>
                  </Radio.Group>
                </div>
              </Grid>
            </form>
          </FormikProvider>
          <Grid>
            <Grid.Col span={{ base: 12, md: 6, lg: 2 }}>
              <div className="px-2 mr-2 mb-3 mt-2">
                <Button
                  autoContrast
                  m={5}
                  size="sm"
                  styles={{
                    root: {
                      background: '#D83F87',
                    },
                  }}
                  style={{
                    height: 50,
                    width: '100%',
                  }}
                  className="px-2"
                  onClick={() => {
                    //   handleNext();
                    formik.handleSubmit();
                  }}
                  disabled={isLoading}
                >
                  <div className="font-productsans font-semibold text-[14px] text-[#ffffff] ">
                    {getTranslation(
                      language,
                      'event',
                      `${isLoading ? 'Loading' : 'Next'}`
                    )}
                  </div>
                </Button>
              </div>
            </Grid.Col>
          </Grid>
        </div>
      </Paper>
      <Modal
        opened={opened}
        onClose={close}
        title="Select Location"
        size="lg"
        centered
      >
        <GoogleMapSearch
          onSelect={data => {
            formik.setFieldValue('venue', data.address);
            formik.setFieldValue('lat', data.lat);
            formik.setFieldValue('lng', data.lng);
            close();
          }}
        />
      </Modal>
    </>
  );
};

export default CreateEvent;
