import * as React from 'react';
import dayjs, { Dayjs } from 'dayjs';
import Badge from '@mui/material/Badge';
import { AdapterDayjs } from '@mui/x-date-pickers/AdapterDayjs';
import { LocalizationProvider } from '@mui/x-date-pickers/LocalizationProvider';
import { PickersDay, PickersDayProps } from '@mui/x-date-pickers/PickersDay';
import { DateCalendar } from '@mui/x-date-pickers/DateCalendar';
import { DayCalendarSkeleton } from '@mui/x-date-pickers/DayCalendarSkeleton';
import { FaCaretLeft, FaCaretRight } from 'react-icons/fa';
import { fetchData } from '../../api/api';

import isSameOrBefore from 'dayjs/plugin/isSameOrBefore';
import {
  handleError,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { useSelector } from 'react-redux';
import { useParams } from 'react-router-dom';

dayjs.extend(isSameOrBefore);

function ServerDay(
  props: PickersDayProps<Dayjs> & {
    highlightedDays?: { date: string; color: string }[];
  }
) {
  const { highlightedDays = [], day, outsideCurrentMonth, ...other } = props;

  const highlight = highlightedDays.find(highlighted =>
    day.isSame(dayjs(highlighted.date), 'day')
  );

  const isSelected = !outsideCurrentMonth && !!highlight;
  const isOutsideCurrentMonth = outsideCurrentMonth && !!highlight;

  return (
    <Badge key={day.toString()} overlap="circular" badgeContent={null}>
      <PickersDay
        {...other}
        outsideCurrentMonth={outsideCurrentMonth}
        day={day}
        style={{
          position: 'relative',
          backgroundColor: isSelected
            ? highlight?.color
            : isOutsideCurrentMonth
              ? ''
              : '',
          border: isSelected ? `2px solid ${highlight?.color}` : 'none',
          borderRadius: '50%',
          width: '36px',
          height: '36px',
          display: 'flex',
          justifyContent: 'center',
          alignItems: 'center',
        }}
      />
    </Badge>
  );
}

export default function BasicDateCalendar() {
  const [isLoading, setIsLoading] = React.useState(false);
  const [highlightedDays, setHighlightedDays] = React.useState<
    { date: string; color: string }[]
  >([]);

  const { id } = useParams();

  React.useEffect(() => {
    const handleFetchDataByFilter = async () => {
      setIsLoading(true);
      const filters = removeEmptyValueFilters([
        {
          field: 'user_availabilities.userId',
          operator: 'eq',
          value: id,
        },
      ]);

      const filterObject = JSON.stringify({ filter: filters });
      const fetchUrl = `user-availability?filter=${filterObject}`;

      try {
        const response: any = await fetchData(fetchUrl);

        const processedDates: { date: string; color: string }[] = [];

        response.data?.forEach((entry: any) => {
          const from = dayjs(entry.dateFrom);
          const to = dayjs(entry.dateTo);

          const color =
            entry.status === 'BOOKED'
              ? '#D83F87'
              : entry.status === 'UNAVAILABLE'
                ? '#7B7B7B'
                : null;

          if (color) {
            for (
              let d = from;
              d.isSameOrBefore(to, 'day');
              d = d.add(1, 'day')
            ) {
              processedDates.push({
                date: d.format('YYYY-MM-DD'),
                color,
              });
            }
          }
        });

        setHighlightedDays(processedDates);
      } catch (error: any) {
        handleError(error, 'Failed to fetch availability data');
      } finally {
        setIsLoading(false);
      }
    };

    handleFetchDataByFilter();
  }, []);

  return (
    <LocalizationProvider dateAdapter={AdapterDayjs}>
      <DateCalendar
        showDaysOutsideCurrentMonth
        loading={isLoading}
        renderLoading={() => <DayCalendarSkeleton />}
        slotProps={{
          day: {
            highlightedDays,
          } as any,
        }}
        sx={{
          width: '100%',
          maxWidth: '200%',
          flexWrap: 'wrap',
          color: 'white',
          fontFamily: 'Sans-Serif',
          '.MuiTypography-root': {
            color: '#ABADBF',
          },
          '.MuiDayCalendar-weekDayLabel': {
            padding: '4px 8px', // tighter weekday label spacing
            color: 'white',
            gap: '4px',
          },
          '.MuiDayCalendar-weekContainer': {
            display: 'flex',
            justifyContent: 'center',
            gap: '4px', // horizontal gap
            marginBottom: '2px', // reduce vertical spacing between rows
          },
          '.MuiPickersDay-root': {
            color: 'white',
            margin: '1px', // reduce both horizontal & vertical spacing
            width: '32px',
            height: '30px',
            padding: 0,
          },
          '& .MuiPickersDay-dayOutsideMonth': {
            color: '#9E9E9E',
          },
          '.MuiPickersDay-root.Mui-selected': {
            backgroundColor: 'transparent',
            color: 'white',
          },
          '.MuiPickersCalendarHeader-root': {
            color: 'white',
            fontFamily: 'Sans-Serif',
            fontSize: '12px',
            paddingLeft: '16px',
            width: '100%',
            marginBottom: '0px', // reduce space below header
          },
          '.MuiPickersCalendarHeader-label': {
            color: 'white',
            fontSize: '16px',
          },
          '.MuiPickersCalendarHeader-switchViewButton': {
            display: 'none',
          },
        }}
        slots={{
          day: ServerDay,
          leftArrowIcon: (props: any) => (
            <button
              {...props}
              style={{
                color: '#ffffff',
                fontSize: '2rem',
                border: 'none',
                background: 'transparent',
              }}
              className="-mr-3"
            >
              <FaCaretLeft />
            </button>
          ),
          rightArrowIcon: (props: any) => (
            <button
              {...props}
              style={{
                color: '#ffffff',
                fontSize: '2rem',
                border: 'none',
                background: 'transparent',
              }}
              className=""
            >
              <FaCaretRight />
            </button>
          ),
        }}
      />
    </LocalizationProvider>
  );
}
