import { Grid, Loader, Paper, Select, Text, TextInput } from '@mantine/core';
import { DatePickerInput } from '@mantine/dates';
import { Box, Rating, Stack, Typography } from '@mui/material';
import {
  IconArrowLeft,
  IconChevronDown,
  IconHeart,
  IconHeartFilled,
  IconStar,
  IconStarFilled,
} from '@tabler/icons-react';
import { useEffect, useState } from 'react';
import { useSelector } from 'react-redux';
import {
  useLocation,
  useNavigate,
  useParams,
  useSearchParams,
} from 'react-router-dom';
import { fetchData, postData } from '../../api/api';
import img from '../../assets/images/gigoffer.png';
import { Button } from '../../components';
import { DebouncedTextInput } from '../../components/DebouncedTextInput';
import EntertainerImage from '../../components/Image';
import useScreenSize from '../../hooks/useScreenSize';
import { getTranslation } from '../../layout/languages';
import { inputStyle4 } from '../../theme/common.style';
import { paginationInfoValue } from '../../utils/common/constant.objects';
import {
  handleError,
  handleSetParams,
  handleSuccess,
  organizeDropDownData,
  removeEmptyValueFilters,
} from '../../utils/common/function';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import { SearchFilter } from './initial.values';
import StepperComponent from './stepper';
const AddEntertainer = () => {
  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  const [searchParams, setSearchParams] = useSearchParams();
  const initialSearchValues = {
    searchValue: '',
    type: '',
    rate: '',
    location: '',
  };
  const initializeStateFromQueryParams = () => {
    const searchValue =
      searchParams.get('searchValue') ?? initialSearchValues.searchValue;
    const type = searchParams.get('type') ?? initialSearchValues.type;
    const rate = searchParams.get('rate') ?? initialSearchValues.rate;
    const location =
      searchParams.get('location') ?? initialSearchValues.location;

    return {
      searchValue,
      type,
      rate,
      location,
    };
  };
  const [searchValues, setSearchValues] = useState(
    initializeStateFromQueryParams()
  );

  const [value, setValue] = useState<Date | null>(null);

  const [selectedValue, setSelectedValue] = useState<number | null>(null);
  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  const [offeredGigs, setOfferedGigs] = useState([]);
  // State for reset button
  const [resetTable, setResetTable] = useState(false);
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);
  const navigate = useNavigate();
  const { isExtraLargeScreen } = useScreenSize();

  const { id: eventid } = useParams();

  const fetchGigsOffered = () => {
    setIsLoading(true);
    const filters = removeEmptyValueFilters([
      {
        field: 'gig.eventId',
        operator: 'eq',
        value: searchParams.get('eventId'),
      },
    ]);

    const filterObject = JSON.stringify({ filter: filters });
    const queryParams = new URLSearchParams({
      filter: filterObject,
    });

    const fetchUrl = `gig?${queryParams.toString()}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        setOfferedGigs(response?.data);
      })
      .catch(error => console.error(error, 'Failed to find Gigs offered'))
      .finally(() => setIsLoading(false));
  };

  useEffect(() => fetchGigsOffered(), []);

  const handleNavigation = (Id: string) => {
    if (location.pathname === `/upcomingEvents/${eventid}/add-entertainer`) {
      navigate(`/upcomingEvents/${eventid}/entertainer-profile/${Id}`);
    } else if (
      location.pathname === `/upcomingEvents/Search-Entertainer` &&
      new URLSearchParams(location.search).get('eventId') === eventId
    ) {
      navigate(`/upcomingEvents/${eventId}/entertainer-profile/${Id}`);
    } else if (location.pathname.includes('/entertainer')) {
      navigate(`/entertainer/entertainer-profile/${Id}`);
    } else {
      navigate(`/entertainer/entertainer-profile`);
    }
  };

  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const { userInfo } = useSelector((state: any) => state?.userInfo);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currentRole = userInfo?.roleId;
  const [activeButton, setActiveButton] = useState('entertainers');

  const location = useLocation();

  const activeStep = location.state?.step || 1;
  const [isLoading, setIsLoading] = useState(false);

  const eventId = searchParams.get('eventId');
  const handleNext = () => {
    if (location.pathname === `/upcomingEvents/Search-Entertainer` && eventId) {
      navigate(`/upcomingEvents/Event-Summary?eventId=${eventId}`, {
        state: { step: activeStep + 1 },
      });
    }
  };
  const [entertainers, setEntertainers] = useState<any[]>([]);

  const handleFetchDataByFilter = () => {
    setIsLoading(true);

    const filters = removeEmptyValueFilters([
      {
        field: 'users.fullNameOrBusinessName',
        operator: 'like',
        value: searchValues?.searchValue,
      },
      {
        field: 'users.entertainerTypeId',
        operator: 'eq',
        value: searchValues?.type,
      },
      {
        field: 'users.roleId',
        operator: 'eq',
        value: '2',
      },
      {
        field: 'users.ratePerHour',
        operator: 'lte',
        value: searchValues?.rate,
      },
      {
        field: 'users.address',
        operator: 'like',
        value: searchValues?.location,
      },
      ...(selectedValue !== null
        ? [
            {
              field: 'users.rating',
              operator: 'eq',
              value: selectedValue,
            },
          ]
        : []),
      ...(activeButton !== 'favorites'
        ? []
        : [
            {
              field: 'users.favoriteOf',
              operator: 'in',
              value: [userInfo?.id],
            },
          ]),
    ]);

    const filterObject = JSON.stringify({ filter: filters });
    const fetchUrl = `users?filter=${filterObject}`;

    fetchData(fetchUrl)
      .then((response: any) => {
        setEntertainers(response.data);
      })
      .catch((error: any) => {
        handleError(error, 'Failed to fetch entertainers');
      })
      .finally(() => {
        setIsLoading(false);
      });
  };

  const handleResetButtonClick = () => {
    const newParams = new URLSearchParams();
    const rowPerPage =
      searchParams.get('rowPerPage') || paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ||
        paginationInfoValue.currentPage?.toString()
    );
    if (rowPerPage !== '5') newParams.set('rowPerPage', rowPerPage);
    if (currentPage > 1) newParams.set('currentPage', currentPage.toString());
    setSelectedValue(null);
    setSearchParams(newParams);
    setSearchValues(initialSearchValues);
    setPaginationInfo({ ...paginationInfo, currentPage: currentPage });
    setResetTable(!resetTable);
  };

  const handleSearchButtonClick = () => {
    handleSetParams(
      searchParams,
      searchValues,
      initialSearchValues,
      setSearchParams
    );
    handleFetchDataByFilter();
  };

  // Effect for handling search button click
  useEffect(() => {
    handleSearchButtonClick();
  }, [
    resetTable,
    activeButton,
    searchValues.searchValue,
    searchValues.type,
    searchValues.location,
    searchValues.rate,
    selectedValue,
  ]);

  // Effect to update state when searchParams change
  useEffect(() => {
    const newSearchValues = initializeStateFromQueryParams();
    if (JSON.stringify(newSearchValues) !== JSON.stringify(searchValues)) {
      setSearchValues(newSearchValues);
    }
  }, [searchParams]);
  const setValuesById = (valuesById: Partial<SearchFilter>) => {
    setSearchValues(prevFormValues => ({
      ...prevFormValues,
      ...valuesById, // Merge the new values with the existing state
    }));
  };

  const [favoriteStates, setFavoriteStates] = useState<Record<string, boolean>>(
    {}
  );
  const [pressStates, setPressStates] = useState<Record<string, boolean>>({});

  useEffect(() => {
    if (entertainers?.length) {
      const initialStates = entertainers.reduce(
        (acc, ent) => {
          acc[ent.id] = ent.favoriteOf?.includes(userInfo?.id ?? '') || false;
          return acc;
        },
        {} as Record<string, boolean>
      );
      setFavoriteStates(initialStates);
    }
  }, [entertainers, userInfo]);

  return (
    <>
      <Paper
        className="flex justify-between items-center p-4 min-h-screen"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="mt-6 mr-7">
          {location.pathname ===
            `/upcomingEvents/${eventid}/add-entertainer` && (
            <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
              <div className="ml-4 text-[26px] font-semibold text-[#ffffff] font-productsans flex flex-row justify-start">
                <span className="mr-2 mt-1">
                  <div
                    className={`rounded-md border-none shadow-sm border bg-color-white bg-opacity-5 w-8 h-8  flex items-center justify-center`}
                  >
                    <IconArrowLeft
                      className=" align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                      color="#ffffff"
                      onClick={() => navigate(-1)}
                      size={20}
                    />
                  </div>
                </span>

                <div>{` ${getTranslation(
                  language,
                  'dashboard',
                  'Events'
                )} `}</div>
                <div className="ml-2"> {`>`}</div>

                <span className="text-[23px] text-[#ffffff] font-semibold mt-1 ml-2 flex flex-row justify-start">
                  <div>{` ${getTranslation(
                    language,
                    'event',
                    'Event Details'
                  )} `}</div>

                  <div className="ml-2">{`>`}</div>
                </span>
                <span className="text-[23px] text-[#D83F87] font-normal mt-1 ml-2">
                  {getTranslation(language, 'entertainer', 'Add Entertainers')}
                </span>
              </div>
            </Grid.Col>
          )}
          {location.pathname === '/upcomingEvents/Search-Entertainer' && (
            <>
              <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                <div
                  className="ml-4 text-[26px] font-semibold text-[#ffffff] font-productsans flex flex-row justify-start"
                  onClick={handleSearchButtonClick}
                >
                  <span className="mr-2 mt-1">
                    <div
                      className={`rounded-md border-none shadow-sm border bg-color-white bg-opacity-5 w-8 h-8  flex items-center justify-center`}
                    >
                      <IconArrowLeft
                        className=" align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                        color="#ffffff"
                        onClick={() => navigate(-1)}
                        size={20}
                      />
                    </div>
                  </span>
                  {`${getTranslation(language, 'entertainer', 'Search Entertainers')}`}
                </div>
              </Grid.Col>
              <Grid.Col span={{ base: 12, md: 6, lg: 6 }}>
                <StepperComponent />
              </Grid.Col>
            </>
          )}
          {location.pathname === '/entertainer' && (
            <>
              <Grid.Col span={{ base: 12, md: 6, lg: 8 }}>
                <div className="ml-4 text-[26px] font-semibold text-[#ffffff] font-productsans flex flex-row justify-start">
                  <span className="mr-2 rtl:ml-2 rtl:mr-0 mt-1">
                    <div
                      className={`rounded-md border-none shadow-sm border bg-color-white bg-opacity-5 w-8 h-8  flex items-center justify-center`}
                    >
                      <IconArrowLeft
                        className=" align-middle select-none bg-transparent text-center transition-all cursor-pointer"
                        color="#ffffff"
                        onClick={() => navigate(-1)}
                        size={20}
                      />
                    </div>
                  </span>
                  {`${getTranslation(language, 'entertainer', 'Entertainer')}`}
                </div>
              </Grid.Col>
              <Grid.Col span={{ base: 12, md: 6, lg: 4 }}>
                <div className="flex flex-row justify-between">
                  <Button
                    autoContrast
                    m={5}
                    size="sm"
                    onClick={() => setActiveButton('entertainers')}
                    styles={{
                      root: {
                        background:
                          activeButton === 'entertainers'
                            ? '#D83F87'
                            : 'transparent',
                      },
                    }}
                    style={{
                      height: 42,
                      width: '60%',
                      borderRadius: '20px',
                      border: '1.5px solid #624B8A',
                    }}
                    className="rounded-full"
                  >
                    <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start rounded-full">
                      <div className="ml-1 mt-0.5">
                        {getTranslation(
                          language,
                          'entertainer',
                          'Entertainers List'
                        )}
                      </div>
                    </div>
                  </Button>
                  <Button
                    autoContrast
                    m={5}
                    onClick={() => setActiveButton('favorites')}
                    size="sm"
                    styles={{
                      root: {
                        background:
                          activeButton === 'favorites'
                            ? '#D83F87'
                            : 'transparent',
                        borderRadius: '20px',
                        border: '1.5px solid #624B8A',
                      },
                    }}
                    style={{
                      height: 42,
                      width: '70%',
                    }}
                    className=""
                  >
                    <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                      <div className="ml-1 mt-0.5">
                        {getTranslation(
                          language,
                          'entertainer',
                          'Favorites Entertainers'
                        )}
                      </div>
                    </div>
                  </Button>
                </div>
              </Grid.Col>
            </>
          )}
        </Grid>

        <div
          className="rounded-2xl ml-4 mr-2 mt-4 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
          style={{ height: 'auto' }}
        >
          <div className="text-[16px] text-[#ffffff] font-semibold mt-1 ml-2">
            {getTranslation(language, 'entertainer', 'Browse Entertainers')}
          </div>
          <Grid className="mt-4">
            <Grid.Col span={{ base: 12, md: 6, lg: 3.25 }}>
              <DebouncedTextInput
                className="custom-input  py-4"
                label={getTranslation(
                  language,
                  'entertainer',
                  'Search Entertainers'
                )}
                placeholder={getTranslation(language, 'user', 'Search By Name')}
                value={searchValues.searchValue ?? ''}
                onDebouncedChange={val => setValuesById({ searchValue: val })}
                radius="8px"
                styles={inputStyle4}
              />
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 6, lg: 3.25 }}>
              <Select
                className="custom-input  py-4"
                label={getTranslation(
                  language,
                  'entertainer',
                  'Entertainers Type'
                )}
                placeholder={getTranslation(
                  language,
                  'entertainer',
                  'Select Your Entertainers Type'
                )}
                data={organizeDropDownData(referenceData?.entertainerTypes)}
                value={searchValues.type ?? ''}
                onChange={value => setValuesById({ type: value ?? '' })}
                radius="8px"
                styles={inputStyle4}
                rightSection={<IconChevronDown size={18} color="#ffffff" />}
              />
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 6, lg: 3.25 }} className="mt-4">
              <DatePickerInput
                label={getTranslation(language, 'entertainer', 'Availability')}
                placeholder={getTranslation(
                  language,
                  'entertainer',
                  'DD/MM/YY'
                )}
                value={value}
                onChange={setValue}
                style={inputStyle4}
                styles={{
                  label: {
                    fontWeight: '400',
                    fontSize: '16px',
                    color: '#ffffff',
                    marginBottom: '0',
                    fontFamily: 'Product Sans', // Updated font family
                    position: 'absolute',
                    top: '-12px',
                    left: '30px',

                    backgroundColor: 'rgba(103, 63, 170, 0.4)',
                    backdropFilter: 'blur(3px)',
                    padding: '0 8px',
                    zIndex: 1,
                  },
                  input: {
                    backgroundColor: 'transparent',
                    color: '#ffffff',
                    border: '1px solid #ffffff',
                    height: '50.84px',
                    paddingLeft: '22px',
                    width: '100%',
                    fontSize: '14px',
                    fontWeight: 'normal',
                    fontFamily: 'Product Sans', // Updated font family
                  },
                  root: {
                    position: 'relative',
                    maxWidth: '100%',
                  },
                }}
              />
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 6, lg: 2.25 }}>
              <TextInput
                className="custom-input  py-4"
                label={getTranslation(language, 'entertainer', 'Budget')}
                placeholder={getTranslation(
                  language,
                  'entertainer',
                  'In Between 500-2000$'
                )}
                value={searchValues.rate ?? ''}
                onChange={e =>
                  setValuesById({
                    rate: e.target.value ?? '',
                  })
                }
                radius="8px"
                styles={inputStyle4}
              />
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 6, lg: 3 }}>
              <TextInput
                className=" py-4"
                label={getTranslation(language, 'entertainer', 'Location')}
                placeholder={getTranslation(
                  language,
                  'entertainer',
                  'Enter Your Preferred Location'
                )}
                value={searchValues.location ?? ''}
                onChange={e =>
                  setValuesById({
                    location: e.target.value ?? '',
                  })
                }
                radius="8px"
                styles={inputStyle4}
              />
            </Grid.Col>
            <Grid.Col span={{ base: 12, md: 6, lg: 3 }}>
              <div className="font-productsans ">
                {getTranslation(language, 'entertainer', 'Rating and Reviews')}
              </div>
              <Stack spacing={2}>
                <Rating
                  name="size-small"
                  defaultValue={0}
                  size="small"
                  getLabelText={value => `${value}`}
                  onChange={(_, newValue) => setSelectedValue(newValue)}
                  IconContainerComponent={({ value, ...props }) => (
                    <Box
                      {...props}
                      sx={{
                        border: '1px solid white',
                        borderRadius: '4px',
                        padding: '4px',
                        display: 'flex',
                        flexDirection: 'row', // Align items in a row
                        alignItems: 'center',
                        justifyContent: 'center',
                        gap: '10px', // Increased spacing between number and box content
                        width: '60px', // Adjusted width for better spacing
                        height: '40px', // Adjusted height
                        marginRight: '8px', // Increased spacing between boxes
                        backgroundColor:
                          selectedValue && value <= selectedValue
                            ? '#D83F87'
                            : 'transparent', // Apply color only if selected
                        color: 'white',
                      }}
                    >
                      <Typography variant="body1">{value}</Typography>

                      <IconStar fontSize="inherit" size={20} />
                    </Box>
                  )}
                  emptyIcon={
                    <Box
                      sx={{
                        border: '1px solid white',
                        borderRadius: '4px',
                        padding: '6px',
                        display: 'flex',
                        flexDirection: 'row',
                        alignItems: 'center',
                        justifyContent: 'center',
                        gap: '6px',
                        width: '50px',
                        height: '40px',
                        backgroundColor: 'transparent', // Empty box when not selected
                        color: 'white',
                      }}
                    >
                      <Typography variant="body1">0</Typography>
                    </Box>
                  }
                />
              </Stack>
            </Grid.Col>
          </Grid>
          <Grid>
            <Grid.Col span={{ base: 12, md: 6, lg: 3 }}>
              <div className=" mb-3">
                <Button
                  autoContrast
                  m={5}
                  size="sm"
                  styles={{
                    root: {
                      background: '#D83F87',
                    },
                  }}
                  style={{
                    height: 42,
                    width: '100%',
                  }}
                  className=""
                  onClick={handleSearchButtonClick}
                >
                  <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                    <div className="ml-1 mt-0.5">
                      {getTranslation(
                        language,
                        'entertainer',
                        'Search Entertainers'
                      )}
                    </div>
                  </div>
                </Button>
              </div>
            </Grid.Col>
            {location.pathname === '/entertainer' && (
              <Grid.Col span={{ base: 12, md: 6, lg: 2.5 }}>
                <div
                  className="underline test-[14px] font-productsans font-normal flex items-center justify-center mt-3 cursor-pointer "
                  onClick={handleResetButtonClick}
                >
                  {getTranslation(language, 'entertainer', 'Clear All Filters')}
                </div>
              </Grid.Col>
            )}
          </Grid>
          <div className="border-t border-gray-300 my-4 mx-1"></div>
          <div className="ml-4 text-[16px] font-semibold text-[#ffffff] font-productsans flex flex-row justify-start">
            {searchValues.type && (
              <>
                {`${getTranslation(language, 'entertainer', 'FILTERED')} >`}
                <span className="text-[16px] text-[#D83F87] font-semibold ml-2 rtl:mr-2 rtl:ml-0">
                  {
                    referenceData?.entertainerTypes?.find(
                      (e: { id: string }) => e.id === searchValues.type
                    )?.name
                  }
                </span>
              </>
            )}
          </div>

          <Grid
            className="flex flex-row justify-between py-5 px-4 "
            gutter="xl"
          >
            {isLoading ? (
              <div
                className="flex items-center justify-center ml-96 mt-20 mb-9"
                style={{ marginLeft: '45%' }}
              >
                <Loader size="lg" color="#C28CFF" />
              </div>
            ) : entertainers.length === 0 ? (
              <Text
                style={{
                  color: 'white',
                  fontSize: '16px',
                  fontFamily: 'Product Sans',
                  textAlign: 'center',
                  marginTop: '50px',
                  marginLeft: '20px',
                  marginBottom: '10px',
                }}
              >
                {getTranslation(
                  language,
                  'entertainer',
                  'No entertainer found.'
                )}
              </Text>
            ) : (
              entertainers.map((entertainer, index) => (
                <Grid.Col key={index} span={{ base: 12, md: 6, lg: 3.25 }}>
                  <div className="cursor-pointer relative">
                    <div onClick={() => handleNavigation(entertainer?.id)}>
                      <EntertainerImage
                        entertainer={entertainer}
                        className="px-4"
                      />
                      {location.pathname ===
                        '/upcomingEvents/Search-Entertainer' &&
                        offeredGigs?.find(
                          (element: any) =>
                            element.entertainerId === entertainer?.id
                        ) && (
                          <div className="absolute bottom-14 right-4 bg-[#FF7900] p-2 rounded-full shadow-md">
                            <img src={img} style={{ width: 30 }} />
                          </div>
                        )}

                      {location.pathname === '/entertainer' && (
                        <div
                          className={`absolute top-2 right-4 bg-[#ffffff] bg-opacity-20 p-2 rounded-full shadow-md transition-transform ${
                            pressStates[entertainer.id] ? 'press-animation' : ''
                          }`}
                          onClick={e => {
                            e.stopPropagation();

                            const currentState = favoriteStates[entertainer.id];
                            const newState = !currentState;

                            setFavoriteStates(prev => ({
                              ...prev,
                              [entertainer.id]: newState,
                            }));

                            setPressStates(prev => ({
                              ...prev,
                              [entertainer.id]: true,
                            }));
                            setTimeout(() => {
                              setPressStates(prev => ({
                                ...prev,
                                [entertainer.id]: false,
                              }));
                            }, 150);

                            postData(
                              `users/${userInfo?.id}/favorite/${entertainer?.id}`,
                              {}
                            )
                              .then(() => {
                                setResetTable(prev => !prev);
                                handleSuccess(
                                  `Added to ${newState ? 'favorites' : 'unfavorite'} successfully.`
                                );
                              })
                              .catch(error => {
                                console.error(
                                  'Error toggling favorite:',
                                  error
                                );
                                handleError(error);

                                setFavoriteStates(prev => ({
                                  ...prev,
                                  [entertainer.id]: currentState,
                                }));
                              });
                          }}
                        >
                          {favoriteStates[entertainer.id] ? (
                            <IconHeartFilled size={20} color="#ffffff" />
                          ) : (
                            <IconHeart size={20} color="#ffffff" />
                          )}
                        </div>
                      )}
                    </div>
                    <div className="flex flex-row justify-between px-2 py-2">
                      <div>{entertainer?.fullNameOrBusinessName}</div>
                      <div
                        className={`${
                          isExtraLargeScreen
                            ? 'flex flex-row justify-start font-normal mr-9'
                            : 'flex flex-row justify-start font-normal'
                        }`}
                      >
                        <span className="mr-1 mt-0.5">
                          <IconStarFilled size={15} color="#D83F87" />
                        </span>
                        {entertainer?.rating || '0'}
                      </div>
                    </div>
                  </div>
                </Grid.Col>
              ))
            )}
          </Grid>

          <Grid>
            {location.pathname !== '/entertainer' && (
              <Grid.Col span={{ base: 12, md: 6, lg: 2.5 }}>
                <div className="ml-2 mb-3">
                  <Button
                    autoContrast
                    m={5}
                    size="sm"
                    styles={{
                      root: {
                        background: '#D83F87',
                      },
                    }}
                    style={{
                      height: 42,
                      width: '100%',
                    }}
                    className=""
                    onClick={handleNext}
                  >
                    <div className="font-productsans font-normal text-[15px] text-[#ffffff] flex flex-row justify-start">
                      <div className="ml-1 mt-0.5">
                        {getTranslation(language, 'entertainer', 'Next Step')}
                      </div>
                    </div>
                  </Button>
                </div>
              </Grid.Col>
            )}
          </Grid>
        </div>
      </Paper>
    </>
  );
};

export default AddEntertainer;
