// components/ErrorBoundary.tsx
import React, { ErrorInfo, ReactNode } from 'react';
import { Button, Text, Group, Card } from '@mantine/core';
import { IconAlertCircle, IconReload } from '@tabler/icons-react';

interface ErrorBoundaryProps {
  children: ReactNode;
  /**
   * Custom fallback component to render when an error occurs
   */
  fallback?: (error: Error, resetError: () => void) => ReactNode;
  /**
   * Callback when an error is caught
   */
  onError?: (error: Error, errorInfo: ErrorInfo) => void;
  /**
   * Whether to show a reset button
   */
  showReset?: boolean;
}

interface ErrorBoundaryState {
  hasError: boolean;
  error?: Error;
}

class ErrorBoundary extends React.Component<
  ErrorBoundaryProps,
  ErrorBoundaryState
> {
  constructor(props: ErrorBoundaryProps) {
    super(props);
    this.state = { hasError: false };
  }

  static getDerivedStateFromError(error: Error): ErrorBoundaryState {
    return { hasError: true, error };
  }

  componentDidCatch(error: Error, errorInfo: ErrorInfo) {
    console.error('ErrorBoundary caught an error:', error, errorInfo);
    this.props.onError?.(error, errorInfo);

    // You can log errors to an error tracking service here
    // logErrorToService(error, errorInfo);
  }

  resetError = () => {
    this.setState({ hasError: false, error: undefined });
  };

  render() {
    if (this.state.hasError) {
      // Use custom fallback if provided
      if (this.props.fallback) {
        return this.props.fallback(this.state.error!, this.resetError);
      }

      // Default fallback UI
      return (
        <Card
          padding="lg"
          radius="md"
          withBorder
          style={{ maxWidth: 500, margin: '0 auto' }}
        >
          <Group mb="md" align="center">
            <IconAlertCircle size={24} color="red" />
            <Text size="lg" w={500}>
              Something went wrong
            </Text>
          </Group>

          <Text color="dimmed" mb="md">
            {this.state.error?.message || 'An unexpected error occurred'}
          </Text>

          {this.props.showReset !== false && (
            <Button
              leftSection={<IconReload size={16} />}
              onClick={this.resetError}
              variant="light"
            >
              Try again
            </Button>
          )}
        </Card>
      );
    }

    return this.props.children;
  }
}

export default ErrorBoundary;
