import { Box, Flex, Grid, Paper, rem } from '@mantine/core';
import {
  IconBriefcaseFilled,
  IconCalendar,
  IconChevronDown,
  IconDots,
  IconHeartFilled,
} from '@tabler/icons-react';
import { BsFillCalendarEventFill } from 'react-icons/bs';
// import { Button, Table } from '../../concave.agri/components';
import { useEffect, useState } from 'react';
import { useSelector } from 'react-redux';
import { useSearchParams } from 'react-router-dom';
import { Button, Table } from '../../components';
import { getTranslation } from '../../layout/languages';
import { paginationInfoValue } from '../../utils/common/constant.objects';
import { handlePaginationValue } from '../../utils/common/pagination.Helper';
import BasicDateCalendar from './Calendar';

import { DatePickerInput } from '@mantine/dates';
import { postData, putData } from '../../api/api';
import useScreenSize from '../../hooks/useScreenSize';
import { filterStyle } from '../../theme/common.style';
import { GigStatus } from '../../utils/common/enums';
import { handleError, handleSuccess } from '../../utils/common/function';
import SalesReport from './charts/sales-report';
const Dashboard = () => {
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);

  const [value, setValue] = useState<[Date | null, Date | null]>([null, null]);

  const initialPaginationFromQueryParams = () => {
    const rowPerPage =
      searchParams.get('rowPerPage') ?? paginationInfoValue.rowPerPage;
    const currentPage = Number(
      searchParams.get('currentPage') ??
        paginationInfoValue.currentPage?.toString()
    );
    return { ...paginationInfoValue, rowPerPage, currentPage };
  };
  const handlePagination = (actionType: string, value?: any) =>
    handlePaginationValue(
      actionType,
      value,
      searchParams,
      paginationInfo,
      setPaginationInfo,
      setSearchParams
    );
  const [searchParams, setSearchParams] = useSearchParams();
  const { isSmallScreen } = useScreenSize();
  const [userData, setUserData] = useState<any>();
  const [isLoading, setIsLoading] = useState(true);

  const [selectedRange, setSelectedRange] = useState<
    'daily' | 'weekly' | 'monthly' | 'annual'
  >('daily');
  const [chartData, setChartData] = useState<{
    categories: string[];
    values: number[];
  }>({
    categories: [],
    values: [],
  });

  useEffect(() => {
    const fetchDashboardStats = async () => {
      if (!userInfo?.id) return;

      const formatDate = (date: Date) => {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
      };

      let startDate: string;
      let endDate: string;

      const today = new Date();

      switch (selectedRange) {
        case 'daily': {
          startDate = formatDate(today);
          endDate = formatDate(today);
          break;
        }
        case 'weekly': {
          const startOfWeek = new Date(today);
          startOfWeek.setDate(today.getDate() - 6);
          startDate = formatDate(startOfWeek);
          endDate = formatDate(today);
          break;
        }
        case 'monthly': {
          const startOfMonth = new Date(
            today.getFullYear(),
            today.getMonth(),
            1
          );
          startDate = formatDate(startOfMonth);
          endDate = formatDate(today);
          break;
        }
        case 'annual': {
          const startOfYear = new Date(today.getFullYear(), 0, 1);
          startDate = formatDate(startOfYear);
          endDate = formatDate(today);
          break;
        }
        default: {
          startDate = formatDate(today);
          endDate = formatDate(today);
          break;
        }
      }

      // If custom range selected
      if (value[0] && value[1]) {
        startDate = formatDate(value[0]);
        endDate = formatDate(value[1]);
      }

      const payload = {
        id: userInfo.id,
        startDate,
        endDate,
      };

      setIsLoading(true);

      postData('statistics/entertainer-dashboard', payload)
        .then((response: any) => {
          const formatted = response.data.performanceFeeOverTime.map(
            (item: any) => ({
              category: item.date, // format if needed
              value: item.fee,
            })
          );

          setChartData({
            categories: formatted.map(
              (item: { category: any }) => item.category
            ),
            values: formatted.map((item: { value: any }) => item.value),
          });

          setUserData(response);

          // Optional success notification:
          // handleSuccess('Entertainer dashboard data loaded.');
        })
        .catch(error => {
          console.error('Error:', error);
          handleError(error); // Use your error notification utility
        })
        .finally(() => {
          setIsLoading(false);
        });
    };

    fetchDashboardStats();
  }, [selectedRange, value]);

  const [paginationInfo, setPaginationInfo] = useState(
    initialPaginationFromQueryParams()
  );
  useEffect(() => {
    const newPaginationInfo = initialPaginationFromQueryParams();

    if (JSON.stringify(newPaginationInfo) !== JSON.stringify(paginationInfo)) {
      setPaginationInfo(newPaginationInfo);
    }
  }, [searchParams]);

  const handleGigStatusUpdate = (
    gigId: string,
    newStatus: GigStatus.ACCEPTED | GigStatus.REJECTED
  ) => {
    if (newStatus === GigStatus.ACCEPTED) {
      putData(`gig/${gigId}/accept`, {})
        .then(() => {
          handleSuccess(`Gig ${newStatus} successfully`);
          setUserData((prev: any) => ({
            ...prev,
            data: {
              ...prev.data,
              gigs: prev.data.gigs.filter((gig: any) => gig.id !== gigId),
            },
          }));
        })
        .catch((error: any) => {
          handleError(error);
        });
    } else if (newStatus === GigStatus.REJECTED) {
      putData(`gig/${gigId}/reject`, {})
        .then(() => {
          handleSuccess(`Gig ${newStatus} successfully`);
          setUserData((prev: any) => ({
            ...prev,
            data: {
              ...prev.data,
              gigs: prev.data.gigs.filter((gig: any) => gig.id !== gigId),
            },
          }));
        })
        .catch((error: any) => {
          handleError(error);
        });
    }
  };

  const columns = [
    {
      header: (
        <div className="flex text-start ml-1 rtl:ml-0 rtl:mr-3 font-normal font-productsans text-[14px]">
          {getTranslation(language, 'dashboard', 'ID')}
        </div>
      ),
      accessorKey: 'orderId',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex text-start ml-2">
            <p className="text-start font-normal font-productsans text-[14px]">
              {rowObject?.orderId}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div
          className={` ${isSmallScreen ? 'flex text-center items-center ml-3 rtl:ml-0 rtl:mr-3 font-normal font-productsans text-[14px]' : 'flex text-center items-center ml-12 rtl:ml-0 rtl:mr-4 font-normal font-productsans text-[14px] '}`}
        >
          {getTranslation(language, 'dashboard', 'Events')}
        </div>
      ),
      accessorKey: 'title',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div className="flex items-center text-start">
            <div>
              <img
                src={rowObject?.event?.eventImage}
                alt="image"
                className="rounded-md w-9 h-6 object-cover"
                loading="lazy"
              />
            </div>
            <p className="text-start ml-2 font-productsans font-normal text-[14px]">
              {rowObject?.event?.title}
            </p>
          </div>
        );
      },
    },
    {
      header: (
        <div
          className={`${
            isSmallScreen
              ? 'flex text-start ml-3 font-normal font-productsans text-[14px]'
              : 'flex text-start ml-7 font-normal font-productsans text-[14px] rtl:mr-4 rtl:ml-0'
          }`}
        >
          {getTranslation(language, 'dashboard', 'Date')}
        </div>
      ),
      accessorKey: 'createdAt', // actual key used in the data
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        const formattedDate = new Date(
          rowObject?.createdAt
        ).toLocaleDateString(); // Formats as MM/DD/YYYY or based on locale

        return (
          <div className="flex text-start">
            <p className="text-start font-productsans font-normal text-[14px]">
              {formattedDate}
            </p>
          </div>
        );
      },
    },

    {
      header: (
        <div className="flex items-end justify-end mr-20 font-normal font-productsans text-[14px] rtl:ml-56 rtl:mr-0">
          {getTranslation(language, 'dashboard', 'Status')}
        </div>
      ),
      accessorKey: 'status',
      cell: (info: any) => {
        const rowObject = info?.row?.original;
        return (
          <div
            className={` ${isSmallScreen ? 'flex items-center justify-center font-semibold ml-3 -mb-2' : 'flex items-center justify-center font-semibold ml-3 -mt-2 -mb-2'}`}
          >
            <Button
              autoContrast
              m={5}
              styles={{
                root: {
                  background: '#7E3DEE',
                },
              }}
              style={{ height: 30, width: 'auto' }}
              onClick={() =>
                handleGigStatusUpdate(rowObject.id, GigStatus.ACCEPTED)
              }
            >
              <div className="font-productsans font-normal text-[14px] text-[#ffffff]">
                {getTranslation(language, 'dashboard', 'Accept')}
              </div>
            </Button>

            <Button
              autoContrast
              m={5}
              size="sm"
              styles={{
                root: {
                  background: '#ED272B',
                },
              }}
              style={{ height: 30, width: 79 }}
              onClick={() =>
                handleGigStatusUpdate(rowObject.id, GigStatus.REJECTED)
              }
            >
              <div className="font-productsans font-normal text-[14px] text-[#ffffff] ml-0.5 w-10">
                {getTranslation(language, 'dashboard', 'Reject')}
              </div>
            </Button>
          </div>
        );
      },
    },
  ];

  const filters: ('daily' | 'weekly' | 'monthly')[] = [
    'daily',
    'weekly',
    'monthly',
  ];
  return (
    <>
      <Paper
        className="flex justify-between items-center overflow-x-hidden p-4 min-h-screen "
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid
          className="flex flex-row justify-between py-5 px-4 "
          gutter={'xl'}
        >
          {(isLoading
            ? [1, 2, 3].map(i => ({
                title: (
                  <div className="h-4 w-32 bg-gray-400 rounded animate-pulse mt-2" />
                ),
                value: (
                  <div className="h-4 w-20 bg-gray-400 rounded animate-pulse mt-2" />
                ),
                icon: (
                  <div className="animate-pulse w-20 h-16 bg-gray-400 rounded-full" />
                ),
                colorIcon: 'transparent',
                iconbg: 'transparent',
              }))
            : [
                {
                  title: `${userData?.data?.performanceFeeTotal ? userData.data.performanceFeeTotal + '+' : '0'}`,
                  value: 'Sales',
                  icon: (
                    <IconHeartFilled
                      style={{ width: rem(26), height: rem(26) }}
                    />
                  ),
                  colorIcon: '#8640FD',
                  iconbg: '#623F9D',
                },
                {
                  title: `${userData?.data?.eventsCount ? userData.data.eventsCount + '+' : '0'}`,
                  value: 'Events',
                  icon: (
                    <BsFillCalendarEventFill
                      style={{ width: rem(24), height: rem(24) }}
                    />
                  ),
                  colorIcon: '#FF69B4',
                  iconbg: '#90438E',
                },
                {
                  title: `${userData?.data?.gigsCount ? userData.data.gigsCount + '+' : '0'}`,
                  value: 'Gigs',
                  icon: (
                    <IconBriefcaseFilled
                      style={{ width: rem(32), height: rem(32) }}
                    />
                  ),
                  colorIcon: '#ffffff',
                  iconbg: '#605CFF4D',
                },
              ]
          ).map((item: any, index: number) => (
            <Grid.Col key={index} span={{ base: 12, sm: 6, md: 4, lg: 4 }}>
              <div className="flex flex-row justify-start rounded-2xl p-8 space-y-2 h-[100%] w-[99%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
                <div>
                  <div
                    style={{
                      backgroundColor: item?.iconbg,
                      color: item?.colorIcon,
                      width: '64px',
                      height: '64px',
                    }}
                    className="rounded-full shadow-sm flex items-center justify-center"
                  >
                    {item?.icon}
                  </div>
                </div>
                <div className="ml-6 rtl:mr-6 rtl:ml-0">
                  <div className="font-productsans text-[#ffffff] font-bold text-[20px]">
                    {item?.title}
                  </div>
                  <div className="font-productsans text-[14px] text-[#ABADBF]">
                    {getTranslation(language, 'dashboard', item?.value)}
                  </div>
                </div>
              </div>
            </Grid.Col>
          ))}
        </Grid>

        <Grid className="px-4 py-4" gutter={'xl'}>
          <Grid.Col span={{ base: 12, md: 12, lg: 7.75 }}>
            <div className="rounded-2xl font-semibold font-productsans text-[22px] h-[100%] text-[#ffffff] p-5 mt-1 bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <Grid>
                <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
                  <div className="ml-1.5">
                    {getTranslation(language, 'dashboard', 'Performance')}
                  </div>
                </Grid.Col>
                <Flex
                  wrap="wrap"
                  gap={10}
                  align="center"
                  style={{
                    width: '100%',
                  }}
                >
                  {filters.map(filter => (
                    <Box key={filter}>
                      <Button
                        m={5}
                        styles={{
                          root: {
                            background:
                              selectedRange === filter
                                ? 'rgba(140, 93, 248, 0.2)'
                                : 'transparent',
                            border:
                              selectedRange === filter
                                ? '1px solid #8C5DF8'
                                : '1px solid #FFFFFF26',
                            '&:hover': {
                              background: 'rgba(140, 93, 248, 0.1)',
                            },
                          },
                        }}
                        style={{
                          height: 35,
                          minWidth: 100,
                        }}
                        onClick={() => setSelectedRange(filter)}
                      >
                        <div
                          className="font-productsans font-normal text-[14px]"
                          style={{ color: '#ffffff' }}
                        >
                          {getTranslation(
                            language,
                            'dashboard',
                            filter.charAt(0).toUpperCase() + filter.slice(1)
                          )}
                        </div>
                      </Button>
                    </Box>
                  ))}

                  <Box mt={5} style={{ flexShrink: 0 }}>
                    <div
                      className="text-[14px] font-productsans"
                      style={{ color: '#ffffff' }}
                    >
                      {getTranslation(language, 'dashboard', 'Select Date')}
                    </div>
                  </Box>

                  <Box
                    style={{
                      minWidth: 200,
                      flexShrink: 0,
                    }}
                  >
                    <DatePickerInput
                      type="range"
                      placeholder={getTranslation(
                        language,
                        'dashboard',
                        'Pick Date Range'
                      )}
                      required
                      value={value}
                      onChange={setValue}
                      valueFormat="M/D/YY"
                      style={filterStyle}
                      styles={{
                        input: {
                          backgroundColor: 'transparent',
                          borderColor: '#FFFFFF26',
                          color: '#ffffff',
                          fontFamily: 'Product Sans',
                          fontSize: '12px',
                        },
                        placeholder: { color: 'white', fontWeight: 'normal' },
                      }}
                      rightSection={
                        <IconChevronDown size={16} color="#ffffff" />
                      }
                      leftSection={<IconCalendar size={20} color="#ffffff" />}
                    />
                  </Box>
                </Flex>
              </Grid>
              <div className="mt-3 ">
                <SalesReport
                  data={{
                    categories: chartData?.categories,
                    values: chartData?.values,
                  }}
                />
              </div>
            </div>
          </Grid.Col>
          <Grid.Col span={{ base: 12, md: 12, lg: 4.25 }}>
            <div className="rounded-2xl font-semibold text-[22px] text-[#ffffff] p-5 mt-1 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <div className=" flex flex-row justify-between">
                <div className="">
                  {getTranslation(language, 'dashboard', 'Calendar')}
                </div>
                <div>
                  <IconDots />
                </div>
              </div>
              <div className="mt-4 border border-[#ffffff] rounded-md text-[#ffffff] overflow-hidden">
                <BasicDateCalendar />
              </div>
            </div>
          </Grid.Col>
        </Grid>
        <Grid className="py-5 px-1">
          <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
            <div
              className="rounded-2xl ml-4 mr-2 font-semibold text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]"
              style={{ height: 'auto' }}
            >
              <div className=" flex flex-row justify-between">
                <div className="ml-2 text-[22px]">
                  {getTranslation(language, 'dashboard', 'GigsRequests')}
                </div>
                <div>
                  <IconDots />
                </div>
              </div>
              <div className="mt-7">
                <Table
                  data={userData?.data?.gigs?.filter(
                    (gig: any) => gig.status === GigStatus.PENDING
                  )}
                  columns={columns}
                  isPaginated={false}
                  isLoading={isLoading}
                  paginationInfo={paginationInfo}
                  handlePagination={handlePagination}
                />
              </div>
            </div>
          </Grid.Col>
        </Grid>
      </Paper>
    </>
  );
};

export default Dashboard;
