import { Flex, Text, Box } from '@mantine/core';
import { IconArrowDownLeft, IconArrowUpRight } from '@tabler/icons-react';
import Highcharts from 'highcharts';
import HighchartsReact from 'highcharts-react-official';
import { useSelector } from 'react-redux';
import { getTranslation } from '../../../layout/languages';

interface TimelineData {
  period: string;
  ticketCommission: string;
  performanceCommission: string;
  totalCommission: string;
}
interface ComparisonData {
  currentWeek: string;
  lastWeek: string;
  change: string;
  changePercentage: string;
}

interface TotalRevenueProps {
  timelineData?: TimelineData[];
  comparisonData?: ComparisonData;
}

const TotalRevenue = ({
  timelineData = [],
  comparisonData,
}: TotalRevenueProps) => {
  const formatData = () => {
    if (!timelineData || timelineData.length === 0) {
      return {
        categories: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'],
        ticketData: [0, 0, 0, 0, 0, 0],
        performanceData: [0, 0, 0, 0, 0, 0],
      };
    }

    const sortedData = [...timelineData].sort(
      (a, b) => new Date(a.period).getTime() - new Date(b.period).getTime()
    );

    const categories = sortedData.map(item => {
      const date = new Date(item.period);
      return date.toLocaleString('default', { month: 'short' });
    });

    const ticketData = sortedData.map(item =>
      parseFloat(item.ticketCommission)
    );
    const performanceData = sortedData.map(item =>
      parseFloat(item.performanceCommission)
    );
    const totalRevenue = sortedData.map(item =>
      parseFloat(item.totalCommission).toFixed(3)
    );
    return { categories, ticketData, performanceData, totalRevenue };
  };

  const { categories, ticketData, performanceData, totalRevenue } =
    formatData();

  const growthPercentage = comparisonData?.changePercentage || '0 %';
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  return (
    <Box p="md">
      {/* Custom Legend and Stats */}
      <Flex justify="space-between" align="flex-start" mb="sm">
        {/* Custom Legend */}
        <Box>
          <Flex align="center" gap="xs">
            <Box w={10} h={10} bg="#BC16FE" style={{ borderRadius: '50%' }} />
            <Text c="white" size="sm">
              {getTranslation(language, 'dashboard', 'From Ticket Sales')}
            </Text>
          </Flex>
          <Flex align="center" gap="xs" mt={8}>
            <Box w={10} h={10} bg="#D83F87" style={{ borderRadius: '50%' }} />
            <Text c="white" size="sm">
              {getTranslation(
                language,
                'dashboard',
                'From Service Transaction'
              )}
            </Text>
          </Flex>
        </Box>

        {/* Revenue Summary */}
        <Box ta="right">
          <Text c="white" fw={700} size="xl">
            {` ${currency === '1' ? '$' : 'BHD ≈'} ${totalRevenue || '0'}`}
          </Text>
          <Flex align="center" justify="flex-end" gap={4}>
            {parseFloat(growthPercentage) >= 0 ? (
              <>
                <IconArrowUpRight size={16} color="limegreen" />
                <Text c="limegreen" size="md" fw={500}>
                  {growthPercentage}
                </Text>
              </>
            ) : (
              <>
                <IconArrowDownLeft size={16} color="#d63302ff" />
                <Text c="#d63302ff" size="md" fw={500}>
                  {growthPercentage}
                </Text>
              </>
            )}
            <Text c="white" size="sm">
              {getTranslation(language, 'dashboard', 'vs last week')}
            </Text>
          </Flex>
        </Box>
      </Flex>

      {/* Highcharts Chart */}
      <HighchartsReact
        key={`${JSON.stringify(timelineData)}-${currency}`}
        highcharts={Highcharts}
        options={{
          chart: {
            type: 'column',
            backgroundColor: 'transparent',
            height: '400px',
          },
          title: { text: '' },
          xAxis: {
            categories,
            type: 'linear',
            labels: {
              style: { color: 'white' },
            },
            lineColor: '#E2E7E7',
            tickColor: 'transparent',
            crosshair: true,
          },
          yAxis: {
            labels: { enabled: false },
            gridLineWidth: 1,
            gridLineDashStyle: 'dot',
            title: { text: '' },
          },
          plotOptions: {
            column: { pointPadding: 0.2, borderWidth: 0 },
          },
          legend: {
            enabled: false, // Hide default chart legend
          },
          series: [
            {
              name: 'From Service Transaction',
              color: '#D83F87',
              data: performanceData,
            },
            {
              name: 'From Ticket Sales',
              color: '#BC16FE',
              data: ticketData,
            },
          ],
        }}
      />
    </Box>
  );
};

export default TotalRevenue;
