import { useMemo } from 'react';
import Highcharts from 'highcharts';
import HighchartsReact from 'highcharts-react-official';
import { useSelector } from 'react-redux';
import { getTranslation } from '../../../layout/languages';

interface AnalyticsItem {
  date: string;
  userCount: number;
  customerCount: number;
}

interface LiveAnalyticsProps {
  data?: AnalyticsItem[];
}

const LiveAnalytics = ({ data }: any) => {
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const { categories, userData, customerData } = useMemo(() => {
    if (!data || data.length === 0) {
      return {
        categories: [],
        userData: [],
        customerData: [],
      };
    }

    const sortedData = [...data].sort(
      (a, b) => new Date(a.date).getTime() - new Date(b.date).getTime()
    );

    const categories = sortedData.map(item => {
      const date = new Date(item.date);

      const month = date.toLocaleString('en-US', { month: 'short' }); // "May"
      const day = date.getDate(); // 13
      const year = date.getFullYear().toString().slice(-2); // "25"

      const hours = date.getHours();
      const minutes = date.getMinutes().toString().padStart(2, '0');
      const ampm = hours >= 12 ? 'PM' : 'AM';
      const displayHours = hours % 12 || 12;

      return `${month} ${day} ${year}, ${displayHours}:${minutes} ${ampm}`;
    });

    const userData = sortedData.map(item => item.userCount);
    const customerData = sortedData.map(item => item.customerCount);

    return { categories, userData, customerData };
  }, [data]);

  const fallbackCategories = [
    '10am',
    '11am',
    '12pm',
    '01pm',
    '02pm',
    '03pm',
    '04pm',
    '05pm',
    '06pm',
    '07pm',
  ];

  const chartKey = categories.join('-') || 'default-key';

  return (
    <HighchartsReact
      key={chartKey}
      highcharts={Highcharts}
      options={{
        chart: {
          type: 'spline',
          backgroundColor: 'transparent',
          height: '345px',
        },
        title: {
          text: '',
        },
        xAxis: {
          categories: categories.length ? categories : fallbackCategories,
          gridLineColor: '#824DDD',
          labels: {
            style: {
              color: '#ABADBF',
              fontWeight: 'normal',
            },
          },
          crosshair: {
            color: 'transparent',
            width: 0,
          },
        },
        yAxis: {
          title: {
            text: '',
          },
          gridLineColor: '#824DDD',
          labels: {
            style: {
              color: '#ABADBF',
              fontWeight: 'normal',
            },
          },
        },
        tooltip: {
          crosshairs: true,
          shared: false,
        },
        plotOptions: {
          spline: {
            marker: {
              radius: 16,
              lineColor: '#824DDD',
              lineWidth: 4,
            },
          },
        },
        legend: {
          enabled: true,
          layout: 'horizontal',
          align: 'left',
          verticalAlign: 'top',
          y: -18,
          x: -15,
          itemStyle: {
            color: '#ffffff',
            fontWeight: 'normal',
            fontFamily: 'Product Sans',
            fontSize: '16px',
          },
          itemHoverStyle: {
            color: '#ffffff',
          },
        },
        series: [
          {
            color: {
              linearGradient: { x1: 0, y1: 0, x2: 1, y2: 0 },
              stops: [
                [0, '#8C5DF8'],
                [1, '#C508FF'],
              ],
            },
            lineWidth: 5,
            name: ` ${getTranslation(language, 'dashboard', 'Users Engagement')}`,
            marker: {
              color: '#000000',
              symbol: 'circle',
              radius: 6,
              fillColor: '#BC16FE',
              lineColor: '#ffffff',
              lineWidth: 3,
            },
            data: userData.length ? userData : Array(10).fill(0),
          },
          {
            color: '#D83F87',
            name: ` ${getTranslation(language, 'dashboard', 'Customers Engagement')}`,
            lineWidth: 5,
            marker: {
              color: '#000000',
              symbol: 'circle',
              radius: 6,
              fillColor: '#F85154',
              lineColor: '#ffffff',
              lineWidth: 3,
            },
            data: customerData.length ? customerData : Array(10).fill(0),
          },
        ],
      }}
    />
  );
};

export default LiveAnalytics;
