import Highcharts from 'highcharts';
import HighchartsReact from 'highcharts-react-official';
import { useSelector } from 'react-redux';
import { getTranslation } from '../../../layout/languages';

interface CommissionData {
  ticketSoldCommission: number;
  performanceCommission: number;
  total: number;
}

interface ComparisonData {
  currentWeek: number;
  lastWeek: number;
  change: number;
  changePercentage: string;
}

interface CommissionOverViewProps {
  commissionData?: CommissionData;
  comparisonData?: ComparisonData;
}

const CommissionOverView = ({
  commissionData,
  comparisonData,
}: CommissionOverViewProps) => {
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  // Format the total amount for the center label
  const formattedTotal = commissionData
    ? `${commissionData.total.toFixed(3)} `
    : `0`;
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );

  return (
    <HighchartsReact
      highcharts={Highcharts}
      options={{
        chart: {
          type: 'pie',
          backgroundColor: 'transparent',
          height: '400x',
          custom: {},
          events: {
            render() {
              // eslint-disable-next-line @typescript-eslint/no-this-alias
              const chart: any = this,
                series = chart.series[0];
              let customLabel = chart.options.chart.custom.label;

              if (!customLabel) {
                customLabel = chart.options.chart.custom.label = chart.renderer
                  .label(
                    ` ${getTranslation(language, 'dashboard', 'Total Earn')}<br/><strong>${currency === '1' ? '$' : 'BHD ≈'}  ${formattedTotal}</strong>`
                  )
                  .css({
                    color: '#ffffff',
                    fontFamily: 'Product Sans',
                    textAnchor: 'middle',
                  })
                  .add();
              }

              const x = series.center[0] + chart.plotLeft,
                y =
                  series.center[1] +
                  chart.plotTop -
                  customLabel.attr('height') / 2;

              customLabel.attr({
                x,
                y,
              });
              // Set font size based on chart diameter
              customLabel.css({
                fontSize: `${series.center[2] / 12}px`,
              });
            },
          },
        },
        accessibility: {
          point: {
            valueSuffix: '%',
          },
        },
        title: {
          text: ` ${getTranslation(language, 'dashboard', 'Commission Overview')}`,
          align: 'center',
          style: {
            color: '#ffffff',
            fontWeight: 'bold',
            fontSize: '24px',
          },
        },
        legend: {
          enabled: true,
          align: 'left',
          verticalAlign: 'bottom',
          y: -18,
          x: 15,
          itemStyle: {
            color: '#ffffff',
            fontWeight: 'normal',
            fontFamily: 'Product Sans',
            fontSize: '12px',
          },
          itemMarginBottom: 8,
          useHTML: true,
          labelFormatter: function (this: Highcharts.Point) {
            return `
                <div style="display: flex; flex-direction: column; align-items: center; font-family: 'Product Sans'; color: #ffffff;">
                  <span>${this.name}</span>
                  <strong>${this.percentage?.toFixed(0)}%</strong>
                </div>
              `;
          },
        },
        tooltip: {
          valueDecimals: 2,
          valueSuffix: '%',
        },
        plotOptions: {
          series: {
            borderWidth: 0,
            colorByPoint: true,
            type: 'pie',
            size: '100%',
            innerSize: '80%',
            dataLabels: {
              enabled: false,
              crop: false,
              distance: '-10%',
              style: {
                fontWeight: 'bold',
                fontSize: '16px',
              },
              connectorWidth: 0,
            },
            showInLegend: true,
          },
          pie: {
            allowPointSelect: true,
            cursor: 'pointer',
            dataLabels: {
              enabled: false,
            },
            showInLegend: true,
          },
        },
        colors: ['#8C5DF8', '#D83F87'],
        series: [
          {
            name: 'Commission Overview',
            colorByPoint: true,
            innerSize: '65%',
            data: [
              {
                name: ` ${getTranslation(language, 'dashboard', 'From Ticket Sales')}`,
                y: commissionData?.ticketSoldCommission || 0,
                color: {
                  linearGradient: { x1: 0, y1: 0, x2: 1, y2: 0 },
                  stops: [
                    [0, '#8C5DF8'],
                    [1, '#C508FF'],
                  ],
                },
              },
              {
                name: ` ${getTranslation(language, 'dashboard', 'From Service Transaction')}`,
                y: commissionData?.performanceCommission || 0,
              },
            ],
          },
        ],
      }}
      key={`${JSON.stringify(commissionData)}-${currency}`}
    />
  );
};

export default CommissionOverView;
