import * as React from 'react';
import dayjs, { Dayjs } from 'dayjs';
import Badge from '@mui/material/Badge';
import { AdapterDayjs } from '@mui/x-date-pickers/AdapterDayjs';
import { LocalizationProvider } from '@mui/x-date-pickers/LocalizationProvider';
import { PickersDay, PickersDayProps } from '@mui/x-date-pickers/PickersDay';
import { DateCalendar } from '@mui/x-date-pickers/DateCalendar';
import { DayCalendarSkeleton } from '@mui/x-date-pickers/DayCalendarSkeleton';
import { FaCaretLeft, FaCaretRight } from 'react-icons/fa';
import { useSelector } from 'react-redux';
import { postData } from '../../../api/api';
import { handleError } from '../../../utils/common/function';

const fakeFetch = (date: Dayjs, { signal }: { signal: AbortSignal }) =>
  new Promise<{ daysToHighlight: { date: string; color: string }[] }>(
    resolve => {
      setTimeout(() => {
        const mockData = [
          {
            date: date.startOf('month').add(2, 'day').format('YYYY-MM-DD'),
            color: 'green',
          },
          {
            date: date.startOf('month').add(5, 'day').format('YYYY-MM-DD'),
            color: 'red',
          },
        ];
        resolve({ daysToHighlight: mockData });
      }, 500);
    }
  );

function ServerDay(
  props: PickersDayProps<Dayjs> & {
    highlightedDays?: { date: string; color: string }[];
  }
) {
  const { highlightedDays = [], day, outsideCurrentMonth, ...other } = props;

  // Find the matching day with the full date (day, month, year)
  const highlight = highlightedDays.find(
    highlighted => day.isSame(dayjs(highlighted.date), 'day') // Check if the day matches the date
  );

  const isSelected = !outsideCurrentMonth && !!highlight;
  const isOutsideCurrentMonth = outsideCurrentMonth && !!highlight;
  return (
    <Badge key={day.toString()} overlap="circular" badgeContent={null}>
      <PickersDay
        {...other}
        outsideCurrentMonth={outsideCurrentMonth}
        day={day}
        style={{
          position: 'relative',
          backgroundColor: isSelected
            ? highlight?.color
            : isOutsideCurrentMonth
              ? '#FFE347'
              : '',

          border: isSelected ? `2px solid ${highlight?.color}` : 'none',
          borderRadius: '50%',
          width: '36px',
          height: '36px',
          display: 'flex',
          justifyContent: 'center',
          alignItems: 'center',
        }}
      />
    </Badge>
  );
}

export default function BasicDateCalendarBussiness() {
  const requestAbortController = React.useRef<AbortController | null>(null);
  const [isLoading, setIsLoading] = React.useState(false);
  const [highlightedDays, setHighlightedDays] = React.useState<
    { date: string; color: string }[]
  >([]);
  const [userData, setUserData] = React.useState<any>();
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  // This runs once userData is set after fetch
  React.useEffect(() => {
    if (userData?.data?.upcomingEventsForCalender) {
      fetchHighlightedDays(dayjs()); // Run for current month
    }
  }, [userData]);

  React.useEffect(() => {
    const fetchDashboardStats = async () => {
      if (!userInfo?.id) return;

      const url = 'statistics/business-dashboard';

      const today = new Date();
      const tomorrow = new Date(today);
      tomorrow.setDate(tomorrow.getDate() + 1);

      const formatDate = (date: Date) => {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
      };

      const payload = {
        id: userInfo.id,
        startDate: formatDate(today),
        endDate: formatDate(tomorrow),
      };

      setIsLoading(true);

      postData(url, payload)
        .then((response: any) => {
          setUserData(response);
          // Optional: handleSuccess('Entertainer dashboard data loaded successfully.');
        })
        .catch(error => {
          console.error('Error fetching entertainer dashboard stats:', error);
          handleError(error); // Use your standard error handler
        })
        .finally(() => {
          setIsLoading(false); // Always stop loading
        });
    };

    fetchDashboardStats();
  }, []);

  const fetchHighlightedDays = (date: Dayjs) => {
    const controller = new AbortController();

    // Make sure you cancel previous requests
    requestAbortController.current = controller;

    if (!userData?.data?.upcomingEventsForCalender) return;

    const highlights: { date: string; color: string }[] = [];

    userData?.data?.upcomingEventsForCalender.forEach((entry: any) => {
      const start = dayjs(entry.startDateTime);
      const end = dayjs(entry.endDateTime);

      // Red highlight for dateFrom
      highlights.push({ date: start.format('YYYY-MM-DD'), color: '#FF2125' });

      // Green highlight for dateTo
      highlights.push({ date: end.format('YYYY-MM-DD'), color: '#418F90' });

      // Green highlight for dates in between
      let current = start.add(1, 'day');
      while (current.isBefore(end, 'day')) {
        highlights.push({
          date: current.format('YYYY-MM-DD'),
          color: '#418F90',
        });
        current = current.add(1, 'day');
      }
    });

    setHighlightedDays(highlights);
    setIsLoading(false);
  };

  React.useEffect(() => {
    fetchHighlightedDays(dayjs()); // Load with current month
    return () => requestAbortController.current?.abort();
  }, []);

  const handleMonthChange = (date: Dayjs) => {
    if (requestAbortController.current) {
      requestAbortController.current.abort();
    }

    setIsLoading(true);
    setHighlightedDays([]);
    fetchHighlightedDays(date);
  };

  return (
    <LocalizationProvider dateAdapter={AdapterDayjs}>
      <DateCalendar
        showDaysOutsideCurrentMonth
        loading={isLoading}
        onMonthChange={handleMonthChange}
        renderLoading={() => <DayCalendarSkeleton />}
        slotProps={{
          day: {
            highlightedDays,
          } as any,
        }}
        sx={{
          width: '100%',
          maxWidth: '100%',
          flexWrap: 'wrap',
          color: 'white',
          fontFamily: 'Sans-Serif',
          '.MuiTypography-root': {
            color: '#ABADBF',
          },
          '.MuiPickersDay-root': {
            color: 'white',
          },
          '& .MuiPickersDay-dayOutsideMonth': {
            color: '#9E9E9E',
          },
          '.MuiPickersDay-root.Mui-selected': {
            backgroundColor: '#418F90',
            color: 'white',
          },
          '.MuiPickersCalendarHeader-root': {
            color: 'white',
            fontSize: '12px',
          },
          '.MuiPickersCalendarHeader-label': {
            color: 'white',
            fontSize: '16px',
          },
          '.MuiPickersCalendarHeader-switchViewButton': {
            display: 'none',
          },
        }}
        slots={{
          day: ServerDay,
          leftArrowIcon: (props: any) => (
            <button
              {...props}
              style={{
                color: '#ABADBF',
                fontSize: '1.5rem',
                border: 'none',
                background: 'transparent',
              }}
              className="-mr-3.5 rtl:-ml-7 rtl:mr-0 "
            >
              <FaCaretLeft />
            </button>
          ),
          rightArrowIcon: (props: any) => (
            <button
              {...props}
              style={{
                color: '#ABADBF',
                fontSize: '1.5rem',
                border: 'none',
                background: 'transparent',
              }}
              className="mr-2.5 rtl:ml-2.5 rtl:mr-0"
            >
              <FaCaretRight />
            </button>
          ),
        }}
      />
    </LocalizationProvider>
  );
}
