import { Grid, Loader, Paper, rem } from '@mantine/core';
import { DatePickerInput } from '@mantine/dates';
import {
  IconBriefcaseFilled,
  IconCalendar,
  IconChevronDown,
  IconDots,
  IconHeartFilled,
  IconTicket,
} from '@tabler/icons-react';
import { BsFillCalendarEventFill } from 'react-icons/bs';
import { Button, Table } from '../../components';

import { useCallback, useEffect, useMemo, useState } from 'react';
import { useSelector } from 'react-redux';
import { postData } from '../../api/api';
import useScreenSize from '../../hooks/useScreenSize';
import { getTranslation } from '../../layout/languages';
import { handleError } from '../../utils/common/function';

import SaleView from '../sales/sales-views';
import BasicDateCalendarBussiness from './charts/calendar-bussiness';
import SalesReport from './charts/sales-report';

interface DashboardStats {
  totalSalesRevenue: number;
  totalTicket: number;
  totalEvent: number;
  upcomingEvents: number;
  lastTicketSoldOut: any[];
  upcomingEventsForCalender: Array<{
    date: string;
    fee: number;
  }>;
}

interface ChartData {
  categories: string[];
  values: number[];
}

type TimeRange = 'daily' | 'weekly' | 'monthly' | 'annual';

const BusinessDashboard = () => {
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );

  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );

  const [value, setValue] = useState<[Date | null, Date | null]>([null, null]);
  const { isSmallScreen, isMediumScreen, isLargeScreen } = useScreenSize();
  const [userData, setUserData] = useState<{ data: DashboardStats } | null>(
    null
  );
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const [isLoading, setIsLoading] = useState(true);
  const [isRefreshing, setIsRefreshing] = useState(false);

  const [error, setError] = useState<string | null>(null);
  const [lastUpdated, setLastUpdated] = useState<Date>(new Date());
  const { referenceData } = useSelector((state: any) => state?.referenceData);
  const [accountInfo, setaccountInfo] = useState({
    isModalOpen: false,
    id: '',
    row: null,
  });

  const columns = useMemo(
    () => [
      {
        header: (
          <div className="flex text-start font-normal font-sans text-[13px] sm:text-[15px]">
            {getTranslation(language, 'dashboard', 'Customer Name')}
          </div>
        ),
        accessorKey: 'name',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-start items-center">
              <div className="flex-shrink-0">
                <img
                  src="https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcTnci86jWZWUlwH_ZwKn3hlGxwL2lEUfScKzA&s"
                  alt="Customer"
                  className="rounded-full w-6 h-6 sm:w-7 sm:h-7"
                  loading="lazy"
                />
              </div>
              <p className="text-start ml-2 font-normal font-sans text-[12px] sm:text-[14px] truncate max-w-[120px] sm:max-w-none">
                {rowObject?.user?.fullNameOrBusinessName || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center justify-center font-normal font-sans text-[13px] sm:text-[15px]">
            {getTranslation(language, 'dashboard', 'Email')}
          </div>
        ),
        accessorKey: 'email',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex text-center justify-center">
              <p className="font-sans font-normal text-[12px] sm:text-[14px] truncate max-w-[150px]">
                {rowObject?.user?.email || 'N/A'}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex text-center items-center justify-center font-normal font-sans text-[13px] sm:text-[15px]">
            {getTranslation(language, 'dashboard', 'Ticket Type')}
          </div>
        ),
        accessorKey: 'type',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          const ticketType = rowObject?.ticket?.name || 'N/A';
          return (
            <div className="flex items-center justify-center">
              <p className="text-center font-sans font-normal text-[12px] sm:text-[14px] truncate max-w-[100px]">
                {ticketType}
              </p>
            </div>
          );
        },
      },
      {
        header: (
          <div className="flex items-center justify-end mr-4 rtl:ml-4 rtl:mr-0 font-normal font-sans text-[13px] sm:text-[15px]">
            {getTranslation(language, 'dashboard', 'Action')}
          </div>
        ),
        accessorKey: 'status',
        cell: (info: any) => {
          const rowObject = info?.row?.original;
          return (
            <div className="flex items-center justify-end">
              <Button
                autoContrast
                size={isSmallScreen ? 'xs' : 'sm'}
                onClick={() =>
                  setaccountInfo({
                    isModalOpen: true,
                    id: rowObject?.id,
                    row: rowObject,
                  })
                }
                styles={{
                  root: {
                    background: '#D83F87',
                    '&:hover': {
                      background: '#B8366F',
                    },
                  },
                }}
                style={{
                  height: isSmallScreen ? 28 : 30,
                  width: isSmallScreen ? 100 : 120,
                  fontSize: isSmallScreen ? '12px' : '14px',
                }}
              >
                {getTranslation(language, 'dashboard', 'View Detail')}
              </Button>
            </div>
          );
        },
      },
    ],
    [language, isSmallScreen]
  );

  const [chartData, setChartData] = useState<ChartData>({
    categories: [],
    values: [],
  });
  const [selectedRange, setSelectedRange] = useState<TimeRange>('daily');

  // Enhanced date formatting with error handling
  const formatDate = useCallback((date: Date): string => {
    try {
      const year = date.getFullYear();
      const month = String(date.getMonth() + 1).padStart(2, '0');
      const day = String(date.getDate()).padStart(2, '0');
      return `${year}-${month}-${day}`;
    } catch (error) {
      console.error('Date formatting error:', error);
      return formatDate(new Date());
    }
  }, []);

  // Enhanced date range calculation
  const calculateDateRange = useCallback(
    (range: TimeRange): { startDate: string; endDate: string } => {
      const today = new Date();
      let startDate: string;
      let endDate: string;

      switch (range) {
        case 'daily': {
          startDate = formatDate(today);
          endDate = formatDate(today);
          break;
        }
        case 'weekly': {
          const startOfWeek = new Date(today);
          startOfWeek.setDate(today.getDate() - 6);
          startDate = formatDate(startOfWeek);
          endDate = formatDate(today);
          break;
        }
        case 'monthly': {
          const startOfMonth = new Date(
            today.getFullYear(),
            today.getMonth(),
            1
          );
          startDate = formatDate(startOfMonth);
          endDate = formatDate(today);
          break;
        }
        case 'annual': {
          const startOfYear = new Date(today.getFullYear(), 0, 1);
          startDate = formatDate(startOfYear);
          endDate = formatDate(today);
          break;
        }
        default: {
          startDate = formatDate(today);
          endDate = formatDate(today);
          break;
        }
      }

      // Override with custom range if selected
      if (value[0] && value[1]) {
        startDate = formatDate(value[0]);
        endDate = formatDate(value[1]);
      }

      return { startDate, endDate };
    },
    [value, formatDate]
  );

  useEffect(() => {
    fetchDashboardStats();
    console.log('language', language);
  }, [currency]);
  // Enhanced API call with better error handling and loading states
  const fetchDashboardStats = useCallback(
    async (isRefresh = false) => {
      if (!userInfo?.id) {
        setError('User information not available');
        setIsLoading(false);
        return;
      }

      try {
        if (isRefresh) {
          setIsRefreshing(true);
        } else {
          setIsLoading(true);
        }

        setError(null);

        const { startDate, endDate } = calculateDateRange(selectedRange);

        const payload = {
          id: userInfo.id,
          startDate,
          endDate,
        };

        const response: any = await postData(
          'statistics/business-dashboard',
          payload
        );

        if (response?.data) {
          const formatted =
            response?.data?.upcomingEventsForCalender?.map((item: any) => ({
              category: item.date,
              value: item.fee || 0,
            })) || [];

          setChartData({
            categories: formatted.map((item: any) => item.category),
            values: formatted.map((item: any) => item.value),
          });

          setUserData(response);
          setLastUpdated(new Date());
        }
      } catch (error: any) {
        console.error('Dashboard fetch error:', error);
        setError(error?.message || 'Failed to fetch dashboard data');
        handleError(error);
      } finally {
        setIsLoading(false);
        setIsRefreshing(false);
      }
    },
    [userInfo?.id, selectedRange, calculateDateRange]
  );

  // Enhanced effect with cleanup
  useEffect(() => {
    const controller = new AbortController();

    fetchDashboardStats();

    return () => {
      controller.abort();
    };
  }, [fetchDashboardStats]);

  // Manual refresh handler
  const handleRefresh = useCallback(() => {
    fetchDashboardStats(true);
  }, [fetchDashboardStats]);

  // Enhanced stats cards with better responsive design
  const statsCards = useMemo(
    () => [
      {
        title: `${userData?.data?.totalSalesRevenue ?? 0}${userData?.data?.totalSalesRevenue ? '+' : ''}`,
        value: 'Sales',
        icon: <IconHeartFilled style={{ width: rem(20), height: rem(20) }} />,
        iconbg: '#623F9D',
        colorIcon: '#8640FD',
      },
      {
        title: `${userData?.data?.totalTicket ?? 0}${userData?.data?.totalTicket ? '+' : ''}`,
        value: 'Tickets',
        icon: <IconTicket style={{ width: rem(22), height: rem(22) }} />,
        iconbg: '#2FE5A74D',
        colorIcon: '#2DA49A',
      },
      {
        title: `${userData?.data?.totalEvent ?? 0}${userData?.data?.totalEvent ? '+' : ''}`,
        value: 'Total Events',
        icon: (
          <BsFillCalendarEventFill
            style={{ width: rem(18), height: rem(18) }}
          />
        ),
        iconbg: '#90438E',
        colorIcon: '#FF69B4',
      },
      {
        title: `${userData?.data?.upcomingEvents ?? 0}${userData?.data?.upcomingEvents ? '+' : ''}`,
        value: 'Upcoming Events',
        icon: (
          <IconBriefcaseFilled style={{ width: rem(24), height: rem(24) }} />
        ),
        iconbg: '#605CFF4D',
        colorIcon: '#ffffff',
      },
    ],
    [userData, currency]
  );

  // Enhanced button component for time range selection
  const TimeRangeButton = ({
    range,
    label,
  }: {
    range: TimeRange;
    label: string;
  }) => (
    <Button
      autoContrast
      size={isSmallScreen ? 'xs' : 'sm'}
      styles={{
        root: {
          background:
            selectedRange === range ? 'rgba(140, 93, 248, 0.2)' : 'transparent',
          border:
            selectedRange === range
              ? '1px solid #8C5DF8'
              : '1px solid #FFFFFF26',
          '&:hover': {
            background: 'rgba(140, 93, 248, 0.1)',
          },
        },
      }}
      style={{
        height: isSmallScreen ? 32 : 35,
        minWidth: isSmallScreen ? 60 : 80,
        fontSize: isSmallScreen ? '12px' : '14px',
      }}
      onClick={() => setSelectedRange(range)}
    >
      {getTranslation(language, 'dashboard', label)}
    </Button>
  );

  if (error && !userData) {
    return (
      <Paper
        className="flex justify-center items-center min-h-screen"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <div className="text-center">
          <p className="text-red-500 mb-4">{error}</p>
          <Button onClick={handleRefresh}>Retry</Button>
        </div>
      </Paper>
    );
  }

  return (
    <>
      <Paper
        className="flex justify-between items-center p-2 sm:p-4 min-h-screen"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        {/* Stats Cards Section */}
        <Grid
          className="flex flex-row justify-between px-2 sm:px-4 py-3 sm:py-5"
          gutter="sm"
        >
          {isLoading
            ? Array.from({ length: 4 }).map((_, i) => (
                <Grid.Col
                  key={i}
                  span={{ base: 12, xs: 6, sm: 6, md: 3, lg: 3 }}
                >
                  <div className="rounded-xl sm:rounded-2xl p-4 sm:p-6 lg:p-8 h-full bg-[#ffffff1a] border border-[#623F9D] animate-pulse">
                    <div className="flex items-center space-x-3 sm:space-x-4">
                      <div className="w-12 h-12 sm:w-16 sm:h-16 bg-gray-400 rounded-full flex-shrink-0" />
                      <div className="space-y-2 flex-1">
                        <div className="h-3 sm:h-4 bg-gray-400 rounded w-3/4" />
                        <div className="h-3 sm:h-4 bg-gray-400 rounded w-1/2" />
                      </div>
                    </div>
                  </div>
                </Grid.Col>
              ))
            : statsCards.map((item, index) => (
                <Grid.Col
                  key={index}
                  span={{ base: 12, xs: 6, sm: 6, md: 3, lg: 3 }}
                >
                  <div className="flex flex-row justify-start rounded-xl sm:rounded-2xl p-4 sm:p-6 lg:p-8 h-full bg-transparent bg-opacity-5 bg-[#ffffff] border border-[#623F9D] hover:border-[#8C5DF8] transition-colors duration-200">
                    <div className="flex-shrink-0">
                      <div
                        style={{
                          backgroundColor: item.iconbg,
                          color: item.colorIcon,
                          width: isSmallScreen ? '48px' : '56px',
                          height: isSmallScreen ? '48px' : '56px',
                        }}
                        className="rounded-full shadow-sm flex items-center justify-center"
                      >
                        {item.icon}
                      </div>
                    </div>
                    <div className="ml-3 sm:ml-5 lg:ml-7 rtl:mr-3 sm:rtl:mr-5 lg:rtl:mr-7 rtl:ml-0 flex-1 min-w-0">
                      <div className="font-sans text-[#ffffff] font-bold text-[16px] sm:text-[18px] lg:text-[20px] truncate">
                        {item.title}
                      </div>
                      <div className="font-sans text-[12px] sm:text-[13px] lg:text-[14px] text-[#ABADBF] truncate">
                        {getTranslation(language, 'dashboard', item?.value)}
                      </div>
                    </div>
                  </div>
                </Grid.Col>
              ))}
        </Grid>

        {/* Charts and Calendar Section */}
        <Grid className="px-2 sm:px-4 py-2 sm:py-4" gutter="sm">
          <Grid.Col span={{ base: 12, lg: 8.25 }}>
            <div className="rounded-xl sm:rounded-2xl font-semibold font-sans text-[18px] sm:text-[20px] lg:text-[22px] text-[#ffffff] p-3 sm:p-4 lg:p-5 mt-1 h-full bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <Grid gutter="xs">
                <Grid.Col span={{ base: 12, sm: 12, md: 12, lg: 3 }}>
                  <div className="flex items-center justify-between sm:justify-start">
                    <div className="ml-1.5 text-[16px] sm:text-[18px] lg:text-[22px]">
                      {getTranslation(language, 'dashboard', 'Sale Revenue')}
                    </div>
                  </div>
                </Grid.Col>

                {/* Time Range Buttons */}
                <Grid.Col span={{ base: 12, sm: 12, md: 12, lg: 12 }}>
                  <div className="flex flex-wrap gap-2 sm:gap-3 items-center">
                    <TimeRangeButton range="daily" label="Daily" />
                    <TimeRangeButton range="weekly" label="Weekly" />
                    <TimeRangeButton range="monthly" label="Monthly" />
                    <TimeRangeButton range="annual" label="Annual" />
                    <div className="flex items-center gap-2 ml-2">
                      <div className="text-[12px] sm:text-[14px] font-ptsans hidden sm:block">
                        {getTranslation(language, 'dashboard', 'Select Date')}
                      </div>
                      <div className="w-full sm:w-auto">
                        <DatePickerInput
                          type="range"
                          placeholder={getTranslation(
                            language,
                            'dashboard',
                            'Pick Date Range'
                          )}
                          value={value}
                          onChange={setValue}
                          valueFormat="M/D/YY"
                          size={isSmallScreen ? 'xs' : 'sm'}
                          styles={{
                            input: {
                              backgroundColor: 'transparent',
                              borderColor: '#FFFFFF26',
                              color: '#ffffff',
                              fontFamily: 'PT Sans',
                              fontSize: isSmallScreen ? '11px' : '12px',
                              minWidth: isSmallScreen ? '140px' : '180px',
                            },
                            placeholder: {
                              color: 'white',
                              fontWeight: 'normal',
                            },
                          }}
                          rightSection={
                            <IconChevronDown size={14} color="#ffffff" />
                          }
                          leftSection={
                            <IconCalendar size={16} color="#ffffff" />
                          }
                        />
                      </div>
                    </div>
                  </div>
                </Grid.Col>
              </Grid>

              <div className="mt-3 sm:mt-4 min-h-[200px] sm:min-h-[300px]">
                {isLoading ? (
                  <div className="flex justify-center items-center h-[200px] sm:h-[300px]">
                    <Loader size="lg" color="#C28CFF" />
                  </div>
                ) : (
                  <SalesReport
                    data={{
                      categories: chartData?.categories || [],
                      values: chartData?.values || [],
                    }}
                  />
                )}
              </div>
            </div>
          </Grid.Col>

          <Grid.Col span={{ base: 12, lg: 3.75 }}>
            <div className="rounded-xl sm:rounded-2xl font-semibold text-[18px] sm:text-[20px] lg:text-[22px] text-[#ffffff] p-3 sm:p-4 lg:p-5 mt-1 font-sans h-full bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <div className="flex flex-row justify-between">
                <div>{getTranslation(language, 'dashboard', 'Calendar')}</div>
                <div>
                  <IconDots />
                </div>
              </div>
              <div className="mt-3 sm:mt-4 border border-[#ffffff] rounded-md text-[#ffffff] overflow-hidden">
                <BasicDateCalendarBussiness />
              </div>
            </div>
          </Grid.Col>
        </Grid>

        {/* Sold Tickets Table Section */}
        <Grid className="py-3 sm:py-5 px-1 sm:px-1">
          <Grid.Col span={12}>
            <div className="rounded-xl sm:rounded-2xl ml-2 sm:ml-4 mr-1 sm:mr-2 font-semibold text-[#ffffff] p-3 sm:p-4 font-sans h-full bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <div className="flex flex-row justify-between items-center mb-3 sm:mb-4">
                <div className="ml-1 sm:ml-2 text-[18px] sm:text-[20px] lg:text-[22px]">
                  {getTranslation(language, 'dashboard', 'Sold Tickets')}
                </div>
                <div className="flex items-center gap-2">
                  <div className="text-[10px] sm:text-[12px] text-[#ABADBF] hidden sm:block">
                    Last updated: {lastUpdated.toLocaleTimeString()}
                  </div>
                  <IconDots />
                </div>
              </div>
              <div className="overflow-x-auto">
                <Table
                  data={userData?.data?.lastTicketSoldOut || []}
                  columns={columns}
                  isPaginated={false}
                  isLoading={isLoading}
                  handlePagination={() => {}}
                />
              </div>
            </div>
          </Grid.Col>
        </Grid>
      </Paper>
      <SaleView
        isTicket
        opened={accountInfo?.isModalOpen}
        setOpened={setaccountInfo}
        id={accountInfo?.id}
        row={accountInfo?.row}
        isDashboard
      />
    </>
  );
};

export default BusinessDashboard;
