import {
  Box,
  Chip,
  Flex,
  Grid,
  Loader,
  Paper,
  rem,
  ScrollArea,
  Text,
} from '@mantine/core';
import {
  IconCalendar,
  IconCheck,
  IconChevronDown,
  IconClockFilled,
  IconDots,
  IconFlag3Filled,
  IconHexagonFilled,
  IconX,
} from '@tabler/icons-react';
import { BsFillCalendarEventFill } from 'react-icons/bs';

import img from '../../assets/images/music.png';
import { Button } from '../../components';

import { DatePickerInput } from '@mantine/dates';
import { useEffect, useState } from 'react';
import { FaCalendar, FaUser } from 'react-icons/fa';
import { RiFlag2Line } from 'react-icons/ri';
import { useSelector } from 'react-redux';
import { postData, putData, fetchData } from '../../api/api';
import { filterStyle } from '../../theme/common.style';
import CommissionOverView from './charts/commission-overview';
import LiveAnalytics from './charts/live-analytics';
import TotalRevenue from './charts/total-revenue';
import { handleError, handleSuccess } from '../../utils/common/function';
import { getTranslation } from '../../layout/languages';
import NotificationCard from '../notification/NotificationCard';

const AdminDashboard = () => {
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const [value, setValue] = useState<[Date | null, Date | null]>([null, null]);
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const [isLoading, setIsLoading] = useState(true);
  const [dashboardData, setDashboardData] = useState<any>([]);
  const [notifications, setNotifications] = useState<any[]>([]);
  const [chartData, setChartData] = useState();
  const currency = useSelector(
    (state: any) => state.userInfo?.preferences?.currency?.preferredCurrencyId
  );
  const [selectedRange, setSelectedRange] = useState<
    'daily' | 'weekly' | 'monthly' | 'annual'
  >('daily');
  const [resetTable, setResetTable] = useState(false);
  const [selectedEarnRange, setSelectedEarnRange] = useState('All Time');

  const ranges = [
    { id: 'Daily', label: 'Daily', width: 79 },
    { id: 'Weekly', label: 'Weekly', width: 79 },
    { id: 'Monthly', label: 'Monthly', width: 'auto' },
    { id: 'Annual', label: 'Annual', width: 'auto' },
    { id: 'All Time', label: 'All Time', width: 'auto' },
  ];

  useEffect(() => {
    const fetchDashboardStats = async () => {
      const formatDate = (date: Date) => {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
      };

      let startDate: string;
      let endDate: string;

      const today = new Date();

      switch (selectedRange) {
        case 'daily': {
          startDate = formatDate(today);
          const tomorrow = new Date(today);
          tomorrow.setDate(today.getDate() + 1);
          endDate = formatDate(tomorrow);
          break;
        }
        case 'weekly': {
          const startOfWeek = new Date(today);
          startOfWeek.setDate(today.getDate() - 6);
          startDate = formatDate(startOfWeek);
          endDate = formatDate(today);
          break;
        }
        case 'monthly': {
          const startOfMonth = new Date(
            today.getFullYear(),
            today.getMonth(),
            1
          );
          startDate = formatDate(startOfMonth);
          endDate = formatDate(today);
          break;
        }
        case 'annual': {
          const startOfYear = new Date(today.getFullYear(), 0, 1);
          startDate = formatDate(startOfYear);
          endDate = formatDate(today);
          break;
        }
        default: {
          startDate = formatDate(today);
          const tomorrow = new Date(today);
          tomorrow.setDate(today.getDate() + 1);
          endDate = formatDate(tomorrow);
          break;
        }
      }

      // Custom range
      if (value[0] && value[1]) {
        startDate = formatDate(value[0]);
        const isSameDay = value[0].toDateString() === value[1].toDateString();

        if (isSameDay) {
          const nextDay = new Date(value[1]);
          nextDay.setDate(nextDay.getDate() + 1);
          endDate = formatDate(nextDay);
        } else {
          endDate = formatDate(value[1]);
        }
      }

      const payload = {
        startDate,
        endDate,
        commissionTimeline: selectedEarnRange,
      };

      setIsLoading(true);

      postData('statistics/admin-dashboard', payload)
        .then((response: any) => {
          setDashboardData(response.data);

          if (response.data.userEngagement) {
            setChartData(response.data.userEngagement);
          }
          if (response.data.notification) {
            setNotifications(response.data.notification);
          }
        })
        .catch(error => {
          console.error('Error fetching dashboard stats:', error);
          handleError(error);
        })
        .finally(() => {
          setIsLoading(false);
        });
    };

    fetchDashboardStats();
  }, [selectedRange, value, selectedEarnRange, currency]);

  const handleAccept = async (notification: any) => {
    setIsLoading(true);
    try {
      if (notification.type === 'Profile') {
        await putData(`users/${notification.profileId}`, {
          isApproved: true,
        });
      } else if (notification.type === 'Event') {
        await putData(`events/${notification.eventId}`, {
          isApproved: true,
          status: 'Live',
        });
      }

      await putData(
        `notifications/${notification.id}/mark-as-read/${userInfo.id}`,
        {}
      );

      handleSuccess('Accepted successfully');
      setResetTable(!resetTable);
    } catch (error: any) {
      handleError(error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleReject = async (notification: any) => {
    try {
      setIsLoading(true);

      if (notification.type === 'Profile') {
        await putData(`profiles/${notification.profileId}`, {
          isRejected: true,
        });
        await putData(
          `notifications/${notification.id}/mark-as-read/${userInfo.id}`,
          {}
        );
      } else if (notification.type === 'Event') {
        await putData(`events/${notification.eventId}`, { status: 'Rejected' });
        await putData(
          `notifications/${notification.id}/mark-as-read/${userInfo.id}`,
          {}
        );
      }

      setResetTable(!resetTable);

      handleSuccess('Notification rejected successfully!');
    } catch (error) {
      console.error('Error rejecting notification:', error);
      handleError(error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleBookmark = async (notificationId: string) => {
    try {
      await putData(
        `notifications/${notificationId}/mark-as-bookmarked/${userInfo.id}`,
        {}
      );

      setNotifications(prev =>
        prev.map(n =>
          n.id === notificationId ? { ...n, isBookMarked: !n.isBookMarked } : n
        )
      );

      handleSuccess('Bookmark status updated successfully!');
    } catch (error) {
      console.error('Error bookmarking notification:', error);
      handleError(error);
    }
  };

  const filters: Array<'daily' | 'weekly' | 'monthly'> = [
    'daily',
    'weekly',
    'monthly',
  ];
  return (
    <>
      <Paper
        className="flex justify-between items-center p-4 min-h-screen overflow-x-hidden"
        radius={0}
        style={{
          backgroundColor: '#3B1E6D',
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
        }}
      >
        <Grid className="flex flex-row justify-between px-4 py-5" gutter={'xl'}>
          {(isLoading
            ? [1, 2, 3].map(i => ({
                title: (
                  <div className="h-4 w-32 bg-gray-400 rounded animate-pulse mt-2" />
                ),
                value: (
                  <div className="h-4 w-20 bg-gray-400 rounded animate-pulse mt-2" />
                ),
                icon: (
                  <div className="animate-pulse w-20 h-16 bg-gray-400 rounded-full" />
                ),
                colorIcon: 'transparent',
                iconbg: 'transparent',
              }))
            : [
                {
                  title: `${dashboardData?.totalUsers ?? 0}+`,
                  value: 'Total Users',
                  icon: (
                    <div>
                      <FaUser style={{ width: rem(21), height: rem(21) }} />
                    </div>
                  ),
                  colorIcon: '#8640FD',
                  iconbg: '#623F9D',
                },
                {
                  title: `${dashboardData?.totalEvents ?? 0}+`,
                  value: 'Total Events',
                  icon: (
                    <div>
                      <BsFillCalendarEventFill
                        style={{ width: rem(23), height: rem(23) }}
                      />
                    </div>
                  ),
                  colorIcon: '#FF69B4',
                  iconbg: '#90438E',
                },
                {
                  title: `${dashboardData?.commission?.total?.toFixed(3) ?? 0}+`,
                  value: 'Total Sales',
                  icon: (
                    <div>
                      <IconHexagonFilled
                        style={{ width: rem(28), height: rem(28) }}
                      />
                    </div>
                  ),
                  colorIcon: '#2DA49A',
                  iconbg: '#2FE5A74D',
                },
              ]
          ).map((item: any, index: number) => (
            <Grid.Col key={index} span={{ base: 12, md: 12, lg: 4 }}>
              <div className="flex flex-row justify-start rounded-2xl p-8 space-y-2 h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
                <div>
                  <div
                    style={{
                      backgroundColor: item?.iconbg,
                      color: item?.colorIcon,
                      width: '62px',
                      height: '62px',
                    }}
                    className="rounded-full shadow-sm flex items-center justify-center"
                  >
                    {item?.icon}
                  </div>
                </div>
                <div className="ml-7 rtl:mr-16 rtl:ml-0">
                  <div className="font-productsans text-[#ffffff] font-bold text-[20px]">
                    {item?.title}
                  </div>
                  <div className="font-productsans text-[14px] text-[#ABADBF]">
                    {getTranslation(language, 'dashboard', item?.value)}
                  </div>
                </div>
              </div>
            </Grid.Col>
          ))}
        </Grid>

        <Grid className="px-4 py-4" gutter={'xl'}>
          <Grid.Col span={{ base: 12, md: 12, lg: 8.25 }}>
            <div className="rounded-2xl font-semibold font-productsans text-[22px] h-[100%] text-[#ffffff]  p-4  bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <Grid>
                <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
                  <div className="ml-1.5">
                    {getTranslation(language, 'dashboard', 'Live Analytics')}
                  </div>
                </Grid.Col>

                <Flex wrap="wrap" gap={10} align="center">
                  {/* Map Buttons */}
                  {filters.map(filter => (
                    <Box key={filter}>
                      <Button
                        m={5}
                        styles={{
                          root: {
                            background:
                              selectedRange === filter
                                ? 'rgba(140, 93, 248, 0.2)'
                                : 'transparent',
                            border:
                              selectedRange === filter
                                ? '1px solid #8C5DF8'
                                : '1px solid #FFFFFF26',
                            '&:hover': {
                              background: 'rgba(140, 93, 248, 0.1)',
                            },
                          },
                        }}
                        style={{
                          height: 35,
                          minWidth: 100,
                        }}
                        onClick={() => setSelectedRange(filter)}
                      >
                        <div
                          className="font-productsans font-normal text-[14px]"
                          style={{ color: '#ffffff' }}
                        >
                          {getTranslation(
                            language,
                            'dashboard',
                            filter.charAt(0).toUpperCase() + filter.slice(1)
                          )}
                        </div>
                      </Button>
                    </Box>
                  ))}

                  {/* Select Date label */}
                  <Box mt={5}>
                    <div
                      className="text-[14px] font-productsans"
                      style={{ color: '#ffffff' }}
                    >
                      {getTranslation(language, 'dashboard', 'Select Date')}
                    </div>
                  </Box>

                  {/* Date Picker */}
                  <Box style={{ minWidth: 200 }}>
                    <DatePickerInput
                      type="range"
                      placeholder={getTranslation(
                        language,
                        'dashboard',
                        'Pick Date Range'
                      )}
                      required
                      value={value}
                      onChange={setValue}
                      valueFormat="M/D/YY"
                      style={filterStyle}
                      styles={{
                        input: {
                          backgroundColor: 'transparent',
                          borderColor: '#FFFFFF26',
                          color: '#ffffff',
                          fontFamily: 'Product Sans',
                          fontSize: '12px',
                        },
                        placeholder: { color: 'white', fontWeight: 'normal' },
                      }}
                      rightSection={
                        <IconChevronDown size={16} color="#ffffff" />
                      }
                      leftSection={<IconCalendar size={20} color="#ffffff" />}
                    />
                  </Box>
                </Flex>
              </Grid>
              <div className="mt-3 ">
                <LiveAnalytics data={chartData} />
              </div>
            </div>
          </Grid.Col>
          <Grid.Col span={{ base: 12, md: 12, lg: 3.75 }}>
            <div className="rounded-2xl font-semibold font-productsans text-[22px] h-[100%] text-[#ffffff]  p-4  bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <CommissionOverView
                commissionData={dashboardData?.commission}
                comparisonData={
                  dashboardData?.ticketSoldCommissionOverTime?.comparison
                }
              />
            </div>
          </Grid.Col>
          <Grid.Col span={{ base: 12, md: 12, lg: 6 }}>
            <div className="rounded-2xl font-semibold text-[22px] text-[#ffffff] p-4 font-productsans h-[100%] bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <div className="flex flex-row justify-between">
                <div className="ml-2">
                  {getTranslation(language, 'dashboard', 'Notification')}
                </div>
                <div>
                  <IconDots />
                </div>
              </div>
              <div className="mt-4">
                <ScrollArea h={480} scrollbars="y" className="custom-scroll">
                  <Grid className="flex flex-row justify-between px-1 py-2">
                    {isLoading ? (
                      <div className="flex items-center justify-center ml-64 mt-56 ">
                        <Loader size="lg" color="#C28CFF" />
                      </div>
                    ) : notifications?.length === 0 ? (
                      <Text
                        style={{
                          color: 'white',
                          fontSize: '16px',
                          fontFamily: 'Product Sans',
                          textAlign: 'center',
                          marginTop: '50px',
                          marginLeft: '20px',
                        }}
                      >
                        {getTranslation(
                          language,
                          'dashboard',
                          'No notifications found.'
                        )}
                      </Text>
                    ) : (
                      notifications?.map((item: any, index: number) => {
                        return (
                          <NotificationCard
                            viewDetails={false}
                            key={item.id}
                            notification={item}
                            userInfo={userInfo}
                            language={language}
                            handleAccept={handleAccept}
                            handleReject={handleReject}
                            handleBookmark={handleBookmark}
                          />
                        );
                      })
                    )}
                  </Grid>
                </ScrollArea>
              </div>
            </div>
          </Grid.Col>

          <Grid.Col span={{ base: 12, md: 12, lg: 6 }}>
            <div className="rounded-2xl font-semibold font-productsans text-[22px] h-auto text-[#ffffff]  p-4  bg-transparent bg-[#ffffff] bg-opacity-5 border border-[#623F9D]">
              <Grid>
                <Grid.Col span={{ base: 12, md: 12, lg: 12 }}>
                  <div className="ml-1.5">
                    {getTranslation(language, 'dashboard', 'Total Revenue')}
                  </div>
                </Grid.Col>

                <div
                  style={{
                    display: 'flex',
                    flexWrap: 'wrap',
                    gap: '8px',
                    marginTop: '12px',
                  }}
                >
                  {['Daily', 'Weekly', 'Monthly', 'Annual', 'All Time'].map(
                    label => (
                      <Button
                        key={label}
                        autoContrast
                        size="sm"
                        onClick={() => setSelectedEarnRange(label)}
                        styles={{
                          root: {
                            background:
                              selectedEarnRange === label
                                ? 'rgba(140, 93, 248, 0.2)'
                                : 'transparent',
                            border:
                              selectedEarnRange === label
                                ? '1px solid #8C5DF8'
                                : '1px solid #FFFFFF26',
                            '&:hover': {
                              background: 'rgba(140, 93, 248, 0.1)',
                            },
                          },
                        }}
                        style={{
                          height: 35,
                          padding: '0 16px',
                          minWidth: 79,
                          flex: '0 1 auto', // important: allows flexible wrapping
                        }}
                      >
                        <div
                          className="font-productsans font-normal text-[14px] ml-0.5"
                          style={{
                            color:
                              selectedEarnRange === label
                                ? '#8C5DF8'
                                : '#ffffff',
                            whiteSpace: 'nowrap',
                          }}
                        >
                          {getTranslation(language, 'dashboard', label)}
                        </div>
                      </Button>
                    )
                  )}
                </div>
              </Grid>
              <TotalRevenue
                timelineData={
                  dashboardData?.ticketSoldCommissionOverTime?.timeline || []
                }
                comparisonData={
                  dashboardData?.ticketSoldCommissionOverTime?.comparison
                }
              />
            </div>
          </Grid.Col>
        </Grid>
      </Paper>
    </>
  );
};

export default AdminDashboard;
