import { useEffect, useState } from 'react';
import { Calendar, momentLocalizer } from 'react-big-calendar';
import moment from 'moment';
import 'react-big-calendar/lib/css/react-big-calendar.css';
import {
  Card,
  Title,
  Text,
  Container,
  Modal,
  Button,
  Select,
  Paper,
  Loader,
} from '@mantine/core';
import { IconCalendar } from '@tabler/icons-react';
import { inputStyle2 } from '../../theme/common.style';
import { fetchData, postData, putData } from '../../api/api';
import { useSelector } from 'react-redux';
import * as Yup from 'yup';
import { Formik } from 'formik';
import { DatePickerInput } from '@mantine/dates';
import { getTranslation } from '../../layout/languages';
import { BsChevronLeft, BsChevronRight } from 'react-icons/bs';
import { handleSuccess, handleError } from '../../utils/common/function';

const MyCalendar = () => {
  const userInfo = useSelector((state: any) => state?.userInfo?.userInfo);
  const localizer = momentLocalizer(moment);
  const language = useSelector(
    (state: any) => state.userInfo?.preferences?.language ?? 'en'
  );
  const [modalOpened, setModalOpened] = useState(false);
  const [fromDate, setFromDate] = useState<Date | null>(null);
  const [toDate, setToDate] = useState<Date | null>(null);
  const [userData, setUserData] = useState<any>();
  const [userURLData, setUserURLData] = useState<any>();
  const [loading, setLoading] = useState<boolean>(true);

  // Handle date selection
  const handleDateClick = (date: Date) => {
    setFromDate(date);
    setToDate(date);
    setModalOpened(true);
  };

  const AvailabilitySchema = Yup.object().shape({
    dateFrom: Yup.string().required('From date is required'),
    dateTo: Yup.string().required('To date is required'),
    status: Yup.string().required('Status is required'),
  });

  useEffect(() => {
    const url = `users/${userInfo?.id}`;
    const fetchUserAvailability = async () => {
      setLoading(true);
      try {
        const response = await fetchData(url);
        setUserData(response);
      } catch (error) {
        handleError(error, 'Failed to fetch user availability');
      } finally {
        setLoading(false);
      }
    };

    if (userInfo?.id) {
      fetchUserAvailability();
    }
  }, [userInfo?.id]);

  const CustomToolbar = (toolbar: any) => {
    const goToBack = () => {
      toolbar.onNavigate('PREV');
    };

    const goToNext = () => {
      toolbar.onNavigate('NEXT');
    };

    const currentDate = moment(toolbar.date).format('MMMM D, YYYY');

    return (
      <div className="flex justify-between items-center mb-4 px-4 py-2 text-white rounded-md">
        <div className="text-lg font-semibold">{currentDate}</div>
        <div className="flex items-center space-x-2">
          <Button
            onClick={goToBack}
            size="sm"
            variant="subtle"
            style={{ padding: '4px', minWidth: 'auto' }}
          >
            <BsChevronLeft size={16} style={{ fill: 'white' }} />
          </Button>
          <Button
            onClick={goToNext}
            size="sm"
            variant="subtle"
            style={{ padding: '4px', minWidth: 'auto' }}
          >
            <BsChevronRight size={16} style={{ fill: 'white' }} />
          </Button>
        </div>
      </div>
    );
  };

  const fetchUserAvailability = async () => {
    try {
      const response = await fetchData(`users/${userInfo?.id}`);
      setUserData(response);
    } catch (error) {
      handleError(error, 'Failed to fetch user availability');
    }
  };

  return (
    <div style={{ backgroundColor: '#3B1E6D', padding: '40px 0' }}>
      <Title
        order={1}
        style={{
          color: 'white',
          marginBottom: 3,
          marginLeft: '4%',

          marginRight: '4%',
          fontSize: '44px',
          fontFamily: 'Product Sans',
        }}
      >
        {getTranslation(language, 'calendar', 'Calendar')}
      </Title>
      <Text
        size="lg"
        color="white"
        style={{
          marginBottom: 15,
          marginTop: 10,
          marginLeft: '4%',
          marginRight: '4.5%',
          fontSize: '14px',
          fontFamily: 'Product Sans',
        }}
      >
        {getTranslation(language, 'calendar', 'Mark Your Available Status')}
      </Text>
      <Paper
        style={{
          background: 'radial-gradient( #6534BA 43%, #3B1E6D)',
          minHeight: '100vh',
        }}
      >
        <Container size="lg" className="p-4 md:p-8">
          <Card padding="lg" className="Calender_webapp">
            <div className="h-[80vh] md:h-[600px] w-full overflow-auto">
              {loading ? (
                <div className="flex justify-center items-center h-full">
                  <Loader color="white" size="lg" />
                </div>
              ) : (
                <Calendar
                  drilldownView={null}
                  localizer={localizer}
                  startAccessor="start"
                  endAccessor="end"
                  style={{ height: '100%', width: '100%' }}
                  selectable
                  onSelectSlot={slotInfo => handleDateClick(slotInfo.start)}
                  dayPropGetter={date => {
                    const today = new Date();
                    const isToday =
                      date.getDate() === today.getDate() &&
                      date.getMonth() === today.getMonth() &&
                      date.getFullYear() === today.getFullYear();

                    const availability = userData?.availabilities?.find(
                      (avail: {
                        dateFrom: string | number | Date;
                        dateTo: string | number | Date;
                      }) => {
                        const startDate = new Date(avail.dateFrom);
                        const endDate = new Date(avail.dateTo);
                        return date >= startDate && date <= endDate;
                      }
                    );

                    return {
                      style: {
                        backgroundColor: availability
                          ? '#418F90'
                          : isToday
                            ? '#3B1E6D'
                            : 'white',
                        color: availability ? 'white' : 'black',
                        fontWeight: isToday ? 'bold' : 'normal',
                      },
                    };
                  }}
                  components={{
                    toolbar: CustomToolbar,
                    dateCellWrapper: ({ children, value }) => {
                      const availability = userData?.availabilities?.find(
                        (avail: {
                          dateFrom: string | number | Date;
                          dateTo: string | number | Date;
                        }) => {
                          const startDate = new Date(avail.dateFrom);
                          const endDate = new Date(avail.dateTo);
                          return value >= startDate && value <= endDate;
                        }
                      );

                      return (
                        <div
                          style={{
                            position: 'relative',
                            height: '100%',
                            width: '100%',
                          }}
                        >
                          <div
                            style={{
                              height: '100%',
                              width: '100%',
                              display: 'flex',
                              alignItems: 'center',
                              justifyContent: 'center',
                              flexDirection: 'column',
                              position: 'relative',
                              paddingBottom: availability ? '20px' : '0px',
                            }}
                          >
                            {children}
                            {availability && (
                              <div
                                style={{
                                  position: 'absolute',
                                  bottom: 2,
                                  left: '50%',
                                  transform: 'translateX(-50%)',
                                  backgroundColor: '#D83F87',
                                  color: 'white',
                                  fontFamily: 'Product Sans',
                                  fontSize: '10px',
                                  padding: '2px 6px',
                                  borderRadius: '4px',
                                }}
                              >
                                {availability.status}
                              </div>
                            )}
                          </div>
                        </div>
                      );
                    },
                    header: ({ label }) => {
                      return (
                        <div
                          style={{
                            textAlign: 'center',
                            padding: '8px',
                            fontSize: '16px',
                            border: 'none',
                            fontWeight: 'normal',
                          }}
                        >
                          {label}
                        </div>
                      );
                    },
                  }}
                  formats={{
                    dayFormat: 'D', // Day number only
                    weekdayFormat: 'dd', // Short weekday name (Mo, Tu, etc.)
                  }}
                />
              )}
            </div>
          </Card>
        </Container>
      </Paper>

      {/* Modal for Selecting Status */}
      <Modal
        opened={modalOpened}
        centered
        size="md"
        onClose={() => setModalOpened(false)}
        withCloseButton={false}
        styles={{
          content: {
            borderRadius: '15px',
            backgroundColor: '#452975',
            padding: 0,
          },
          body: {
            padding: 0,
          },
          inner: {
            padding: 0,
            margin: 0,
            maxWidth: '100vw',
          },
        }}
        overlayProps={{
          backgroundOpacity: 0.55,
          blur: 3,
        }}
        transitionProps={{ transition: 'fade-up', duration: 300 }}
        title=""
      >
        <div className="relative w-full flex flex-row justify-between">
          <div className="mt-4 text-[#ffffff] font-productsans ml-4  text-[18px]">
            Update Your Availability
          </div>
          <Modal.CloseButton
            onClick={() => setModalOpened(false)}
            style={{
              position: 'absolute',
              top: 10,
              right: 10,
              border: 'none',
              outline: 'none',
              boxShadow: 'none',
              color: '#ffffff',
              backgroundColor: 'transparent',
              transition: 'background-color 0.2s ease-in-out',
              width: 33,
              height: 33,
            }}
          />
        </div>

        <Formik
          initialValues={{ dateFrom: '', dateTo: '', status: '' }}
          validationSchema={AvailabilitySchema}
          onSubmit={async (values, { setSubmitting }) => {
            setSubmitting(true);

            const rawDateFrom = new Date(values.dateFrom);
            rawDateFrom.setHours(0, 0, 0, 0);
            const cleanedDateFrom = rawDateFrom.toISOString();

            const payload = {
              userId: userInfo?.id,
              dateFrom: cleanedDateFrom,
              dateTo: new Date(values.dateTo).toISOString(),
              status: values.status,
            };

            try {
              const result = await fetchData<any>(
                `user-availability?userId=${userInfo?.id}`
              );
              const allAvailabilities = result.data || [];

              const existing = allAvailabilities.find(
                (a: { dateFrom: string | number | Date }) => {
                  const existingDate = new Date(a.dateFrom);
                  existingDate.setHours(0, 0, 0, 0);
                  return existingDate.toISOString() === cleanedDateFrom;
                }
              );

              if (existing) {
                await putData(`user-availability/${existing.id}`, payload);
                handleSuccess('Availability updated successfully!');
              } else {
                const response = await postData('user-availability', {
                  ...payload,
                  createdBy: userInfo?.id,
                });
                setUserURLData(response);
                handleSuccess('Availability submitted successfully!');
              }

              setModalOpened(false);
              await fetchUserAvailability();
            } catch (error) {
              console.error('Error submitting availability:', error);
              handleError(error);
            } finally {
              setSubmitting(false);
            }
          }}
        >
          {({
            values,
            errors,
            touched,
            handleChange,
            handleSubmit,
            setFieldValue,
            isSubmitting,
          }) => (
            <form onSubmit={handleSubmit} className="mt-8">
              <DatePickerInput
                className="custom-input px-4 py-4"
                label="From"
                placeholder="DD/MM/YY"
                name="dateFrom"
                value={values.dateFrom ? new Date(values.dateFrom) : null}
                onChange={value =>
                  setFieldValue('dateFrom', value?.toISOString())
                }
                radius="8px"
                styles={inputStyle2}
                error={touched.dateFrom && errors.dateFrom}
                rightSection={<IconCalendar size={20} color="white" />}
              />
              <DatePickerInput
                className="custom-input px-4 py-4"
                label="To"
                placeholder="DD/MM/YY"
                name="dateTo"
                value={values.dateTo ? new Date(values.dateTo) : null}
                onChange={value =>
                  setFieldValue('dateTo', value?.toISOString())
                }
                radius="8px"
                styles={inputStyle2}
                error={touched.dateTo && errors.dateTo}
                rightSection={<IconCalendar size={20} color="white" />}
              />

              <Select
                className="custom-input px-4 py-4"
                label="Status"
                placeholder="Available"
                name="status"
                value={values.status}
                data={[
                  { value: 'AVAILABLE', label: 'Available' },
                  { value: 'UNAVAILABLE', label: 'Unavailable' },
                ]}
                onChange={value => setFieldValue('status', value)}
                radius="8px"
                styles={inputStyle2}
                error={touched.status && errors.status}
              />

              <div className="px-2 mr-2 mb-3">
                <Button
                  type="submit"
                  autoContrast
                  m={5}
                  size="sm"
                  disabled={isSubmitting}
                  loading={isSubmitting}
                  styles={{
                    root: {
                      background: '#D83F87',
                      cursor: isSubmitting ? 'not-allowed' : 'pointer',
                    },
                  }}
                  style={{
                    height: 50,
                    width: '100%',
                  }}
                  className="px-2"
                >
                  <div className="font-productsans font-normal text-[14px] text-[#ffffff]">
                    {isSubmitting ? 'Updating...' : 'Update'}
                  </div>
                </Button>
              </div>
            </form>
          )}
        </Formik>
      </Modal>

      <style>
        {`
          .rbc-btn-group {
            display: none;
          }
          .rbc-header {
            border-bottom: none !important;
            font-size: 14px !important;
            font-weight: bold !important;
          }
          .rbc-month-view {
            border: none !important;
          }
          .rbc-month-row {
            border: none !important;
          }
        .rbc-row-bg {
    border: 1px solid rgba(0, 0, 0, 0.08) !important; /* Very light grey with low opacity */
  }
    .rbc-col-bg {
    border: 1px solid rgba(0, 0, 0, 0.05) !important; /* Light borders for individual day cells */
  }
    .rbc-col-bg {
  box-sizing: border-box;
}
        `}
      </style>
    </div>
  );
};

export default MyCalendar;
