import { PasswordInput, useMantineTheme } from '@mantine/core';
import { useFormik } from 'formik';
import { useEffect, useState } from 'react';
import * as Yup from 'yup';
import { postData } from '../../api/api';
import { IconEyeCheck, IconEyeOff } from '@tabler/icons-react';
import { LuLockKeyhole } from 'react-icons/lu';
import { useNavigate, useSearchParams } from 'react-router-dom';
import { default as leftBackgroundImageSrc } from '../../assets/images/leftImage.png';
import backgroundImageSrc from '../../assets/images/login bg img.jpg';
import logo from '../../assets/images/muv-logo.svg';

import useScreenSize from '../../hooks/useScreenSize';
import { inputStyle1 } from '../../theme/common.style';
import { handleError, handleSuccess } from '../../utils/common/function';

const initialNotification = {
  isSuccess: true,
  isEnable: false,
  title: '',
  message: '',
};

export function PasswordResetPage() {
  const theme = useMantineTheme();
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const resetToken = searchParams.get('token');
  const { isLargeScreen } = useScreenSize();

  const [isLoading, setIsLoading] = useState(false);
  const [imagesLoaded, setImagesLoaded] = useState({
    background: false,
    leftBackground: false,
  });

  useEffect(() => {
    // If no token, navigate to root
    if (!resetToken) {
      navigate('/');
      return;
    }

    // Preload images
    const backgroundImage = new Image();
    backgroundImage.src = backgroundImageSrc;

    const leftBackgroundImage = new Image();
    leftBackgroundImage.src = leftBackgroundImageSrc;

    backgroundImage.onload = () => {
      setImagesLoaded(prev => ({ ...prev, background: true }));
    };
    leftBackgroundImage.onload = () => {
      setImagesLoaded(prev => ({ ...prev, leftBackground: true }));
    };
  }, [resetToken, navigate]);

  const resetForm = useFormik({
    initialValues: {
      password: '',
      confirmPassword: '',
    },
    validationSchema: Yup.object({
      password: Yup.string()
        .required('Required')
        .min(8, 'Password must be at least 8 characters long'),
      confirmPassword: Yup.string()
        .required('Required')
        .oneOf([Yup.ref('password')], 'Passwords must match'),
    }),
    onSubmit: async values => {
      setIsLoading(true);
      try {
        await postData('/auth/verify-reset-token', {
          token: resetToken,
          newPassword: values.password,
        });

        handleSuccess('Password Updated Successfully');

        setTimeout(() => navigate('/'), 1500);
      } catch (error) {
        handleError(error);
      } finally {
        setIsLoading(false);
      }
    },
  });

  const VisibilityToggleIcon = ({ reveal }: { reveal: boolean }) =>
    reveal ? (
      <IconEyeOff
        style={{
          width: '32px',
          height: '32px',
          color: '#ffffff',
        }}
      />
    ) : (
      <IconEyeCheck
        style={{
          width: '26px',
          height: '26px',
          color: '#ffffff',
        }}
      />
    );

  if (!resetToken) {
    return null; // Will be redirected by useEffect
  }

  if (!imagesLoaded?.background || !imagesLoaded.leftBackground) {
    return null;
  }

  return isLargeScreen ? (
    <div className="flex justify-between items-stretch h-[100vh]">
      <div className="flex-1 flex justify-center items-center h-full">
        <video
          src={'/loginpage.mp4'}
          className="w-full h-full object-cover"
          autoPlay
          loop
          muted
          playsInline
        ></video>
      </div>
      <div className="relative flex-1 bg-cover bg-no-repeat bg-center bg-[#3B1E6D] h-screen items-center justify-center flex bg-gradient-to-br from-[#563097] via-[#3B1E6D] to-[#563097]">
        <div className="flex flex-col items-center w-full max-w-[500px] px-5">
          <img
            src={logo}
            alt=""
            className="max-h-full h-[92px] w-[226px] mb-[39px] mt-[60px]"
          />
          <div className="text-[#ffffff] text-[36px] font-bold font-productsans leading-[49.7px] mt-1">
            Reset Password
          </div>

          <form
            className="w-full flex flex-col gap-4 mt-[42px]"
            onSubmit={resetForm.handleSubmit}
          >
            <PasswordInput
              label="New Password"
              placeholder="Enter new password"
              name="password"
              value={resetForm.values.password}
              onChange={resetForm.handleChange}
              onBlur={resetForm.handleBlur}
              error={resetForm.touched.password && resetForm.errors.password}
              visibilityToggleIcon={VisibilityToggleIcon}
              className="custom-password-input custom-input"
              radius="6px"
              styles={inputStyle1}
              leftSection={
                <LuLockKeyhole
                  style={{
                    width: '25px',
                    height: '25px',
                    color: '#FFFFFF',
                    position: 'absolute',
                    left: '25px',
                    top: '50%',
                    transform: 'translateY(-50%)',
                  }}
                />
              }
            />

            <PasswordInput
              label="Confirm Password"
              placeholder="Confirm new password"
              name="confirmPassword"
              value={resetForm.values.confirmPassword}
              onChange={resetForm.handleChange}
              onBlur={resetForm.handleBlur}
              error={
                resetForm.touched.confirmPassword &&
                resetForm.errors.confirmPassword
              }
              visibilityToggleIcon={VisibilityToggleIcon}
              className="custom-password-input custom-input"
              radius="6px"
              styles={inputStyle1}
              leftSection={
                <LuLockKeyhole
                  style={{
                    width: '25px',
                    height: '25px',
                    color: '#FFFFFF',
                    position: 'absolute',
                    left: '25px',
                    top: '50%',
                    transform: 'translateY(-50%)',
                  }}
                />
              }
            />

            <div className="mt-[2px] flex flex-col items-center">
              <button
                className="w-[200px] h-[50px] cursor-pointer select-none bg-[#D83F87] py-2 px-5 text-center align-middle text-sm text-color-white shadow font-productsans rounded-md text-[18px] font-bold"
                type="submit"
                disabled={isLoading}
              >
                {isLoading ? 'Loading...' : 'Reset Password'}
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>
  ) : (
    <div className="flex flex-col lg:flex-row justify-between items-stretch h-auto">
      <div className="flex-1 bg-cover bg-no-repeat bg-center h-auto lg:h-screen min-h-screen flex items-center justify-center bg-[#3B1E6D] bg-gradient-to-br from-[#563097] via-[#3B1E6D] to-[#563097]">
        <div className="flex flex-col items-center w-full max-w-[500px] px-5">
          <img
            src={logo}
            alt="Logo"
            className="max-h-[80px] w-auto mb-5 mt-[20px]"
          />
          <div className="text-[#ffffff] text-[36px] font-bold font-productsans leading-[49.7px] mt-1">
            Reset Password
          </div>

          <form
            className="w-full flex flex-col gap-4"
            onSubmit={resetForm.handleSubmit}
          >
            <PasswordInput
              label="New Password"
              placeholder="Enter new password"
              name="password"
              value={resetForm.values.password}
              onChange={resetForm.handleChange}
              onBlur={resetForm.handleBlur}
              error={resetForm.touched.password && resetForm.errors.password}
              visibilityToggleIcon={VisibilityToggleIcon}
              radius="8px"
              styles={inputStyle1}
              leftSection={
                <LuLockKeyhole
                  style={{
                    width: '24px',
                    height: '24px',
                    color: '#FFFFFF',
                  }}
                />
              }
            />

            <PasswordInput
              label="Confirm Password"
              placeholder="Confirm new password"
              name="confirmPassword"
              value={resetForm.values.confirmPassword}
              onChange={resetForm.handleChange}
              onBlur={resetForm.handleBlur}
              error={
                resetForm.touched.confirmPassword &&
                resetForm.errors.confirmPassword
              }
              visibilityToggleIcon={VisibilityToggleIcon}
              radius="8px"
              styles={inputStyle1}
              leftSection={
                <LuLockKeyhole
                  style={{
                    width: '24px',
                    height: '24px',
                    color: '#FFFFFF',
                  }}
                />
              }
            />

            <div className="mt-[32px] flex flex-col items-center">
              <button
                className="w-[200px] h-[50px] cursor-pointer select-none bg-[#D83F87] py-2 px-5 text-center align-middle text-sm text-color-white shadow font-productsans rounded-md text-[16px] sm:text-[18px] font-bold"
                type="submit"
                disabled={isLoading}
              >
                {isLoading ? 'Loading...' : 'Reset Password'}
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>
  );
}
