// Importing necessary Mantine components and hooks
import { useMantineTheme } from '@mantine/core';

// Importing useFormik for form handling
import { useFormik } from 'formik';

// Importing hooks and state management tools
import { useEffect, useState } from 'react';
import { useDispatch } from 'react-redux';

// Importing Yup for form validation
import * as Yup from 'yup';

// Importing custom API function
import { postData } from '../../api/api';

// Importing type definitions

import { LuUserRound } from 'react-icons/lu';
import { useNavigate } from 'react-router-dom';
import { default as leftBackgroundImageSrc } from '../../assets/images/leftImage.png';
import backgroundImageSrc from '../../assets/images/login bg img.jpg';
import logo from '../../assets/images/muv-logo.svg';
import { TextInput } from '../../components';
import useScreenSize from '../../hooks/useScreenSize';
import { inputStyle } from '../../theme/common.style';
import { handleError, handleSuccess } from '../../utils/common/function';

// Initial notification state
const initialNotification = {
  isSuccess: true, // Indicates whether the notification is successful
  isEnable: false, // Indicates whether the notification is enabled
  title: '', // Title of the notification
  message: '', // Message of the notification
};

// Preload the background image
const backgroundImage = new Image();
// eslint-disable-next-line @typescript-eslint/no-require-imports
backgroundImage.src = backgroundImageSrc;

const leftBackgroundImage = new Image();

// eslint-disable-next-line @typescript-eslint/no-require-imports
leftBackgroundImage.src = leftBackgroundImageSrc;

// LoginPage function component
export function ForgotPassword() {
  // Initialize theme hook
  const theme = useMantineTheme();

  const navigate = useNavigate();

  // Initialize dispatch for Redux actions
  const dispatch = useDispatch();

  const { isLargeScreen } = useScreenSize();

  // Initialize notification state

  const [isLoading, setIsLoading] = useState(false);

  const [imagesLoaded, setImagesLoaded] = useState({
    background: false,
    leftBackground: false,
  });

  useEffect(() => {
    // Preload the background image
    const backgroundImage = new Image();
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    backgroundImage.src = backgroundImageSrc;

    const leftBackgroundImage = new Image();
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    leftBackgroundImage.src = leftBackgroundImageSrc;
    backgroundImage.onload = () => {
      setImagesLoaded(prev => ({ ...prev, background: true }));
    };
    leftBackgroundImage.onload = () => {
      setImagesLoaded(prev => ({ ...prev, leftBackground: true }));
    };
  }, []);

  // Initialize formik for form handling
  const formik = useFormik({
    initialValues: {
      email: '',
    },
    validationSchema: Yup.object({
      email: Yup.string().email('Invalid email').required('Required'),
    }),
    onSubmit: async values => {
      setIsLoading(true);

      postData('/auth/generate-reset-token', {
        ...values,
      })
        .then(() =>
          handleSuccess(
            "We've sent you a link to reset your password. Please check your inbox."
          )
        )
        .catch(error => {
          handleError(error);
        })
        .finally(() => setIsLoading(false));
    },
  });

  return imagesLoaded?.background && imagesLoaded.leftBackground ? (
    isLargeScreen ? (
      <>
        <div className="flex justify-between items-stretch h-[100vh]">
          <div className="flex-1 flex justify-center items-center h-full">
            <video
              src={'/loginpage.mp4'}
              className="w-full h-full object-cover"
              autoPlay
              loop
              muted
              playsInline
            ></video>
          </div>
          <div
            className="relative flex-1 bg-cover bg-no-repeat bg-center bg-[#3B1E6D] h-screen items-center justify-center flex bg-gradient-to-br
from-[#563097]
via-[#3B1E6D]
to-[#563097]"
          >
            <div className="flex flex-col items-center w-full max-w-[500px] px-5">
              <img
                src={logo}
                alt=""
                className="max-h-full h-[92px] w-[226px] mb-[39px] mt-[60px]"
              />
              <div className="text-[#ffffff] text-[36px] font-bold font-productsans leading-[49.7px] mt-1">
                Forgot Password
              </div>
              <div className="text-[#ffffff] text-[17px] font-normal font-productsans leading-[49.7px] -mt-5">
                Enter your email address to Request Password Link
              </div>
              <form
                className="w-full flex flex-col gap-4 mt-[42px]"
                onSubmit={formik.handleSubmit}
              >
                <TextInput
                  className="custom-input"
                  label="Email Address"
                  placeholder="Enter your email"
                  value={formik.values?.email}
                  leftSection={
                    <LuUserRound
                      style={{
                        width: '25px',
                        height: '25px',
                        color: '#FFFFFF',
                        position: 'absolute',
                        left: '25px',
                        top: '50%',
                        transform: 'translateY(-50%)',
                      }}
                    />
                  }
                  onChange={(value: any) =>
                    formik.setFieldValue('email', value)
                  }
                  radius="6px"
                  styles={inputStyle}
                  error={formik.touched.email && formik.errors.email}
                />

                <div className="mt-[2px] flex flex-col items-center">
                  <button
                    className="w-[200px] h-[50px] cursor-pointer select-none bg-[#D83F87] py-2 px-5 text-center align-middle text-sm text-color-white shadow font-productsans rounded-md text-[18px] font-bold"
                    type="submit"
                    disabled={isLoading}
                  >
                    {isLoading ? 'Loading' : 'Reset '}
                  </button>
                </div>
              </form>{' '}
              <div
                onClick={() => navigate('/')}
                className="mt-2 cursor-pointer underline font-productsans text-[#ffffff] text-[16px] font-normal"
              >
                Login
              </div>
            </div>
          </div>
        </div>
      </>
    ) : (
      <>
        <div className="flex flex-col lg:flex-row justify-between items-stretch h-auto">
          {/* Right Form Section */}
          <div
            className="flex-1 bg-cover bg-no-repeat bg-center h-auto lg:h-screen min-h-screen flex items-center justify-center bg-[#3B1E6D] bg-gradient-to-br
from-[#563097]
via-[#3B1E6D]
to-[#563097] "
          >
            <div className="flex flex-col items-center w-full max-w-[500px] px-5">
              {/* Logo */}
              <img
                src={logo}
                alt="Logo"
                className="max-h-[80px] w-auto mb-5 mt-[20px] "
              />
              {/* Title */}
              <h1 className="text-[#ffffff] text-[28px] sm:text-[36px] font-bold font-productsans text-center">
                Forgot Password
              </h1>
              {/* Description */}
              <p className="text-[#ffffff] text-[14px] sm:text-[17px] font-normal font-productsans mb-8 text-center">
                Enter your email address to Request Reset Link
              </p>
              {/* Form */}
              <form
                className="w-full flex flex-col gap-4"
                onSubmit={formik.handleSubmit}
              >
                {/* Email Input */}
                <TextInput
                  className="custom-input"
                  label="Email Address"
                  placeholder="Enter your email"
                  value={formik.values?.email}
                  leftSection={
                    <LuUserRound
                      style={{
                        width: '29.26px',
                        height: '32.51px',
                        color: '#FFFFFF',
                      }}
                    />
                  }
                  onChange={(value: any) =>
                    formik.setFieldValue('email', value)
                  }
                  radius="8px"
                  styles={inputStyle}
                  error={formik.touched.email && formik.errors.email}
                />

                {/* Submit Button */}
                <div className="mt-[32px] flex flex-col items-center">
                  <button
                    className="w-[200px] h-[50px] cursor-pointer select-none bg-[#D83F87] py-2 px-5 text-center align-middle text-sm text-color-white shadow font-productsans rounded-md text-[16px] sm:text-[18px] font-bold"
                    type="submit"
                    disabled={isLoading}
                  >
                    {isLoading ? 'Loading' : 'Reset '}
                  </button>

                  {/* Register Link */}
                  <div className="mt-[12px] text-center"></div>
                </div>
              </form>
              <div
                onClick={() => navigate('/')}
                className="mt-2 cursor-pointer underline font-productsans text-[#ffffff] text-[16px] font-normal"
              >
                Login
              </div>
            </div>
          </div>
        </div>
      </>
    )
  ) : null;
}
